from django.conf import settings
from django.contrib import admin
from django.core.urlresolvers import reverse
from django.template.loader import render_to_string
from django.utils.translation import ugettext_lazy as _

from kelvin.common.admin import LabellessRawIdFieldsMixin, superuser_only_action
from kelvin.courses.admin.forms import CourseLessonLinkInlineForm
from kelvin.courses.admin.proxy_models import CourseOriginals
from kelvin.courses.models import CourseLessonLink, PeriodicNotification
from kelvin.mailing_lists.models import CourseMailingList

from ..models.course_tvm import CourseTVMService
from ..models.periodic import PeriodicRoleDigest
from .mixins import CourseLessonLinkInlineAvailableForSupportAdminMixin


class CourseLessonLinkInline(
    LabellessRawIdFieldsMixin,
    CourseLessonLinkInlineAvailableForSupportAdminMixin,
    admin.TabularInline,
):
    """Инлайн m2m связи курс-занятие"""
    model = CourseLessonLink
    form = CourseLessonLinkInlineForm
    verbose_name = u'Занятие курса'
    verbose_name_plural = u'Занятия курса'
    template = 'courses/admin/clessons_inline.html'
    fields = (
        'id',
        'lesson',
        'get_lesson_link',
        'get_clesson_front_link',
        'get_lesson_name',
        'copy_of',
        'order',
        'mode',
        'duration',
        'max_attempts_in_group',
        'show_answers_in_last_attempt',
        'show_all_problems',
        'accessible_to_teacher',
        'lesson_editable',
        'url',
        'start_date',
        'comment',
        'date_assignment',
        'date_completed',
        'get_actions',
        'finish_date',
        'evaluation_date',
        'diagnostics_text',
        'visual_mode',
        'available_for_support',
    )
    labelless_raw_id_fields = (
        'lesson',
        'copy_of',
        'diagnostics_text',
    )
    readonly_fields = (
        'get_lesson_link',
        'get_clesson_front_link',
        'id',
        'lesson_editable',
        'get_lesson_name',
        'get_actions',
    )
    extra = 0

    def get_queryset(self, request):
        """
        Занятие используется в `get_lesson_name`
        """
        return super().get_queryset(request).select_related('lesson')

    def get_lesson_link(self, obj):
        """Ссылка на урок"""
        return (
            u'<a href="{0}" class="changelink" target="_blank"></a>'.format(
                reverse('admin:lessons_lesson_change', args=(obj.lesson.id,)))
            if obj.id else u''
        )
    get_lesson_link.short_description = _(u'Ссылка')
    get_lesson_link.allow_tags = True

    def get_clesson_front_link(self, obj):
        """Ссылка на курсозанятие на фронтенде"""
        if not obj.id:
            return u''

        return (
            u'<a href="{0}lab/courses/{1}/assignments/{2}/" target="_blank">'
            u'Посмотреть</a>'.format(
                settings.FRONTEND_HOST, obj.course_id, obj.id,
            )
        )
    get_clesson_front_link.short_description = _(u'Ссылка на фронт')
    get_clesson_front_link.allow_tags = True

    def get_lesson_name(self, obj):
        """Название занятия"""
        return obj.lesson.name
    get_lesson_name.allow_tags = True
    get_lesson_name.short_description = u'Название занятия'

    @superuser_only_action
    def get_actions(self, obj):
        """Действия с курсозанятием"""
        if obj.id:
            return render_to_string(
                'courses/admin/clesson_inline_actions.html',
                {
                    'clesson': obj,
                    'opts': self.parent_model._meta,
                    'original_course': self.parent_model is CourseOriginals,
                },
            )
        else:
            return ''
    get_actions.short_description = u'Действия'
    get_actions.allow_tags = True


class CourseMailingListInline(admin.StackedInline):
    """
    Инлайн для настройки почтовой рассылки
    """
    model = CourseMailingList
    can_delete = False
    extra = 0


class CourseTVMServiceInline(admin.TabularInline):
    model = CourseTVMService
    readonly_fields = (
        'created_by',
        'modified_by',
    )
    fields = ('tvm_service', ) + readonly_fields
    extra = 0


class PeriodicNotificationInline(admin.TabularInline):
    model = PeriodicNotification
    fields = ('notify_type', 'delay', 'priority', 'parameters')
    extra = 0


class PeriodicRoleDigestInlineAdmin(admin.TabularInline):
    model = PeriodicRoleDigest
    list_display = ('role', 'is_active', 'date_created', 'date_updated')
    raw_id_fields = ('periodic_course',)
    list_filter = ('role', 'is_active')
    extra = 0
