import traceback
from uuid import uuid4

from django_object_actions import DjangoObjectActions, takes_instance_or_queryset
from simple_history.admin import SimpleHistoryAdmin

from django import forms
from django.conf import settings
from django.conf.urls import url
from django.contrib import admin, messages
from django.contrib.admin.views.main import ChangeList
from django.contrib.auth import get_user_model
from django.core.exceptions import PermissionDenied
from django.core.urlresolvers import NoReverseMatch, reverse
from django.db import transaction
from django.http.response import Http404, HttpResponse
from django.shortcuts import redirect, render
from django.template.response import TemplateResponse
from django.utils import timezone
from django.utils.html import format_html
from django.utils.translation import ugettext_lazy as _

from kelvin.accounts.filters import RawIdStudentFilter
from kelvin.achievery.achievery_client import AchieveryHTTPError
from kelvin.achievery.achievery_client import client as achievery_client
from kelvin.common.admin import UserBlameModelAdminMixin, build_dropdown_filter_with_title, superuser_only_action
from kelvin.courses.admin.filters import ClosedCoursesFilter, RawIdCourseFilter
from kelvin.courses.admin.forms import CopiedCourseAccessForm, CopyCourseForm, CourseForm
from kelvin.courses.admin.inline import (
    CourseLessonLinkInline, CourseMailingListInline, CourseTVMServiceInline, PeriodicNotificationInline,
    PeriodicRoleDigestInlineAdmin,
)
from kelvin.courses.admin.proxy_models import CourseContentManagers, CourseOriginals, CourseTeachers
from kelvin.courses.models import (
    AssignmentRule, Course, CourseFeedback, CourseLessonEdge, CourseLessonLink, CourseLessonNode, CoursePermission,
    CourseStudent, CourseTVMService, Criterion, PeriodicCourse, PeriodicStudentNotification, ProgressIndicator,
    UserCLessonState, UserCriterion,
)
from kelvin.courses.models.criterion.action import (
    RequestAchievementAction, RequestIDMRoleAction, RequestTeamAchievementAction,
)
from kelvin.courses.models.invite import CourseInvite, UserCourseInviteActivation
from kelvin.idm.idm_client import IDMHTTPError
from kelvin.idm.idm_client import client as idm_client
from kelvin.lesson_assignments.services import ensure_clesson_assignments
from kelvin.lessons.models import Lesson, LessonProblemLink

from ..models.periodic import ExcludedUser, NotifyDayOff, PeriodicNotification, PeriodicRoleDigest, RoleDigest
from ..services import apply_criterion, periodic_notify_student, prepare_periodic_course_notifications
from ..tasks import apply_criterion_task
from .mixins import CourseAvailableForSupportAdminMixin, CourseLessonLinkAvailableForSupportAdminMixin

User = get_user_model()


class CourseChangeList(ChangeList):
    def __init__(self, *args, **kwargs):
        super(CourseChangeList, self).__init__(*args, **kwargs)
        self.title = self.model_admin.list_page_title


class CourseConvertForm(forms.Form):
    """
    Форма для админки конвертации курса в занятие
    """
    target_lesson_id = forms.IntegerField(label=_(u'ID целевого занятия'),
                                          required=True)

    def clean_target_lesson_id(self):
        """
        Подходит ли целевое занятие для конвертации? Условия:

        - Тип задания: Контрольная работа
        - Не имеет задач
        - Задание-источник не имеет задач с незаполненной группой
        """
        lesson_id = self.cleaned_data['target_lesson_id']

        if not Lesson.objects.filter(pk=lesson_id).exists():
            raise forms.ValidationError(_(u'Занятие с таким ID не существует'))

        if LessonProblemLink.objects.filter(lesson=lesson_id,
                                            group__isnull=True).exists():
            raise forms.ValidationError(_(u'Группа заполнена не у всех '
                                          u'задач источника'))

        if LessonProblemLink.objects.filter(lesson=lesson_id).count() > 0:
            raise forms.ValidationError(_(u'Целевое занятие уже содержит '
                                          u'задачи'))

        return lesson_id

    def get_lesson_id(self):
        """
        Возвращает проверенный идентификатор занятия из формы
        """
        return self.cleaned_data['target_lesson_id']


@admin.register(Course)
class CourseAdmin(UserBlameModelAdminMixin, CourseAvailableForSupportAdminMixin, admin.ModelAdmin):
    """
    Админка модели учебного курса
    """
    # шаблоны
    change_form_template = 'courses/admin/change_form.html'
    copy_view_template = 'courses/admin/copy_form.html'
    clesson_actions_result_template = (
        'courses/admin/clesson_actions_result.html')
    course_convert_view_template = 'courses/admin/course_convert_form.html'

    # настройки страницы списка
    list_page_title = u'Курсы'
    list_display = (
        'name',
        'owner',
        'copy_of',
        'date_closed',
        'date_created',
        'date_updated',
        'project',
        'average_score',
    )
    raw_id_fields = (
        'cover',
        'copy_of',
    )
    list_filter = (
        ClosedCoursesFilter,
        'mode',
        'available_for_support',
        ('copy_of', admin.RelatedOnlyFieldListFilter),
        ('owner', admin.RelatedOnlyFieldListFilter),
        ('subject', admin.RelatedOnlyFieldListFilter),
        ('project', admin.RelatedOnlyFieldListFilter),
    )
    search_fields = ('id', 'name', 'code', )
    list_select_related = (
        'owner',
        'copy_of',
    )
    actions = (
        'close_course',
        'update_journal',
    )

    # настройки страницы курса
    readonly_fields = (
        'get_students_link',
        'get_course_convert_link',
        'created_by',
        'modified_by',
    )
    inlines = (
        CourseMailingListInline, CourseLessonLinkInline, CourseTVMServiceInline,
    )
    filter_horizontal = ('group_levels', 'source_courses')
    form = CourseForm

    def _fill_order(self, data):
        """
        Копирует данные и изменяет их, добавляя порядок занятия,
        если указано занятие, но не указан порядок.
        После чего возвращает копию.

        @param data: данные формы создания/изменения курса
        """
        order_key = 'courselessonlink_set-{0}-order'
        lesson_key = 'courselessonlink_set-{0}-lesson'
        i = 0
        orders_to_set = []
        max_order = 0
        while True:
            if lesson_key.format(i) not in data:
                break
            if not data.get(lesson_key.format(i)):
                i += 1
                continue
            order = data.get(order_key.format(i))
            if order:
                max_order = max(max_order, int(order))
            else:
                orders_to_set.append(order_key.format(i))
            i += 1
        resulting_data = data.copy()
        for key in orders_to_set:
            max_order += 1
            resulting_data[key] = max_order
        return resulting_data

    def add_view(self, request, form_url='', extra_context=None):
        """
        Заполняет поле порядка для курсозанятия
        """
        if request.method == 'POST':
            request.POST = self._fill_order(request.POST)
        return super(CourseAdmin, self).add_view(
            request, form_url, extra_context)

    def change_view(self, request, object_id, form_url='', extra_context=None):
        """
        Заполняет поле порядка для курсозанятия
        """
        if request.method == 'POST':
            request.POST = self._fill_order(request.POST)
        return super(CourseAdmin, self).change_view(
            request, object_id, form_url, extra_context)

    def get_students_link(self, obj):
        """
        Ссылка учеников в курсе
        """
        return (
            u'<a href="{0}?course_id={1}" '
            u'target="_blank">Посмотреть</a>'.format(
                reverse('admin:courses_coursestudent_changelist'),
                obj.id,
            )
        )
    get_students_link.allow_tags = True
    get_students_link.short_description = u'Ученики'

    def get_course_convert_link(self, obj):
        """
        Возвращает ссылку на админку конвертации курса в занятие с вариациями
        """
        try:
            link = u'<a href="{0}" target="_blank">Перейти</a>'.format(
                reverse('admin:courses_course_course_convert', args=(obj.pk, ))
            )

        except NoReverseMatch:
            link = u'Сначала нужно сохранить курс'

        return link

    get_course_convert_link.allow_tags = True
    get_course_convert_link.short_description = (
        u'Создать вариативное занятие из курса')

    def get_changelist(self, request, **kwargs):
        return CourseChangeList

    def get_urls(self):
        """
        Добавить адрес для страницы копирования курса
        """
        info = self.model._meta.app_label, self.model._meta.model_name
        copy_url = url(r'^(.+)/copy/$',
                       self.admin_site.admin_view(self.copy_view),
                       name='{0}_{1}_copy'.format(*info))
        load_url = url(
            r'^load_to_copies/(\d+)/$',
            self.admin_site.admin_view(self.load_to_copies_view),
            name='{0}_{1}_load'.format(*info),
        )
        grant_access_url = url(
            r'^grant_access_to_copies/(\d+)/$',
            self.admin_site.admin_view(self.grant_access_to_copies_view),
            name='{0}_{1}_grant_access'.format(*info),
        )
        ensure_assignments_url = url(
            r'^ensure_assignments/(\d+)/$',
            self.admin_site.admin_view(self.ensure_assignments_view),
            name='{0}_{1}_ensure_assignments'.format(*info),
        )
        course_convert_url = url(
            r'^course_convert/(\d+)/$',
            self.admin_site.admin_view(self.course_convert_view),
            name='{0}_{1}_course_convert'.format(*info),
        )
        return [copy_url,
                load_url,
                grant_access_url,
                ensure_assignments_url,
                course_convert_url] + super(CourseAdmin, self).get_urls()

    def load_to_copies_view(self, request, clesson_id):
        """
        Выгрузить занятие в копии курса, если в них нет копии этого занятия
        """
        try:
            clesson = CourseLessonLink.objects.get(pk=clesson_id)
        except (ValueError, TypeError, Course.DoesNotExist):
            raise Http404

        if not self.has_change_permission(request, clesson):
            raise PermissionDenied

        new_clessons = []
        for course in clesson.course.copies.exclude(
                courselessonlink__in=clesson.copies.all()):
            # Не копируем дату назначения, дату завершения, код доступа
            # Делаем копию недоступной и нередактируемой
            new_clessons.append(CourseLessonLink(
                course=course,
                order=clesson.order,
                accessible_to_teacher=None,
                lesson_editable=False,
                copy_of=clesson,
                lesson_id=clesson.lesson_id,
                mode=clesson.mode,
                duration=clesson.duration,
                finish_date=clesson.finish_date,
                evaluation_date=clesson.evaluation_date,
                max_attempts_in_group=clesson.max_attempts_in_group,
                show_answers_in_last_attempt=(
                    clesson.show_answers_in_last_attempt
                ),
                show_all_problems=clesson.show_all_problems,
                url=clesson.url,
                start_date=clesson.start_date,
                comment=clesson.comment,
            ))
        CourseLessonLink.objects.bulk_create(new_clessons)

        ctx = {
            'clessons': new_clessons,
        }
        return render(request, self.clesson_actions_result_template, ctx)

    def grant_access_to_copies_view(self, request, clesson_id):
        """
        Открыть доступ учителю во всех копиях курсозанятия
        """
        try:
            clesson = CourseLessonLink.objects.get(pk=clesson_id)
        except (ValueError, TypeError, Course.DoesNotExist):
            raise Http404

        if not self.has_change_permission(request, clesson):
            raise PermissionDenied

        # ключ, по которому перезаписываем все даты доступности учителю
        force = request.GET.get('force') == 'true'

        # запрашиваем все копии, чтобы не было запросов в шаблоне
        clessons_to_change = list(
            clesson.copies.all() if force else
            clesson.copies.filter(accessible_to_teacher__isnull=True)
        )

        updated = (
            CourseLessonLink.objects
            .filter(id__in=[obj.id for obj in clessons_to_change])
            .update(accessible_to_teacher=clesson.accessible_to_teacher)
        )

        ctx = {
            'clessons': clessons_to_change,
        }
        return render(request, self.clesson_actions_result_template, ctx)

    def ensure_assignments_view(self, request, clesson_id):
        """
        Пересчет назначений в курсозанятии согласно вариативности в занятии
        """
        try:
            clesson = CourseLessonLink.objects.get(pk=clesson_id)
        except (ValueError, TypeError, Course.DoesNotExist):
            raise Http404

        if not self.has_change_permission(request, clesson):
            raise PermissionDenied

        updated = ensure_clesson_assignments(clesson)
        return HttpResponse('{0} updated'.format(updated) if updated else 'OK')

    def copy_view(self, request, pk, form_url='', extra_context=None):
        """
        Копирование курса
        """
        try:
            course = Course.objects.get(pk=pk)
        except (ValueError, TypeError, Course.DoesNotExist):
            raise Http404

        if course.copy_of or not self.has_change_permission(request, course):
            raise PermissionDenied

        # создание формы; копирование курса
        if request.method == 'POST':
            form = CopyCourseForm(request.POST, instance=course)
            if form.is_valid():
                course = form.save()
                return redirect('admin:courses_course_change', course.id)
        else:
            form = CopyCourseForm(instance=course)

        opts = Course._meta
        context = {
            'title': u'Копирование курса %s' % course,
            'has_change_permission': self.has_change_permission(request,
                                                                course),
            'form_url': form_url,
            'opts': opts,
            'app_label': opts.app_label,
            'original': course,
            'form': form,
        }
        context.update(extra_context or {})

        return render(request, self.copy_view_template, context)

    def course_convert_view(self, request, course_id, form_url=''):
        """
        Сконвертировать курс в контрольную работу
        """
        try:
            course = Course.objects.get(pk=course_id)
        except (ValueError, TypeError, Course.DoesNotExist):
            raise Http404

        form = CourseConvertForm(request.POST or None)

        if form.is_valid():
            target_lesson_id = form.get_lesson_id()
            self._course_convert(course, target_lesson_id)
            messages.add_message(request, messages.SUCCESS,
                                 _(u'Конвертация прошла успешно'))
            return redirect('admin:lessons_lesson_change',
                            target_lesson_id)

        ctx = {
            'form': form,
            'form_url': form_url,
            'course_id': course_id,
            'has_change_permission': self.has_change_permission(request,
                                                                course),
            'opts': Course._meta,
        }

        return render(request, self.course_convert_view_template, ctx)

    def _course_convert(self, src_course, target_lesson_id):
        """
        Метод, выполняющий конвертацию курса в вариативное занятие
        """
        with transaction.atomic():
            problem_links = LessonProblemLink.objects.filter(
                lesson__course=src_course)

            target_lesson = Lesson.objects.get(pk=target_lesson_id)

            for problem_link in problem_links:
                problem_link.pk = None

                # каждое занятие формирует свою группу
                problem_link.group = problem_link.lesson_id

                # в реализации Сириуса порядок проставлялся исходя из группы,
                # что кажется сомнительной фичей
                problem_link.order = problem_link.lesson_id
                problem_link.lesson = target_lesson

            LessonProblemLink.objects.bulk_create(problem_links)

    def update_journal(self, request, queryset):
        from kelvin.result_stats.tasks import calculate_course_journal
        for course in queryset:
            calculate_course_journal(course.id)
    update_journal.short_description = u'Обновить журнал'

    def close_course(self, request, queryset):
        """
        Запускает подсчет статистики для каждой задачи в отдельном селери-таске
        """
        now = timezone.now()
        with transaction.atomic():
            CourseLessonLink.objects.filter(
                course__in=queryset,
                accessible_to_teacher__isnull=False,
                date_assignment__isnull=True,
            ).update(accessible_to_teacher=None, date_updated=now)
            CourseLessonLink.objects.filter(
                course__in=queryset,
                date_assignment__isnull=False,
                date_completed__isnull=True,
            ).update(date_completed=now, date_updated=now)
            queryset.update(date_closed=now, date_updated=now)
    close_course.short_description = u'Закрыть курс'


@admin.register(CourseOriginals)
class CourseOriginalsAdmin(CourseAdmin):
    """ Только оригинальные (нескопированные) курсы """
    dashboard_template = 'courses/admin/original_course_dashboard.html'
    copies_status_template = 'courses/admin/copies_status.html'

    list_page_title = u'Оригинальные курсы'
    list_display = (
        'name',
        'owner',
        'date_created',
        'date_updated',
        'get_dashboard_link',
        'get_copies_status_link',
    )
    list_select_related = (
        'owner',
    )

    list_filter = (
        ('owner', admin.RelatedOnlyFieldListFilter),
        ('subject', admin.RelatedOnlyFieldListFilter),
    )

    def get_queryset(self, request):
        """
        Только оригинальные курсы
        """
        qs = super(CourseAdmin, self).get_queryset(request)
        qs = qs.filter(copy_of__isnull=True)
        return qs

    def get_urls(self):
        """Добавляет адрес дашборда"""
        info = self.model._meta.app_label, self.model._meta.model_name
        dashboard_url = url(
            r'^(.+)/dashboard/$',
            self.admin_site.admin_view(self.dashboard_view),
            name='{0}_{1}_dashboard'.format(*info),
        )
        copies_status_url = url(
            r'^(.+)/copies_status/$',
            self.admin_site.admin_view(self.copies_status_view),
            name='{0}_{1}_copies-status'.format(*info),
        )
        assign_copies_url = url(
            r'^assign_copies/(\d+)/$',
            self.admin_site.admin_view(self.assign_copies_view),
            name='{0}_{1}_assign_copies'.format(*info),
        )
        copy_url_url = url(
            r'^copy_url/(\d+)/$',
            self.admin_site.admin_view(self.copy_url_view),
            name='{0}_{1}_copy_url'.format(*info),
        )
        return [
            dashboard_url,
            copies_status_url,
            assign_copies_url,
            copy_url_url,
        ] + super(CourseOriginalsAdmin, self).get_urls()

    def get_dashboard_link(self, obj):
        """Ссылка на дашборд"""
        return (
            u'<a href="{0}" class="changelink"></a>'
            .format(reverse('admin:courses_courseoriginals_dashboard',
                            args=(obj.id,)))
        )
    get_dashboard_link.allow_tags = True
    get_dashboard_link.short_description = u'Дашборд курса'

    def get_copies_status_link(self, obj):
        """Ссылка на страницу статусов копий курса"""
        return (
            u'<a href="{0}" class="changelink"></a>'
            .format(reverse('admin:courses_courseoriginals_copies-status',
                            args=(obj.id,)))
        )
    get_copies_status_link.allow_tags = True
    get_copies_status_link.short_description = u'Статусы копий'

    def dashboard_view(self, request, pk):
        """Дашборд"""
        return HttpResponse('to be reworked...')

    def copies_status_view(self, request, pk):
        """Статусы копий курса"""
        original = self.get_object(request, pk)

        if request.method == 'POST':
            form = CopiedCourseAccessForm(
                request.POST,
                instance=Course.objects.get(id=request.POST.get('id')),
            )
            if form.is_valid():  # dummy validation now
                form.save()

        original_clessons = {
            clesson.id: clesson.lesson.name
            for clesson in original.courselessonlink_set.all().select_related(
                'lesson')
        }
        copies_models = Course.objects.filter(copy_of=original).select_related(
            'owner').prefetch_related('courselessonlink_set')
        now = timezone.now()
        copies = [
            {
                'id': copy.id,
                'admin_url': reverse(
                    'admin:courses_course_change',
                    args=(copy.id,),
                ),
                'frontend_url': (settings.FRONTEND_HOST +
                                 'lab/courses/{0}/'.format(copy.id)),
                'owner': {
                    'username': copy.owner.username,
                    'full_name': copy.owner.get_full_name(),
                    'admin_url': reverse('admin:accounts_user_change',
                                         args=(copy.owner.id,)),
                },
                'last_lesson': self._get_last_lesson(copy, original_clessons),
                'available_to_assign': sum(
                    1 for clesson in copy.courselessonlink_set.all()
                    if (clesson.accessible_to_teacher and
                        clesson.accessible_to_teacher < now and
                        not clesson.date_assignment)
                ),
                'unaccessible': sum(
                    1 for clesson in copy.courselessonlink_set.all()
                    if (
                        (not clesson.accessible_to_teacher or
                         clesson.accessible_to_teacher > now) and
                        clesson.copy_of_id
                    )
                ),
                'give_access_form': CopiedCourseAccessForm(instance=copy),
            }
            for copy in copies_models
        ]
        context = {
            'opts': self.model._meta,
            'original': original,
            'has_permission': True,  # чтобы показать блок usertools
            'frontend_url': (
                settings.FRONTEND_HOST + 'lab/courses/{0}/'.format(pk)),
            'copies': copies,
        }
        return TemplateResponse(request, self.copies_status_template, context)

    def _get_last_lesson(self, copy, original_clessons):
        """
        Возвращает информацию о последнем назначенном оригинальном курсозанятии
        """
        last_clesson = None
        for clesson in copy.courselessonlink_set.all():
            if clesson.date_assignment:
                if (clesson.copy_of_id in original_clessons and (
                        not last_clesson or clesson.date_assignment >
                        last_clesson.date_assignment)):
                    last_clesson = clesson
        return {
            'name': original_clessons[last_clesson.copy_of_id],
            'date_assignment': last_clesson.date_assignment,
        } if last_clesson else None

    def assign_copies_view(self, request, clesson_id):
        """
        Выставляет текущие дату и время как дату назначения всех копий
        курсозанятия, где даты назначения нет
        """
        now = timezone.now()

        copies_updated = CourseLessonLink.objects.filter(
            copy_of=clesson_id,
            date_assignment__isnull=True,
        ).update(date_assignment=now)

        return HttpResponse(u'Копий выдано: {0}'.format(copies_updated))

    def copy_url_view(self, request, clesson_id):
        """
        Копирует URL курсозанятия всем своим копиям
        """
        clesson = CourseLessonLink.objects.get(id=clesson_id)
        copies_updated = clesson.copies.update(url=clesson.url)

        return HttpResponse(u'Копий измененено: {0}'.format(copies_updated))


@admin.register(CourseContentManagers)
class CourseContentManagersAdmin(CourseAdmin):
    list_page_title = u'Курсы контент-менеджеров'

    def get_queryset(self, request):
        """
        Только курсы контент-менеджеров
        """
        qs = super(CourseAdmin, self).get_queryset(request)
        qs = qs.filter(owner__is_content_manager=True)
        return qs

    def has_add_permission(self, request):
        return False


@admin.register(CourseTeachers)
class CourseTeachersAdmin(CourseAdmin):
    list_page_title = u'Курсы учителей'

    def get_queryset(self, request):
        """
        Только курсы учителей
        """
        qs = super(CourseAdmin, self).get_queryset(request)
        qs = qs.filter(owner__is_content_manager=False, owner__is_teacher=True)
        return qs

    def has_add_permission(self, request):
        return False


@admin.register(CourseLessonLink)
class CourseLessonLinkAdmin(
    UserBlameModelAdminMixin,
    CourseLessonLinkAvailableForSupportAdminMixin,
    admin.ModelAdmin,
):
    raw_id_fields = (
        'lesson',
        'course',
        'copy_of',
        'journal_resource',
    )
    readonly_fields = (
        'created_by',
        'modified_by',
    )


@admin.register(CourseStudent)
class CourseStudentAdmin(admin.ModelAdmin):
    """
    Ученики курса
    """
    list_display = (
        '__str__',
        'student',
        'course',
        'deleted',
        'completed',
        'date_completed',
        'date_created',
        'date_updated',
    )
    list_filter = (
        RawIdCourseFilter,
        RawIdStudentFilter,
    )

    raw_id_fields = (
        'student',
        'course',
    )


@admin.register(ProgressIndicator)
class ProgressIndicatorAdmin(admin.ModelAdmin):
    """
    Прогрессбары
    """
    list_display = (
        'slug',
    )

    search_fields = (
        'slug',
    )

@admin.register(UserCLessonState)
class UserCLessonStateAdmin(admin.ModelAdmin):
    raw_id_fields = ('user', 'clesson', )


@admin.register(CourseTVMService)
class CourseTVMServiceAdmin(UserBlameModelAdminMixin, admin.ModelAdmin):
    raw_id_fields = ('course', )
    list_display = ('course', 'tvm_service')
    list_filter = ('course', 'tvm_service')
    readonly_fields = ('created_by', 'modified_by', 'created', 'modified')
    fields = list_filter + readonly_fields


@admin.register(CoursePermission)
class CoursePermissionAdmin(admin.ModelAdmin):
    list_display = ('course', 'user', 'permission')
    raw_id_fields = ('course', 'user')
    search_fields = ('course_id', 'course__name', 'user_id', 'user__username')


@admin.register(Criterion)
class CriterionAdmin(DjangoObjectActions, admin.ModelAdmin):
    list_display = ('name', 'clesson', 'assignment_rule')
    list_select_related = ('clesson', 'assignment_rule')
    raw_id_fields = ('clesson', 'assignment_rule')
    search_fields = ('name', 'clesson__id', 'assignment_rule__id', 'assignment_rule__title')

    change_actions = (
        'check_achievements_and_IDM',
        'apply_criterion',
        'apply_criterion_force',
    )

    def check_achievements_and_IDM(self, request, obj):
        message_level = messages.SUCCESS

        user_messages = []
        for action in obj.actions:
            if action.get('type') in [RequestAchievementAction.TYPE, RequestTeamAchievementAction.TYPE]:
                no_param = None
                for param in ['achievement_id', 'level', 'comment']:
                    if param not in action:
                        no_param = param
                        break
                if no_param is not None:
                    user_messages.append(_('Ошибка выдачи ачивки: не указан параметр {}').format(param))
                    message_level = messages.ERROR
                    continue

                try:
                    achievery_client.check_achievement(
                        achievement_id=action['achievement_id'],
                        level=str(action['level']),
                    )
                    user_messages.append(_('Ачивка {} готова к использованию').format(action['achievement_id']))

                except AchieveryHTTPError as ex:
                    user_messages.append(
                        _('Ошибка выдачи ачивки: status code: {}, reason: {}, text: {}').format(
                            ex.response.status_code,
                            ex.response.reason,
                            ex.response.text,
                        )
                    )
                    message_level = messages.ERROR

            if action.get('type') == RequestIDMRoleAction.TYPE:
                no_param = None
                for param in ['path', 'type', 'system', 'comment']:
                    if param not in action:
                        no_param = param
                        break
                if no_param is not None:
                    user_messages.append(_('Ошибка запроса IDM-роли: не указан параметр {}').format(param))
                    message_level = messages.ERROR
                    continue

                try:
                    idm_client.request_role(
                        system=action['system'],
                        path=action['path'],
                        fields_data=action.get('fields_data') or {},
                        user=request.user.username,
                        comment=action['comment'],
                        simulate=True,
                    )
                    user_messages.append(
                        _('IDM роль {}{} готова к использованию').format(action['system'], action['path'])
                    )

                except IDMHTTPError as ex:
                    user_messages.append(
                        _(
                            'Ошибка запроса IDM-роли: status code: {}, reason: {}, text: {}, x_system_request_id: {}'
                        ).format(
                            ex.response.status_code,
                            ex.response.reason,
                            ex.response.text,
                            ex.x_system_request_id,
                        )
                    )
                    if ex.response.status_code != 409:
                        message_level = messages.ERROR

        self.message_user(request=request, message=user_messages, level=message_level)

    check_achievements_and_IDM.label = _("Проверить ачивку и IDM-роль")
    check_achievements_and_IDM.short_description = _("Проверить ачивку и IDM-роль")

    @superuser_only_action
    def apply_criterion(self, request, obj: Criterion, force: bool = False):
        apply_criterion_task.delay(criterion_id=obj.id, force=force)

        self.message_user(request=request, message=_("Применение критерия запущено"))

    apply_criterion.label = _("Применить критерий")
    apply_criterion.short_description = _("Применить критерий")

    @superuser_only_action
    def apply_criterion_force(self, request, obj: Criterion):
        self.apply_criterion(request=request, obj=obj, force=True)

    apply_criterion_force.label = _("Применить критерий заново")
    apply_criterion_force.short_description = _("Применить критерий заново")


@admin.register(UserCriterion)
class UserCriterionAdmin(admin.ModelAdmin):
    raw_id_fields = (
        'criterion',
        'user',
    )
    list_filter = (
        (
            'user__username',
            build_dropdown_filter_with_title('Студенты'),
        ),
        (
            'criterion__name',
            build_dropdown_filter_with_title('Критерий'),
        ),
    )
    list_select_related = (
        'user',
        'criterion',
    )


@admin.register(CourseLessonEdge)
class CourseLessonEdgeAdmin(admin.ModelAdmin):
    raw_id_fields = (
        'assignment_rule',
        'parent_clesson',
        'child_clesson',
        'criterion',
    )
    list_display = (
        'assignment_rule',
        'parent_clesson',
        'child_clesson',
        'criterion',
    )
    list_select_related = (
        'assignment_rule',
        'parent_clesson',
        'child_clesson',
        'criterion',
    )


@admin.register(CourseLessonNode)
class CourseLessonNodeAdmin(admin.ModelAdmin):
    raw_id_fields = (
        'assignment_rule',
        'clesson',
    )
    list_display = (
        'assignment_rule',
        'clesson',
        'available',
    )
    list_select_related = (
        'assignment_rule',
        'clesson',
    )


@admin.register(AssignmentRule)
class AssignmentRuleAdmin(admin.ModelAdmin):
    raw_id_fields = (
        'course',
    )
    list_select_related = (
        'course',
    )
    list_display = (
        'course',
        'mandatory',
        'title',
    )


@admin.register(CourseFeedback)
class CourseFeedbackAdmin(admin.ModelAdmin):
    raw_id_fields = (
        'course',
        'user',
    )


@admin.register(CourseInvite)
class CourseInviteAdmin(admin.ModelAdmin):
    raw_id_fields = (
        'course',
    )
    list_display = (
        'course',
        'activation_link_href',
        'is_active',
        'activations',
    )
    list_filter = (
        'course',
    )
    list_select_related = (
        'course',
    )
    readonly_fields = (
        'activation_link_href',
    )

    def activation_link_href(self, obj):
        link = obj.activation_link
        if link:
            return format_html('<a href="{link}">{link}</a>', link=link)
        else:
            return ''

    activation_link_href.short_description = 'Ссылка для активации'
    activation_link_href.allow_tags = True

    def get_form(self, request, obj=None, **kwargs):
        form = super().get_form(request, obj, **kwargs)
        form.base_fields['key'].initial = uuid4().hex
        return form


@admin.register(UserCourseInviteActivation)
class UserCourseInviteActivationAdmin(admin.ModelAdmin):
    raw_id_fields = (
        'user',
        'course_invite',
    )
    readonly_fields = (
        'created',
        'modified',
    )


@admin.register(PeriodicCourse)
class PeriodicCourseAdmin(admin.ModelAdmin):
    list_display = ('course', 'period', 'date_created', 'date_updated')
    list_select_related = ('course',)
    raw_id_fields = (
        'course',
        'previous',
    )

    inlines = [
        PeriodicNotificationInline,
        PeriodicRoleDigestInlineAdmin,
    ]


@admin.register(PeriodicNotification)
class PeriodicNotificationAdmin(admin.ModelAdmin):
    fields = ('periodic_course', 'notify_type', 'delay', 'priority', 'parameters')
    raw_id_fields = ('periodic_course',)
    list_select_related = ('periodic_course',)


@admin.register(PeriodicStudentNotification)
class PeriodicStudentNotificationAdmin(DjangoObjectActions, admin.ModelAdmin):
    list_display = ('student', 'course', 'notification', 'date_created',)
    list_select_related = ('student', 'course', 'notification')
    raw_id_fields = (
        'student', 'course',
    )

    actions = [
        'notify',
    ]

    change_actions = [
        'notify',
    ]

    @takes_instance_or_queryset
    def notify(self, request, queryset):
        queryset = queryset.select_related('notification')
        for student_notification in queryset:
            try:
                message_level = messages.SUCCESS
                periodic_notify_student(student_notification)
                student_notification.status = PeriodicStudentNotification.STATUS_SENT
            except Exception:
                message_level = messages.ERROR
                student_notification.status = PeriodicStudentNotification.STATUS_ERROR
                student_notification.errors = traceback.format_exc()
            finally:
                student_notification.save()

            self.message_user(request, _("Отправка уведомления {}").format(student_notification.id), level=message_level)

    notify.label = _("Отправить уведомление")
    notify.short_description = _("Отправить уведомление")


@admin.register(ExcludedUser)
class ExcludedUserAdmin(admin.ModelAdmin):
    list_display = ('login', 'exclude_from_courses', 'exclude_from_issues', 'date_created')
    list_filter = ('exclude_from_courses', 'exclude_from_issues')
    search_fields = ('login',)


@admin.register(NotifyDayOff)
class NotifyDayOffAdmin(admin.ModelAdmin):
    list_display = ('__str__', 'is_active', 'valid_from', 'valid_to', 'date_created', 'date_updated')
    list_filter = ('is_active',)


@admin.register(RoleDigest)
class RoleDigestAdmin(SimpleHistoryAdmin):
    list_display = (
        'periodic_role_digest', 'user', 'process_status', 'target_issue_status', 'date_created', 'date_updated',
    )
    raw_id_fields = ('periodic_role_digest', 'user')
    list_filter = ('periodic_role_digest', 'user', 'process_status')
    history_list_display = ('tracker_issue_key', 'target_issue_status', 'process_status')


@admin.register(PeriodicRoleDigest)
class PeriodicRoleDigestAdmin(admin.ModelAdmin):
    list_display = ('periodic_course', 'role', 'is_active', 'date_created', 'date_updated')
    raw_id_fields = ('periodic_course',)
    list_filter = ('periodic_course', 'role', 'is_active')
