import logging
import operator
from builtins import filter, map, object, range
from datetime import timedelta
from functools import reduce
from itertools import chain, groupby, islice
from math import ceil

from future.utils import iteritems, itervalues

from django.conf import settings
from django.db import IntegrityError, transaction
from django.db.models import F
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _

from kelvin.accounts.models import User, UserStaffGroups
from kelvin.courses.models import CourseLessonLink
from kelvin.lesson_assignments.models import LessonAssignment
from kelvin.lessons.models import Lesson
from kelvin.lessons.services import copy_lesson
from kelvin.problems.answers import Answer
from kelvin.result_stats.models import CourseLessonStat, StudentCourseStat
from kelvin.result_stats.serializers import StudentCourseStatInJournalSerializer
from kelvin.results.models import AbstractLessonResult  # FIXME сверить с plato - модели различаются!!!
from kelvin.results.models import CourseLessonResult
from kelvin.tags.utils import get_user_staff_groups

# TODO поресерчить зачем была ленивая загрузка
# from plato.lesson_assignments import LessonAssignmentsRegistry
# LessonAssignment = LessonAssignmentsRegistry.get_model_lazy(
#     LessonAssignmentsRegistry.Models.LESSON_ASSIGNMENT,
# )
STAFF_GROUPS_DELIMITER = '/'
logger = logging.getLogger(__name__)


class BaseCourseGroupJournal(object):
    """
    Сводка результатов учеников по курсу
    """
    # Вспомогательные константы для подсчета медалей учеников
    PLACE_KEYS = ('first_place', 'second_place', 'third_place')
    PLACES_DICT = {key: 0 for key in PLACE_KEYS}

    def __init__(self, course):
        """
        Поля курса и данных журнала
        """
        self.course = course
        self.students = dict(
            [(student.id, student) for student in self.course.students.all()]
        )
        self._data = None
        # self._student_ids = None
        self._clesson_ids = None
        self._course_stats = None
        self._stat_map = None

    @property
    def student_ids(self):
        return list(self.students.keys())

    @property
    def clesson_ids(self):
        if self._clesson_ids is None:
            self._clesson_ids = self.course.courselessonlink_set.values_list(
                'id',
                flat=True,
            )
        return self._clesson_ids

    @property
    def course_stats(self):
        if self._course_stats is None:
            self._course_stats = StudentCourseStat.objects.filter(
                course_id=self.course.id,
                student_id__in=self.student_ids,
            ).order_by("-student_id")
        return self._course_stats

    def get_serializer(self):
        return StudentCourseStatInJournalSerializer(
            context={'clessons': self.clesson_ids}
        )

    @property
    def stat_map(self):
        if self._stat_map is None:
            self._stat_map = {
                stat.student_id: stat
                for stat in self.course_stats
            }
        return self._stat_map

    def set_default_places(self, data):
        for student_id in self.student_ids:
            data['students'].setdefault(
                student_id, dict(),
            ).update(self.PLACES_DICT)

    @property
    def data(self):
        """
        Данные журнала курса
        """
        if self._data is None:
            data = {
                'students': {
                    stat.student_id: self.get_serializer()
                        .to_representation(stat)
                    for stat in self.course_stats
                },
            }
            self.set_default_places(data)

            # Подсчет медалей. Пока берем топ по очкам среди всех уроков
            # Люди с нулевым баллом или его отсутствием в рейтинг не попадают
            # TODO: сейчас учитываются только очки. Возможно, нужно что-то еще
            students_have_stat = set(self.stat_map.keys())

            for clesson_id in map(str, self.clesson_ids):
                top = (
                    (
                        student_id,
                        self.stat_map[student_id].clesson_data.get(
                            clesson_id, {}).get('points', 0)
                    )
                    for student_id in students_have_stat
                )

                for place, top_item_group in enumerate(islice(groupby(
                        top, operator.itemgetter(1)), len(self.PLACE_KEYS))):
                    if top_item_group[0] == 0:
                        break
                    for top_item in top_item_group[1]:
                        data['students'][top_item[0]][
                            self.PLACE_KEYS[place]
                        ] += 1

            self._data = data
        return self._data

    def table(self):
        """
        Выгружает журнал в табличном виде. С полями
            id пользователя
            Фамилия
            Имя
            логин
            процент по занятию -| этот блок повторяется для каждого занятия
            ссылка на журнал ученика
        [https://st.yandex-team.ru/EDUCATION-1924](подробное описание формата)
        """
        clessons = (
            self.course.courselessonlink_set
            .filter(date_assignment__isnull=False)
            .order_by('order')
        )

        clesson_ids = [clesson.id for clesson in clessons]

        headers = [u'id пользователя', u'фамилия', u'имя', u'логин']
        headers += [u'{} : {}'.format(i + 1, clesson.lesson.name)
                    for i, clesson in enumerate(clessons)]
        headers += [u'ссылка на журнал ученика']
        table = [headers]

        journal_data = self.data['students']

        for student_id, student_data in sorted(journal_data.items()):
            student = self.students[student_id]

            # Если ученик ничего не решал в занятиии, то в `data` для него не
            # будет ключа 'lessons'. Так как в журнал нам его все равно нужно
            # включить, то введем значение по умолчанию
            student_results = [0] * len(clessons)

            if 'lessons' in student_data:
                student_results = [
                    value or 0
                    for clesson_id, value
                    in sorted(
                        list(student_data['lessons'].items()),
                        key=(lambda cl_id_val: clesson_ids.index(cl_id_val[0])
                             if cl_id_val[0] in clesson_ids else 0),
                    )
                    if clesson_id in clesson_ids
                ]

            table.append(
                [
                    student.id,
                    student.last_name,
                    student.first_name,
                    student.username
                ] +
                student_results +
                [
                    '{}lab/students/{}/courses/{}/'.format(
                        settings.FRONTEND_HOST, student_id, self.course.id
                    )
                ]
            )

        return table


class BaseLessonJournal(object):
    """
    Результаты группы учеников по занятию
    """

    # Очередные константы для использования в журнале по занятию.
    PROBLEM_CORRECT = 1
    PROBLEM_INCORRECT = 2
    PROBLEM_INCORRECT_DONE = 3
    PROBLEM_NO_INFO = 4

    def __init__(self, clesson):
        self.clesson = clesson

        self.students = {
            student.id: student
            for student in User.objects.filter(
                coursestudent__course_id=self.clesson.course_id,
            )
        }

        self._data = None
        self._students_results = None
        self._problem_links = None
        self._students_assignments = None
        self._student_ids = None

    @property
    def students_results(self):
        if self._students_results is None:
            self._students_results = (
                {}
                if not self.clesson.date_assignment else
                {
                    result.summary.student_id: result
                    for result in (
                        CourseLessonResult.objects.filter(
                            summary__clesson=self.clesson,
                            summary__student__isnull=False,
                            date_created__gte=self.clesson.date_assignment,
                        ).select_related('summary')
                        .order_by('date_updated', 'id')
                    )
                }
            )
        return self._students_results

    @property
    def problem_links(self):
        if self._problem_links is None:
            self._problem_links = (
                AbstractLessonResult.get_summarizable_lessonproblemlinks(
                    self.clesson.lesson_id
                )
            )
        return self._problem_links

    @property
    def students_assignments(self):
        if self._students_assignments is None:
            self._students_assignments = {
                assignment.student_id: assignment
                for assignment in LessonAssignment.objects.filter(
                    clesson=self.clesson,
                )
            }
        return self._students_assignments

    @property
    def student_ids(self):
        return list(self.students.keys())

    def get_summary(self, student_id):
        result = self.students_results.get(student_id)
        assigned_links = self.problem_links
        if student_id in self.students_assignments:
            assigned_problem_ids = set(
                self.students_assignments[student_id].problems)
            assigned_links = [
                link for link in assigned_links
                if link.id in assigned_problem_ids
            ]
        return AbstractLessonResult.get_summary(
            assigned_links, result, lesson_scenario=self.clesson,
        )

    @property
    def data(self):
        if self._data is None:
            self._data = dict(
                [
                    (student_id, self.get_summary(student_id))
                    for student_id in self.student_ids
                ]
            )
        return self._data

    def make_homework(self):
        """
        Составить домашнее задание
        """
        # определяем, какие задачи надо назначить каждому ученику
        homework_problem_link_ids_by_students = {}
        for student_id, student_data in iteritems(self.data):
            homework_problem_link_ids_by_students[student_id] = [
                problem_link_id for problem_link_id, summary in
                iteritems(student_data['problems'])
                if summary['status'] != Answer.SUMMARY_CORRECT
            ]
        if not homework_problem_link_ids_by_students:
            # в группе нет учеников
            return

        # множество всех вопросов в домашнем задании
        homework_problem_link_ids = reduce(
            operator.or_,
            list(map(set, itervalues(homework_problem_link_ids_by_students))),
        )

        with transaction.atomic():
            # создаем занятие из нужных вопросов
            old_to_new = copy_lesson(
                self.clesson.lesson,
                include_problem_link_ids=homework_problem_link_ids,
                name=Lesson.get_homework_name(),
            )

            # создаем новое курсозанятие
            clesson = CourseLessonLink.objects.create(
                course_id=self.clesson.course_id,
                lesson_id=old_to_new['new_lesson_id'],
                order=self.clesson.order,
                accessible_to_teacher=timezone.now(),
            )

            # Актуализируем назначения ученикам для копии занятия
            homework_problem_link_ids_by_students = {
                student_id: [
                    old_to_new['problems'].get(problem_link_id,
                                               problem_link_id)
                    for problem_link_id in problem_link_ids
                ]
                for student_id, problem_link_ids
                in iteritems(homework_problem_link_ids_by_students)
            }

            # создаем назначения ученикам
            homework_len = len(homework_problem_link_ids)
            assignments = [
                LessonAssignment(clesson=clesson, student_id=student_id,
                                 problems=student_problems)
                for student_id, student_problems in
                iteritems(homework_problem_link_ids_by_students)
                if len(student_problems) != homework_len
            ]
            LessonAssignment.objects.bulk_create(assignments)

        return clesson

    def table(self):
        """
        Выгружает журнал в табличном виде. С полями
            id пользователя
            Фамилия
            Имя
            логин
            номер задания -| этот блок повторяется для каждой задачи
            число попыток -| в данном clesson
            ссылка на результаты ученика по занятию
        [https://st.yandex-team.ru/EDUCATION-2088](подробное описание формата)
        """

        # В табличке мы хотим выдавать задачки по их порядку, а в журнале
        # хранятся данные только про lesson_problem_link. Так что нам
        # приходится сделать запрос в БД и создать табличку для матчинга.
        # Теоретически в поле `order` problem_link`a могут быть не
        # последовательные числа (например [2, 5, 1]). Тогда нам нужно
        # правильно хранить новый порядок (он будет с нуля)
        matching_id_order = {}
        for new_order, link_id in enumerate(
            self.clesson.lesson.lessonproblemlink_set
            .order_by('order')
            .values_list('id', flat=True)
            .all()
        ):
            matching_id_order[link_id] = new_order
        problems_count = len(matching_id_order)

        def get_status(summary_status, count_attempts, max_attempts):
            """
            Статус по задаче :
            1: решил верно
            2: решил неверно, но есть еще попытки
            3: решил неверно, нет попыток
            4: не нажимал на кнопку "ответить" (не решал)
            """
            if summary_status == Answer.SUMMARY_CORRECT:
                return self.PROBLEM_CORRECT
            if summary_status == Answer.SUMMARY_INCORRECT:
                if count_attempts < max_attempts:
                    return self.PROBLEM_INCORRECT
                return self.PROBLEM_INCORRECT_DONE
            return self.PROBLEM_NO_INFO

        header = (
            [u'id пользователя', u'Фамилия', u'Имя', u'Логин', u'Бывший сотрудник'] +
            list(chain.from_iterable(
                [
                    (i, u'{}-число попыток'.format(i))
                    for i in range(1, problems_count + 1)
                ]
            )) +
            [u'Ссылка на результаты ученика по занятию']
        )
        table = [header]
        for user_id, journal in list(self.data.items()):
            user = self.students[user_id]
            problem_results = [[]] * problems_count
            for problem_link_id, problem_info in list(
                journal['problems'].items()
            ):
                problem_results[matching_id_order[problem_link_id]] = (
                    get_status(
                        problem_info.get('status'),
                        problem_info.get('attempt_number', 0),
                        problem_info.get('max_attempts', 0),
                    ),
                    problem_info.get('attempt_number', 0)
                )
            table.append(
                [
                    user_id,
                    user.last_name,
                    user.first_name,
                    user.username,
                    user.is_dismissed,
                ] +
                list(chain.from_iterable(problem_results)) +
                [
                    '{host}lab/students/{student_id}/courses/'
                    '{course_id}/assignments/{clesson_id}/'.format(
                        host=settings.FRONTEND_HOST,
                        student_id=user_id,
                        course_id=self.clesson.course_id,
                        clesson_id=self.clesson.id,
                    )
                ]
            )

        return table


class StudentGroupMixin(object):
    """Добавляет возможность в журнале получить данные о подразделениях студентов из staff-reader"""

    def __init__(self, *args, **kwargs):
        super(StudentGroupMixin, self).__init__(*args, **kwargs)
        # выбираем из самодельного кеша только то, что младше "сейчас" - TTL записи в кеше
        self.user_staff_groups_map = {
            item.user_id: item.staff_groups
            for item in UserStaffGroups.objects.filter(
                date_updated__gt=timezone.now() - timedelta(seconds=settings.USER_STAFF_GROUP_CACHE_TTL),
            )
        }

    def get_student_groups(self, student_id):
        """Получить стафф-подразделения студента."""

        user_groups = self.user_staff_groups_map.get(student_id, None)

        # тут обязательно нужно проверять на None, потому что user_groups может быть пустой строкой и это ок
        if user_groups is not None:
            return user_groups

        # если в мапе данных не было - это значит, что их либо физически не было, либо они протухли
        user_groups = STAFF_GROUPS_DELIMITER.join(get_user_staff_groups(student_id))
        # создаем новую запись в мапе либо апдейтим протухшую
        try:
            UserStaffGroups.objects.update_or_create(
                user_id=student_id,
                defaults={
                    'staff_groups': user_groups,
                }
            )
        except IntegrityError:
            logger.warning("Seems like neighbour worker already generated record for student {}".format(
                student_id
            ))

        return user_groups

    def get_last_group(self, student_id):
        student_staff_groups = self.get_student_groups(student_id).split(STAFF_GROUPS_DELIMITER)
        return student_staff_groups[-1] if student_staff_groups else ''


class BaseStudentJournal(object):
    """
    Результаты ученика по курсу
    """
    # число занятий, которые учитываются при подсчете
    # процента успеваемости ученика
    LESSONS_FOR_SUCCESS_PERCENT = 7

    def __init__(self, student_id, course):
        self.student_id = student_id
        self.course = course
        self._data = None
        self._clessons = None
        self._clesson_ids = None
        self._student_results = None
        self._student_assignments = None
        self._course_lesson_stats = None
        self.now = timezone.now()

    @property
    def clessons(self):
        if self._clessons is None:
            self._clessons = self.course.courselessonlink_set.filter(
                date_assignment__isnull=False,
            ).order_by('date_assignment')
        return self._clessons

    @property
    def clesson_ids(self):
        if self._clesson_ids is None:
            self._clesson_ids = {link.id for link in self.clessons}
        return self._clesson_ids

    @property
    def student_results(self):
        if self._student_results is None:
            self._student_results = {
                result.summary.clesson_id: result
                for result in CourseLessonResult.objects.filter(
                    summary__student=self.student_id,
                    summary__clesson__in=self.clesson_ids,
                    date_created__gte=F('summary__clesson__date_assignment'),
                ).select_related('summary').order_by('date_updated')
            }
        return self._student_results

    @property
    def student_assignments(self):
        if self._student_assignments is None:
            self._student_assignments = {
                assign.clesson_id: assign
                for assign in LessonAssignment.objects.filter(
                    student=self.student_id,
                    clesson__in=self.clesson_ids,
                )
            }
        return self._student_assignments

    @property
    def course_lesson_stats(self):
        if self._course_lesson_stats is None:
            self._course_lesson_stats = {
                stat.clesson_id: stat
                for stat in CourseLessonStat.objects.filter(
                    clesson__in=self.clesson_ids,
                ).only('clesson_id', 'average_points', 'average_max_points')
            }
        return self._course_lesson_stats

    def check_if_clesson_to_be_counted(self, clesson):
        return not (
            clesson.id in self.student_assignments and
            self.student_assignments[clesson.id].problems == []
        )

    def get_clesson_points(self, clesson):
        if clesson.id in self.course_lesson_stats:
            stat = self.course_lesson_stats[clesson.id]
            clesson_points = stat.average_points
            clesson_max_points = stat.average_max_points
        else:
            clesson_points = 0
            clesson_max_points = 0
        return clesson_points, clesson_max_points

    def check_if_clesson_points_to_be_shown(self, clesson):
        return (
            clesson.mode != CourseLessonLink.CONTROL_WORK_MODE or
            clesson.evaluation_date and clesson.evaluation_date <= self.now
        )

    def get_problem_links(self, clesson, assigned=None):
        problem_links = (
            AbstractLessonResult.get_summarizable_lessonproblemlinks(clesson.lesson_id).select_related('problem')
        )
        if assigned is not None:
            problem_links = list(filter(
                lambda link: link.id in assigned,
                problem_links,
            ))
        return problem_links

    def get_clesson_record(self, clesson):
        clesson_points, clesson_max_points = self.get_clesson_points(clesson)

        record = {
            'id': clesson.id,
            'name': clesson.lesson.name,
            'date': clesson.date_assignment.strftime(
                settings.REST_FRAMEWORK['DATETIME_FORMAT']),
            'clesson_max_points': clesson_max_points,
        }

        # список назначенных вопросов
        record.update(
            AbstractLessonResult.get_summary(
                self.get_problem_links(
                    clesson,
                    (
                        self.student_assignments[clesson.id].problems
                        if clesson.id in self.student_assignments
                        else None
                    ),
                ),
                self.student_results.get(clesson.id),
                format_as_list=True,
                lesson_scenario=clesson,
            )
        )

        if self.check_if_clesson_points_to_be_shown(clesson):
            record['clesson_points'] = clesson_points
        else:
            record.pop('points')

        return record

    @property
    def data(self):
        """
        Данные журнала ученика
        """
        if self._data is None:
            self._data = [
                self.get_clesson_record(clesson)
                for clesson in self.clessons
                if self.check_if_clesson_to_be_counted(clesson)
            ]
        return self._data

    def get_success_percent(self):
        """
        Процент успеваемости ученика

        Считаем по нескольким последним результатам процент набранных баллов,
        округляя значение вверх
        """
        data_len = len(self.data)
        lessons_counted = 0
        points = 0
        max_points = 0
        for i in range(data_len - 1, -1, -1):
            if 'points' in self.data[i]:
                if self.data[i]['points'] is not None:
                    points += self.data[i]['points']
                max_points += self.data[i]['max_points']
                lessons_counted += 1
                if lessons_counted == self.LESSONS_FOR_SUCCESS_PERCENT:
                    break
        return (int(ceil(100 * float(points) / max_points))
                if max_points else None)


class StudentJournal(BaseStudentJournal):
    @property
    def student_results(self):
        if self._student_results is None:
            self._student_results = {
                result.summary.clesson_id: result
                for result in CourseLessonResult.objects.filter(
                    summary__student=self.student_id,
                    summary__clesson__in=self.clesson_ids,
                    date_created__gte=F('summary__clesson__date_assignment'),
                    work_out=False,
                ).select_related('summary').order_by('date_updated')
            }
        return self._student_results


class LessonJournal(StudentGroupMixin, BaseLessonJournal):
    # переопределяем числовые константы в виде строк
    PROBLEM_CORRECT = _(u"Правильно")
    PROBLEM_INCORRECT = _(u"Неправильно (есть попытки)")
    PROBLEM_INCORRECT_DONE = _(u"Неправильно (нет попыток)")
    PROBLEM_NO_INFO = _(u"Нет ответа")

    def __init__(self, clesson):
        super(LessonJournal, self).__init__(clesson)
        self._students_summaries = None

    @property
    def students_results(self):
        if self._students_results is None:
            self._students_results = (
                {}
                if not self.clesson.date_assignment else
                {
                    result.summary.student_id: result
                    for result in (
                        CourseLessonResult.objects.filter(
                            summary__clesson=self.clesson,
                            summary__student__isnull=False,
                            date_created__gte=self.clesson.date_assignment,
                            work_out=False,
                        ).select_related('summary')
                        .order_by('date_updated', 'id')
                    )
                }
            )
        return self._students_results

    @property
    def students_summaries(self):
        """
        Получаем (и сохраняем в инстансе) мапу из студента в саммари (CourseLessonSummary)
        для текущего clesson_id.
        Данная информация используется для обогащения базовой версии отчета датой последненго прохождения модуля,
        которую мы берем из пераметров саммари.
        """
        if self._students_summaries is None:
            self._students_summaries = {}
            if self.clesson.date_assignment:
                for result in CourseLessonResult.objects.filter(
                    summary__clesson=self.clesson,
                    summary__student__isnull=False,
                    date_created__gte=self.clesson.date_assignment,
                    work_out=False,
                ).select_related('summary'):
                    self._students_summaries[result.summary.student_id] = result.summary
        return self._students_summaries

    def table(self):
        base_table = super(LessonJournal, self).table()
        base_table_header_len = len(base_table[0])
        rownum = 0
        for table_record in base_table:
            table_record = table_record[:-1]  # удаляем ненужную колонку (ссылка на стр. модуля) из базовой версии
            if rownum == 0:
                base_table_header_len -= 1
                table_record += [
                    _(u"Дата первого прохождения"),
                    _(u"Дата последнего прохождения"),
                    _(u"Группа на стаффе"),
                ]
            else:
                if len(table_record) < base_table_header_len:
                    table_record += [""] * (base_table_header_len - len(table_record))
                user_id = table_record[0]
                if user_id in list(self.students_summaries.keys()):
                    table_record += [
                        # предполагаем, что в первой колонке идентификатор пользователя и получаем его саммари из мапы
                        self.students_summaries[user_id].date_created.strftime("%Y-%m-%d"),
                        self.students_summaries[user_id].date_updated.strftime("%Y-%m-%d"),
                    ]
                else:
                    table_record += ["", ""]

                table_record += [
                    self.get_student_groups(user_id),

                ]

            base_table[rownum] = table_record
            rownum += 1

        return base_table


class CourseGroupJournal(StudentGroupMixin, BaseCourseGroupJournal):

    def __init__(self, course):
        super(CourseGroupJournal, self).__init__(course)

    def table(self):
        """
        Выгружает журнал в табличном виде. С полями
            id пользователя
            Фамилия
            Имя
            логин
            процент по занятию -| этот блок повторяется для каждого занятия
            ссылка на журнал ученика
        """
        clessons = (
            self.course.courselessonlink_set
                .filter(date_assignment__isnull=False)
                .order_by('order')
        )
        clessons_len = len(clessons)

        clesson_ids = [clesson.id for clesson in clessons]

        headers = [u'id пользователя', u'фамилия', u'имя', u'логин', u'бывший сотрудник']
        for i, clesson in enumerate(clessons):
            headers += [
                u'{} : {}. Баллы'.format(i + 1, clesson.lesson.name),
                u'{} : {}. Прогресс'.format(i + 1, clesson.lesson.name)
            ]
        for i, clesson in enumerate(clessons):
            headers += [
                u'{} : результат по модулю'.format(i + 1)
            ]

        headers += [u'Группа на стаффе']

        table = [headers]

        journal_data = self.data['students']

        for student_id, student_data in sorted(journal_data.items()):
            student = self.students[student_id]

            # Если ученик ничего не решал в занятиии, то в `data` для него не
            # будет ключа 'lessons'. Так как в журнал нам его все равно нужно
            # включить, то введем значение по умолчанию
            student_results = ['0 / 0', 0] * clessons_len
            student_lesson_links = [''] * clessons_len

            if 'lessons' in student_data:
                ordered_lessons = sorted(
                    list(student_data['lessons'].items()),
                    key=lambda cl_id_val: clesson_ids.index(cl_id_val[0]) if cl_id_val[0] in clesson_ids else 0
                )

                student_results = []
                for clesson_id, value in ordered_lessons:
                    if clesson_id in clesson_ids:
                        points = value.get('points', 0)
                        max_points = value.get('max_points', 0)
                        progress = value.get('progress', None)
                        points_str = '{} / {}'.format(points, max_points) if max_points else ''
                        progress_str = progress if progress is not None else ''
                        student_results += [points_str, progress_str]

                student_lesson_links = [
                    '{}laboratory/course/{}/assignments/{}/students/{}'.format(
                        settings.FRONTEND_HOST,
                        clesson.course_id,
                        clesson.id,
                        student_id
                    )
                    for clesson in clessons
                ]

            table.append([
                student.id,
                student.last_name,
                student.first_name,
                student.username,
                student.is_dismissed,
            ] + student_results + student_lesson_links + [
                self.get_student_groups(student.id)
            ])

        return table

    @property
    def data(self):
        """
        Данные журнала курса
        """
        if self._data is None:
            data = {
                'students': {
                    s['student_id']: s
                    for s in StudentCourseStatInJournalSerializer(
                        self.course_stats,
                        many=True,
                        context={'clessons': self.clesson_ids}
                    ).data
                }
            }

            self._data = data
        return self._data

    def data_with_staff_groups(self):
        data = self.data
        for student_data in data['students'].values():
            last_group = self.get_last_group(student_data['student_id'])
            student_data['staff_group'] = {
                'name': last_group,
                'link': settings.STAFF_DEPARTMENT_BASE_URL.format(last_group)
            }

        return data
