from urllib.parse import urljoin

from django.conf import settings
from django.contrib.auth import get_user_model

from kelvin.common.sender import SenderEmail
from kelvin.courses.models import PeriodicStudentNotification

User = get_user_model()


class MandatoryCourseEmail(SenderEmail):
    template_name = 'mandatory_course_email'

    def __init__(self, user: User, course, **kwargs):
        variables = {
            'student_name': user.username,
            'course_link': urljoin(
                settings.FRONTEND_HOST,
                f'courses/my/course/{course.id}',
            ),
            'course_name': course.name,
        }
        super().__init__(user, variables, **kwargs)


class PeriodicNotificationEmail(SenderEmail):
    def __init__(self, student_notification: PeriodicStudentNotification, **kwargs):
        self.notification = student_notification.notification
        user = student_notification.student.student
        course = student_notification.course

        is_newbie = self._check_user_joined_at(user)
        self.template_name = 'new' if is_newbie else 'renew'

        variables = {
            'student_name': user.username,
            'course-name': course.name,
            'is_newbie': is_newbie,
        }
        super().__init__(user, variables, **kwargs)

    def _check_user_joined_at(self, user) -> bool:
        """
        проверяет является ли пользователь новичком,
        по дате выхода на работу (со стаффа)
        """
        from kelvin.tags.models import TagTypeStaffStartDate as Tag
        joined_at = Tag.cast(Tag.get_staff_start_date(user))
        return joined_at >= Tag.cast(settings.PERIODIC_NOTIFICATION_NEWBIE_START_DATE)

    @property
    def template(self) -> str:
        # выбираем нужный шаблон письма из параметров, или берем default
        parameters = self.notification.parameters
        templates = parameters.get('templates')
        template_id = templates.get(self.template_name, 'default')

        if not template_id:
            raise ValueError("Notification '{}' has no template!".format(self.notification.pk))

        return template_id
