import json

from django.core.management import BaseCommand
from django.core.serializers.json import DjangoJSONEncoder

from kelvin.courses.models import CourseStudent, Course, CourseLessonLink
from kelvin.lessons.models import LessonProblemLink
from kelvin.resources.models import Resource
from kelvin.results.models import CourseLessonResult, CourseLessonSummary
from kelvin.scorm.models import Scorm

from tqdm import tqdm


class Command(BaseCommand):
    help = "Эскпортировать данные по курсу, модулям и прогрессу студентов"

    def add_arguments(self, parser):
        parser.add_argument(
            '--course_id', type=int,
            help='ID курса, данные которого хотим получить',
        )
        parser.add_argument(
            '--output', type=str,
            help='Имя файла, в который хотим направить вывод'
        )

    def handle(self, *args, **options):
        output = options.get('output')
        course_id = options['course_id']
        course_data = {'lessons': [], 'results': [], 'resources': {}}

        course = Course.objects.filter(pk=course_id).first()
        if not course:
            self.stderr.write(f"Курса с id {course_id} не найдено")
            return

        students = course.students.all()
        lessons = course.lessons.all()

        course_data['course'] = {
            'id': course.id,
            'name': course.name,
            'owner': {
                'id': course.owner.id,
                'yauid': course.owner.yauid,
                'username': course.owner.username,
            },
            'description': course.description,
            'created': course.date_created,
            'modified': course.date_updated,
        }

        course_data['users'] = [
            {
                'id': cs.student.id,
                'yauid': cs.student.yauid,
                'username': cs.student.username,
                'date_created': cs.date_created,
                'date_completed': cs.date_completed,
                'completed': cs.completed,
            } for cs in CourseStudent.objects.filter(student_id__in=students.values_list('id', flat=True),
                                                     course_id=course.id).select_related('student')
        ]

        for lesson in tqdm(lessons):
            lesson_problems = LessonProblemLink.objects.filter(lesson_id=lesson.id)

            problems = []
            for lesson_problem in lesson_problems:
                if lesson_problem.type == lesson_problem.TYPE_COMMON:
                    problem_type = 'problem'
                    content_attr = 'markup'
                else:
                    problem_type = 'theory'
                    content_attr = 'content'

                problem = getattr(lesson_problem, problem_type)
                problems.append(
                    {
                        'id': problem.id,
                        'name': problem.name,
                        'type': lesson_problem.type,
                        'author': problem.owner.username,
                        'content': getattr(problem, content_attr),
                        'resources': [
                            {
                                'id': resource_id,
                            } for resource_id in problem.resources.values_list('id', flat=True)
                        ]
                    }
                )

                for resource in problem.resources.all():
                    course_data['resources'][resource.id] = {
                        'lesson_id': lesson.id,
                        'problem_id': problem.id,
                        'file': resource.file.url,
                    }

                for layout in getattr(problem, 'markup', {}).get('layout', []):
                    content = layout.get('content', {})
                    if not content.get('type') == 'scorm':
                        continue

                    resource_id = content.get('options', {}).get('sco', {}).get('id')
                    resource = Resource.objects.filter(pk=resource_id).first()
                    if resource:
                        url = resource.file.url

                        course_data['resources'][resource_id] = {
                            'lesson_id': lesson.id,
                            'problem_id': problem.id,
                            'file': url,
                        }

            course_lesson = CourseLessonLink.objects.get(course_id=course_id, lesson_id=lesson.id)
            course_data['lessons'].append(
                {
                    'id': lesson.id,
                    'name': lesson.name,
                    'order': course_lesson.order,
                    'problems': problems,
                }
            )

            self.stdout.write(f"Найдено {lesson_problems.count()} задач в модуле {lesson.id}:{lesson.name}")

            for student in tqdm(students):
                cls = CourseLessonSummary.objects.filter(
                    student_id=student.id,
                    clesson__course_id=course_id,
                    clesson__lesson_id=lesson.id,
                ).first()

                clr = CourseLessonResult.objects.filter(summary=cls).first()
                if not clr:
                    continue

                result = {
                    'user_id': student.id,
                    'lesson_id': lesson.id,
                    'points': clr.points,
                    'answers': clr.answers,
                    'created': clr.date_created,
                    'modified': clr.date_updated,
                }

                scorm = Scorm.objects.filter(
                    student=student.id,
                    clesson__course_id=course_id,
                    clesson__lesson_id=lesson.id,
                ).first()

                if scorm:
                    result['scorm_data'] = scorm.data

                course_data['results'].append(result)

        self.stdout.write(f"Найдено {lessons.count()} модулей, {students.count()} студентов и "
                          f"{len(course_data['resources'])} ресурсов")

        if not output:
            return json.dumps(course_data, cls=DjangoJSONEncoder, ensure_ascii=False, sort_keys=True)
        else:
            with open(output, "w", encoding='utf-8') as output_file:
                json.dump(course_data, output_file, cls=DjangoJSONEncoder, ensure_ascii=False, sort_keys=True)
