import logging
from builtins import object

from django.conf import settings
from django.contrib.postgres.fields import JSONField
from django.core.cache import caches
from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.courses.models import CourseStudent

logger = logging.getLogger(__name__)

user_course_cache = caches['user_course']


@python_2_unicode_compatible
class AssignmentRule(models.Model):
    course = models.ForeignKey(
        'courses.Course',
        verbose_name=_("Курс"),
    )
    formula = JSONField(
        default=[],
        verbose_name=_("Правило назначения в виде псевдо-КНФ"),
    )
    mandatory = models.BooleanField(
        default=False,
        verbose_name=_("Обязательный курс"),
    )
    title = models.CharField(
        verbose_name=_("Название "),
        max_length=512,
        blank=True,
        null=True,
    )

    def __str__(self):
        return "Title: {} Course: {}".format(self.title, self.course.id)

    def save(self, *args, **kwargs):
        from kelvin.courses.tasks import add_course_to_every_student_by_assignment_rule

        old_state = AssignmentRule.objects.filter(pk=self.pk).first()
        super(AssignmentRule, self).save(*args, **kwargs)

        if settings.USER_COURSE_CACHE_ENABLE:
            cached_course_user_ids = user_course_cache.get(
                settings.COURSE_CACHE_KEY_TMPL.format(self.course.id)
            )

            if cached_course_user_ids:
                #  если в кеше есть пользователи, которые сматчены с курсом данного правила назначения,
                #  то надо почистить кеш курсов каждого пользователя и почистить кеш пользователей самого курса
                for user_id in cached_course_user_ids:
                    logger.info("Invalidating user {} cache".format(user_id))
                    user_course_cache.delete(
                        settings.USER_CACHE_KEY_TMPL.format(user_id)
                    )
                    user_course_cache.delete(
                        settings.USER_CACHE_LAST_UPDATE_KEY_TMPL.format(user_id)
                    )
                logger.info("Invalidating course {} cache".format(self.course.id))
                user_course_cache.delete(
                    settings.COURSE_CACHE_KEY_TMPL.format(self.course.id)
                )

        if old_state:
            if old_state.mandatory != self.mandatory:
                if self.mandatory:
                    # мы должны всем пользователям проекта принудительно добавить связь с курсом, если ее еще нет
                    # кроме того, такая связь должна содержать ссылку на это обязательное правило назначения
                    # это нужно для того, чтобы давть возможность пользователю отвязаться от курса, когда правило
                    # назначения перестанет быть обязательным
                    add_course_to_every_student_by_assignment_rule.delay(self.id)
                else:
                    # правило назначения перестало быть обязательным - надо отвязать его от связей CourseStudent """
                    CourseStudent.objects.filter(assignment_rule=self).update(assignment_rule=None)
            elif self.mandatory and old_state.formula != self.formula:
                # значение флага mandatory не изменилось и осталось True, но изменилась формула
                # a значит, вероятно, изменился и скоуп подходящих под правило пользователей
                # надо тоже запустить пересчет
                add_course_to_every_student_by_assignment_rule.delay(self.id)
            else:
                # значение флага mandatory не изменилось, значение формулы - тоже
                # явным образом ничего не делаем в таком случае
                pass
        elif self.mandatory:
            # появилось новое обязательное правило назначения
            # если оно обязательное, то надо всем пользвателям проекта соответствующего курса добавить курс
            add_course_to_every_student_by_assignment_rule.delay(self.id)
        else:
            # появилось новое необязательное правило назначения
            # пока явно ничего не делаем
            pass

    class Meta(object):
        verbose_name = _('Правило назначения курса пользователям')
        verbose_name_plural = _('Правила назначения курса пользователям')
        unique_together = (('course', 'formula'), )
