import logging
from builtins import object

from model_utils import FieldTracker

from django.conf import settings
from django.db import models
from django.utils import timezone
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.accounts.utils import get_user_rules_from_cache
from kelvin.common.model_mixins import AvailableForSupportMixin, InfoMixin, TimeStampMixin, UserBlameMixin
from kelvin.common.utils import COLOR_LENGTH, generate_color_from_set, generate_safe_code
from kelvin.common.validators import validate_color
from kelvin.group_levels.models import GroupLevel
from kelvin.lessons.models import Lesson
from kelvin.subjects.models import Subject

from .course_lesson_graph import CourseLessonGraphFactory
from .course_lesson_link import CourseLessonLink
from .progress_indicator import ProgressIndicator

logger = logging.getLogger(__name__)


@python_2_unicode_compatible
class BaseCourse(InfoMixin, TimeStampMixin):
    """
    Учебный курс
    """
    COLORS_SET = ('#018a96', '#015093', '#684bb3', '#8737a2',
                  '#b51c4f', '#b65a22', '#819922', '#2d934c')
    CODE_LENGTH = 8

    USUAL_COURSE = 1
    BOOK_COURSE = 2
    COURSE_TYPES = (
        (USUAL_COURSE, _('Обычный курс')),
        (BOOK_COURSE, _('Книга')),
    )

    name = models.CharField(
        verbose_name=_('Название учебного курса'),
        max_length=255,
    )
    subject = models.ForeignKey(
        Subject,
        verbose_name=_('Учебный предмет курса'),
    )
    lessons = models.ManyToManyField(
        Lesson,
        through=CourseLessonLink,
        verbose_name=_('Занятия курса'),
        blank=True,
    )
    mode = models.IntegerField(
        verbose_name=_('Тип курса'),
        choices=COURSE_TYPES,
        default=USUAL_COURSE,
    )
    owner = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Владелец курса'),
    )
    source_courses = models.ManyToManyField(
        'self',
        verbose_name=_('Курсы источники'),
        blank=True,
        symmetrical=False,
    )
    author = models.CharField(
        verbose_name=_('Автор'),
        max_length=1024,
        blank=True,
    )
    color = models.CharField(
        verbose_name=_('Отображаемый цвет курса'),
        max_length=COLOR_LENGTH,
        blank=True,
        validators=[validate_color],
    )
    cover = models.ForeignKey(
        'resources.Resource',
        verbose_name=_('Обложка'),
        null=True,
        blank=True,
        related_name='cover_for_courses',
    )
    students = models.ManyToManyField(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Ученики курса'),
        related_name='courses',
        blank=True,
        through='courses.CourseStudent',
    )
    copy_of = models.ForeignKey(
        'self',
        verbose_name=_('Копией какого курса является'),
        related_name='copies',
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )
    group_levels = models.ManyToManyField(
        GroupLevel,
        verbose_name=_('Классы'),
        blank=True,
    )
    free = models.BooleanField(
        default=False,
        verbose_name=_('Бесплатный курс'),
        db_index=True,
    )
    description = models.TextField(
        verbose_name=_('Описание курса'),
        blank=True,
    )
    code = models.CharField(
        verbose_name=_('Код для добавления ученика в группу курса'),
        max_length=CODE_LENGTH,
        blank=True,
        null=True,
        unique=True,
        default=None,
    )
    date_closed = models.DateTimeField(
        verbose_name=_('Дата закрытия курса'),
        default=None,
        null=True,
        blank=True,
    )
    allow_anonymous = models.BooleanField(
        verbose_name=_('Анонимное прохождение курса'),
        default=False,
    )
    news = models.TextField(
        verbose_name=_('Новости курса'),
        blank=True,
    )
    progress_indicator = models.ForeignKey(
        ProgressIndicator,
        verbose_name=_('Индикатор прогресса'),
        on_delete=models.SET_NULL,
        blank=True,
        null=True,
    )
    journal = models.FileField(
        verbose_name=_('Файл с csv журналом'),
        upload_to='course_journal',
        null=True,
        blank=True,
    )

    # def copy(self, owner=None, reset_clessons=True):
    #     """
    #     Copies the course with course lessons and change current instance.
    #
    #     :param owner: owner of copied course
    #     :param reset_clessons: need to edit clessons fields or not
    #     """
    #     copy_course(self, owner, reset_clessons)

    def add_clesson(self, clesson):
        """
        Добавляет занятие к курсу, изменяет параметр `clesson` и копирует его,
        занятие становится нередактируемым

        :type clesson: CourseLessonLink
        """
        clesson.pk = None
        clesson.course_id = self.id
        clesson.order = None
        clesson.accessible_to_teacher = timezone.now()
        clesson.date_assignment = None
        clesson.lesson_editable = False
        clesson.save()

    # def add_student(self, student):
    #     add_student(self, student)

    def __str__(self):
        """
        Название курса с идентификатором
        """
        return '{0}: {1}'.format(self.id, self.name)

    def save(self, *args, **kwargs):
        if not self.color:
            queryset = Course.objects.filter(owner=self.owner)
            if self.pk:
                queryset = queryset.exclude(id=self.id)
            used_colors = queryset.values_list('color', flat=True)

            self.color = generate_color_from_set(
                Course.COLORS_SET,
                used_colors,
            )

        if not self.code:
            """
            При сохранении генерируем код, если его раньше не было
            """
            self.code = generate_safe_code(Course.CODE_LENGTH)

        super(BaseCourse, self).save(*args, **kwargs)

    class Meta(object):
        abstract = True
        ordering = ['id']
        verbose_name = _('Курс')
        verbose_name_plural = _('Курсы')


class Course(BaseCourse, AvailableForSupportMixin, UserBlameMixin):
    """
    Учебный курс
    """
    copyright = models.CharField(
        verbose_name=_('Копирайт'),
        max_length=255,
        blank=True,
    )
    supports_web = models.BooleanField(
        verbose_name=_('Курс поддерживается на вебе'),
        default=True,
    )
    supports_ios = models.BooleanField(
        verbose_name=_('Курс поддерживается на iOS'),
        default=True,
    )
    supports_android = models.BooleanField(
        verbose_name=_('Курс поддерживается на Android'),
        default=False,
    )
    project = models.ForeignKey(
        'projects.Project',
        verbose_name=_('Проект курса'),
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )
    score_count = models.IntegerField(
        verbose_name=_('Количество оценок курса'),
        default=0,
    )
    average_score = models.FloatField(
        verbose_name=_('Средняя оценка курса'),
        default=0,
    )
    journal_resource = models.ForeignKey(
        'resources.Resource',
        verbose_name=_('Файл с журналом'),
        null=True,
        on_delete=models.SET_NULL,
    )
    allowed_tvm_services = models.ManyToManyField(
        to='tvm.TVMService',
        through='courses.CourseTVMService',
        blank=True,
        verbose_name=_('Разрешенные tvm-сервисы'),
    )

    def is_assigned(self, student_id):
        return CourseStudent.objects.filter(
            course_id=self.id,
            student_id=student_id,
            deleted=False
        ).first() is not None

    def is_assigned_mandatory(self, student_id):
        return CourseStudent.objects.filter(
            course_id=self.id,
            student_id=student_id,
            deleted=False,
            assignment_rule_id__gt=0,
        ).first() is not None

    tracker = FieldTracker(fields=[
        'available_for_support',
    ])

    class Meta(BaseCourse.Meta):
        permissions = (
            ('can_edit_course_in_lab', "Can edit course in lab"),
        )


@python_2_unicode_compatible
class BaseCourseStudent(TimeStampMixin):
    """
        Связь ученика и курса
        """
    student = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Ученик'),
    )
    course = models.ForeignKey(
        'courses.Course',
        verbose_name=_('Курс'),
    )

    class Meta(object):
        abstract = True
        verbose_name = _('Ученик курса')
        verbose_name_plural = _('Ученики курсов')
        unique_together = ['student', 'course']

    def __str__(self):
        return 'Курс: {0}, Ученик: {1} (id: {2})'.format(
            self.course_id, self.student_id, self.id)


class CourseStudent(BaseCourseStudent):
    completed = models.BooleanField(
        verbose_name=_('Завершил курс'),
        default=False,
    )
    date_completed = models.DateTimeField(
        verbose_name=_('Дата завершения курса'),
        blank=True,
        null=True,
    )
    deleted = models.BooleanField(
        verbose_name=_('Удален из "моих"'),
        default=False
    )
    assignment_rule = models.ForeignKey(
        'courses.AssignmentRule',
        on_delete=models.SET_NULL,
        verbose_name=_('Обязательное правило назначения'),
        help_text=_('Заполняется и обнуляется автоматически при изменении значения AssignmentRule.mandatory'),
        blank=True,
        null=True,
    )

    tracker = FieldTracker(fields=[
        'completed',
    ])

    def save(self, *args, **kwargs):
        need_graph_init = not self.pk or self.deleted is False

        if self.completed and not self.date_completed:
            self.date_completed = timezone.now()

        super(CourseStudent, self).save(*args, **kwargs)

        if need_graph_init:
            # при добавлении курса в "Мои" хотим заполнить траекторию модулей для пользователя
            # делаем это после основного сохранения, чтобы оно в любом случае продолжило работать
            rule_id = self.assignment_rule_id
            if not rule_id:
                user_matched_to_rules = [
                    x[0] for x in get_user_rules_from_cache(self.student) if x[1] == self.course_id
                ]
                if not user_matched_to_rules:
                    logger.warning("Cannot find any rule matched to user {}. Cannot actualize UserCLessonState.".format(
                        self.student_id
                    ))
                    return
                rule_id = user_matched_to_rules[0]

            # актуализируем граф и используем для этого первое попавшееся правило, с которым матчится пользователь
            CourseLessonGraphFactory.actualize_user_graph(
                user_id=self.student_id,
                assignment_rule_id=rule_id
            )
