from builtins import object

import networkx as nx

from django.apps import apps
from django.conf import settings
from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

DEFAULT_CLESSON_IS_AVAILABLE = settings.COURSES_DEFAULT_CLESSON_IS_AVAILABLE


@python_2_unicode_compatible
class CourseLessonNode(models.Model):
    assignment_rule = models.ForeignKey(
        to='courses.AssignmentRule',
        verbose_name=_('Правило назначения'),
        related_name='clesson_graph_nodes',
        help_text=_('Правило, группирующее пользователей для прохождения графа модулей')
    )
    clesson = models.OneToOneField(
        to='courses.CourseLessonLink',
        verbose_name=_('Курсозанятие'),
        related_name='clesson_graph_node',
    )
    available = models.BooleanField(
        verbose_name=_('Изначальная доступность'),
        default=True,
        blank=False,
        null=False,
    )

    class Meta(object):
        verbose_name = _('Условие открытия курсозанятия')
        verbose_name_plural = _('Условия открытия курсозанятий')

    def __str__(self):
        return "{} -> {}".format(
            self.clesson,
            self.available
        )


@python_2_unicode_compatible
class CourseLessonEdge(models.Model):
    """
    Модель для хранения ребер графа модулей в БД kelvin
    """
    assignment_rule = models.ForeignKey(
        to='courses.AssignmentRule',
        verbose_name=_('Правило назначения'),
        related_name='clesson_graph_edges',
        help_text=_('Правило, группирующее пользователей для прохождения графа модулей')
    )
    parent_clesson = models.ForeignKey(
        to='courses.CourseLessonLink',
        verbose_name=_('Родительское курсозанятие'),
        related_name='clesson_edge_parent',
    )
    child_clesson = models.ForeignKey(
        to='courses.CourseLessonLink',
        verbose_name=_('Дочернее курсозанятие'),
        related_name='clesson_edge_child',
    )
    criterion = models.ForeignKey(
        to='courses.Criterion',
        verbose_name=_('Критерий перехода между модулями')
    )

    def get_tuple(self):
        """
        Возвращает тройку значений: родитель, наследник, критерий в виде tuple
        Может пригодиться для облегчения инициализации графов
        """
        return (self.parent_clesson.id, self.child_clesson.id, self.criterion.id)

    def __str__(self):
        return "{} -> {} ({})".format(
            self.parent_clesson_id,
            self.child_clesson_id,
            self.criterion
        )

    class Meta(object):
        verbose_name = _('Условие перехода между курсозанятиями')
        verbose_name_plural = _('Условия переходов между курсозанятиями')


@python_2_unicode_compatible
class UserCLessonState(models.Model):
    user = models.ForeignKey(
        to='accounts.User',
        verbose_name=_('Пользователь'),
        blank=False,
        null=False,
    )
    clesson = models.ForeignKey(
        to='courses.CourseLessonLink',
        verbose_name=_('Курсозанятие'),
        blank=False,
        null=False,
    )
    assignment_rule = models.ForeignKey(
        to='courses.AssignmentRule',
        verbose_name=_('Правило назначения'),
        related_name='clesson_states',
        help_text=_('Правило, по которому пользователь в курсе')
    )
    available = models.BooleanField(
        verbose_name=_('Доступно'),
        default=True,
        blank=False,
        null=False,
    )

    class Meta(object):
        verbose_name = _('Доступность курсозанятия пользователю')
        verbose_name_plural = _('Доступность курсозанятий пользователям')
        unique_together = ('user', 'clesson')

    def __str__(self):
        return "Студент: {} Курс: {} Модуль: {} Доступен: {}".format(
            self.user.username,
            self.clesson.course_id,
            self.clesson_id,
            self.available
        )


class CourseLessonGraph(object):

    """
    Класс для представления в памяти графа модулей.
    Нужен для инкапсуляции способа хранения ребер графа ( в настоящее время это kelvin БД )
    и предоставления необходимого интерфейса для работы с графом модулей курса.
    """
    AVAILABLE_KEY = 'available'

    graph = property()

    def __init__(self, assignment_rule_id, edges=[], nodes=[], default_node_is_available=DEFAULT_CLESSON_IS_AVAILABLE):
        """
        Инициализация в памяти графа модулей курса по переданному набору рёбер
        :param assignment_rule_id: идентификатор существующего kelvin-правила назначения ( мы привязываем граф
        не к курсу, а к группе пользователей + курс, которая задается правилом назначения. Так как правило
        назначения содержит ссылку на курс, то, получается, что граф привязывается не к курсу, а к паре: правило + курс.
        :param edges: массив троек вида ( (<parent_clesson_id>, <child_clesson_id>, <criterion_id>), (...) )
        """
        # используем ориентированный мульти-граф,
        # потому что допускаем наличие нескольких рёбер между одними и теми же двумя вершинами
        self.__graph = nx.MultiDiGraph(
            assignment_rule=assignment_rule_id
        )

        for edge in edges:
            self.__graph.add_edge(
                edge[0],  # идентификатор родительского модуля в качестве parent-ноды
                edge[1],  # идентификатор родительского модуля в качестве child-ноды
                criterion=edge[2],  # идентификатор критерия в качестве атрибута ребра
            )

        for node in self.__graph.nodes:
            self.__graph.nodes[node][self.AVAILABLE_KEY] = default_node_is_available

        for node in nodes:
            self.__graph.add_node(
                node[0],
                **{
                    self.AVAILABLE_KEY: node[1],
                }
            )

    @graph.getter
    def graph(self):
        return self.__graph

    def get_graph_id(self):
        return self.__graph.graph['assignment_rule']

    def add_edge(self, parent_clesson_id, child_clesson_id, criterion_id):
        self.__graph.add_edge(parent_clesson_id, child_clesson_id, criterion=criterion_id)
        self.__graph.add_node(parent_clesson_id, **{
            self.AVAILABLE_KEY: DEFAULT_CLESSON_IS_AVAILABLE,
        })
        self.__graph.add_node(child_clesson_id, **{
            self.AVAILABLE_KEY: DEFAULT_CLESSON_IS_AVAILABLE,
        })

    def add_node(self, clesson_id, default_node_is_available=DEFAULT_CLESSON_IS_AVAILABLE):
        self.__graph.add_node(clesson_id, available=default_node_is_available)

    def has_node(self, clesson_id):
        """ проверяет, есть ли в графе нода с указанным clesson_id """
        return clesson_id in set(self.__graph.nodes)

    def set_clesson_available(self, clesson_id, available=True):
        if not self.has_node(clesson_id):
            return
        self.__graph.nodes[clesson_id][self.AVAILABLE_KEY] = available

    def get_clesson_available(self, clesson_id):
        """
        возвращает доступность вершины в графе
        для отстутствующей вершины возвращает False
        """
        if not self.has_node(clesson_id):
            return False
        return self.__graph.nodes[clesson_id][self.AVAILABLE_KEY]

    def get_sub_clessons(self, clesson_id, criterion_id=None):
        if criterion_id is None:
            # если не передали критерий, возвращаем все инцидентные вершины
            return list(self.__graph.successors(clesson_id))
        else:
            # иначе возвращаем только вершины, инцидентные ребрам с соответствующим критерием
            out_edges = self.__graph.edges(clesson_id, 'criterion')
            return [x[1] for x in out_edges if x[2] == criterion_id]

    def get_parent_clessons(self, clesson_id):
        return list(self.__graph.predecessors(clesson_id))


class CourseLessonGraphFactory(object):
    """
    Фабрика графов модулей. Десериализует/сериализует граф из/в нескольких CoureseLessonEdge-й.
    """
    USER_ID_ATTR = '__user_id'

    @staticmethod
    def serialize(graph):
        """
        Принимаем на вход экземпляр класса CourseLessonGraph и сериализует его в БД kelvin-а
        через модель CourseLessonEdge ( в БД мы храним набор рёбер , принадлежащий конкретному курсу )
        :param graph_intance: экземпляр класса CourseLessonGraph
        :return: None
        :raise: RuntimeError, если во время сериализации что-то идет "не так"
        """
        user_id = getattr(graph, CourseLessonGraphFactory.USER_ID_ATTR, None)
        if user_id is not None:
            raise RuntimeError("Trying to serialize default graph with user {} state".format(user_id))

        nodes_to_serialize = set([])  # для того, чтобы при сериализации нод не делать лишних запросов

        edges = dict(graph.graph.edges)
        for edge_key in edges:
            edge, _ = CourseLessonEdge.objects.update_or_create(
                assignment_rule_id=graph.get_graph_id(),
                parent_clesson_id=edge_key[0],
                child_clesson_id=edge_key[1],
                criterion_id=edges[edge_key]['criterion'],
            )
            nodes_to_serialize.add(edge_key[0])
            nodes_to_serialize.add(edge_key[1])

        for node_key in nodes_to_serialize:
            node, _ = CourseLessonNode.objects.update_or_create(
                assignment_rule_id=graph.get_graph_id(),
                clesson_id=node_key,
                defaults={
                    'available': graph.graph.nodes[node_key].get(
                        CourseLessonGraph.AVAILABLE_KEY,
                        settings.COURSES_DEFAULT_CLESSON_IS_AVAILABLE
                    )
                }
            )

    @staticmethod
    def serialize_for_user(graph):
        """
        Сериализуем граф в модель состояния пользователь-модуль
        """
        user_id = getattr(graph, CourseLessonGraphFactory.USER_ID_ATTR, None)
        if user_id is None:
            raise RuntimeError("Trying to serialize default graph for user with no user")

        nodes = dict(graph.graph.nodes)

        for node_key in nodes:
            node, _ = UserCLessonState.objects.update_or_create(
                user_id=user_id,
                clesson_id=node_key,
                assignment_rule_id=graph.get_graph_id(),
                defaults={
                    'available': nodes[node_key].get(
                        CourseLessonGraph.AVAILABLE_KEY,
                        settings.COURSES_DEFAULT_CLESSON_IS_AVAILABLE
                    ),
                }
            )

    @staticmethod
    def actualize_user_graph(user_id, assignment_rule_id):
        """
        Актуализирует состояние траектории конкретного пользователя в курсе.
        Метод нужен для того, чтобы во внешнем коде не надо было думать о деталях сериализации и вызывать парные методы
        deserialize + serialize
        :param user_id: id-пользователя, для которого надо пересохранить траекторию
        :param assignment_rule_id id-правила, траекторию которого актуализируем (зная правило, мы также знаем курс)
        :return: возвращает граф пользователя, который был получен при десериализации
        """

        graph = CourseLessonGraphFactory.deserialize_for_user(
            user_id=user_id,
            assignment_rule_id=assignment_rule_id,
        )
        if graph:
            CourseLessonGraphFactory.serialize_for_user(graph)

        return graph

    @staticmethod
    def deserialize(assignment_rule_id):
        """
        Принимает на вход идентификатор kelvin-правила назначения и пытается вытащить из базы все его рёбра и создать
        экземпляр класса CourseLessonGraph
        :param assignment_rule_id: идентификатор kelvin-правила назначения (зная правило, мы также знаем и курс)
        :return: эксземпляр класса CourseLessonGraph
        """
        edges = CourseLessonEdge.objects.filter(assignment_rule_id=assignment_rule_id).values_list(
            "parent_clesson_id",
            "child_clesson_id",
            "criterion_id",
        )
        nodes = CourseLessonNode.objects.filter(assignment_rule_id=assignment_rule_id).values_list(
            "clesson_id",
            "available"
        )

        return CourseLessonGraph(assignment_rule_id=assignment_rule_id, edges=edges, nodes=nodes)

    @staticmethod
    def deserialize_for_user(user_id, assignment_rule_id):
        assignment_rule_model = apps.get_model('courses', 'AssignmentRule')
        rule = assignment_rule_model.objects.filter(id=assignment_rule_id).first()

        if not rule:
            return

        # десериализуем оригинальный граф модулей курса с дефолтными стейтами модулей
        original_graph = CourseLessonGraphFactory.deserialize(assignment_rule_id)

        # обогащаем дефолтный граф состояниями вершин пользователя
        user_clesson_nodes = UserCLessonState.objects.filter(
            user_id=user_id,
            clesson__course_id=rule.course_id
        ).values_list("clesson_id", "available")

        for user_node_state in user_clesson_nodes:
            original_graph.set_clesson_available(
                clesson_id=user_node_state[0],
                available=user_node_state[1],
            )

        # аттрибутируем созданный граф пользователем для возможности защиты от неверной сериализации
        setattr(original_graph, CourseLessonGraphFactory.USER_ID_ATTR, user_id)

        return original_graph
