from builtins import object
from datetime import timedelta

from model_utils import FieldTracker

from django.db import models
from django.utils import timezone
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.common.fields import JSONField
from kelvin.common.model_mixins import AvailableForSupportMixin, TimeStampMixin, UserBlameMixin
from kelvin.lessons.models import Lesson, TextTemplate

from .progress_indicator import ProgressIndicator


class LessonScenarioMixin(models.Model):
    TRAINING_MODE = 1
    CONTROL_WORK_MODE = 2
    WEBINAR_MODE = 3
    DIAGNOSTICS_MODE = 4

    LESSON_MODES = (
        (TRAINING_MODE, _('Тренировка')),
        (CONTROL_WORK_MODE, _('Контрольная работа')),
        (WEBINAR_MODE, _('Вебинар')),
        (DIAGNOSTICS_MODE, _('Диагностика')),
    )
    EVALUATION_LESSON_MODES = (
        CONTROL_WORK_MODE,
        DIAGNOSTICS_MODE,
    )

    class VisualModes(object):
        SEPARATE = 0
        BLOCKS = 1
        CHOICES = (
            (SEPARATE, _('По отдельности')),
            (BLOCKS, _('Блоками')),
        )

    lesson = models.ForeignKey(
        Lesson,
        verbose_name=_('Занятие'),
    )
    mode = models.IntegerField(
        verbose_name=_('Режим занятия'),
        choices=LESSON_MODES,
        default=TRAINING_MODE,
    )
    duration = models.IntegerField(
        verbose_name=_('Длительность занятия (минуты)'),
        blank=True,
        null=True,
    )
    finish_date = models.DateTimeField(
        verbose_name=_('Время окончания контрольной'),
        blank=True,
        null=True,
    )
    evaluation_date = models.DateTimeField(
        verbose_name=_('Время публикации результатов контрольной'),
        blank=True,
        null=True,
    )
    show_answers_in_last_attempt = models.BooleanField(
        verbose_name=_('Показывать верные ответы в последней попытке "группы" попыток'),
        default=True,
    )
    url = JSONField(
        verbose_name=_('Адреса для вебинара'),
        blank=True,
        null=True,
    )
    start_date = models.DateTimeField(
        verbose_name=_('Дата и время начала трансляции вебинара'),
        blank=True,
        null=True,
    )
    visual_mode = models.IntegerField(
        choices=VisualModes.CHOICES,
        verbose_name=_('Отображать задачи'),
        default=VisualModes.SEPARATE,
    )
    allow_time_limited_problems = models.BooleanField(
        verbose_name=_('Разрешены задачи, ограниченные по времени'),
        blank=False,
        null=False,
        default=False,
    )

    max_attempts_in_group = models.IntegerField(
        verbose_name=_('Количество возможных попыток "в группе"'),
        default=2,
    )

    show_all_problems = models.BooleanField(
        verbose_name=_('Показывать сразу все задачи'),
        default=True,
    )

    comment = models.TextField(
        verbose_name=_('Комментарий к вебинару'),
        blank=True,
    )

    diagnostics_text = models.ForeignKey(
        TextTemplate,
        verbose_name=_('Шаблон текста для результатов диагностики'),
        blank=True,
        null=True,
    )

    def clean(self):
        """
        Библиотечный метод, валидирует модель перед сохранением.
        Эта валидация работает при сохранении моделей в админке.
        """
        self.get_validator().validate(self)
        super(LessonScenarioMixin, self).clean()

    @staticmethod
    def get_validator():
        """
        Возвращает валидатор для данной модели
        """
        # FIXME Should not survive!
        from kelvin.lessons.validators import AbstractLessonScenarioValidator

        return AbstractLessonScenarioValidator

    @property
    def can_be_evaluated(self):
        """
        Сценарий, результаты которого могуть быть опубликованы
        """
        return self.mode in self.EVALUATION_LESSON_MODES

    class Meta(object):
        abstract = True


class ProgressIndicatorMixin(models.Model):
    progress_indicator = models.ForeignKey(
        ProgressIndicator,
        verbose_name=_('Индикатор прогресса'),
        on_delete=models.SET_NULL,
        blank=True,
        null=True,
    )

    class Meta(object):
        abstract = True


class DateAssignmentMixin(models.Model):
    class Meta(object):
        abstract = True

    date_assignment = models.DateTimeField(
        verbose_name=_('Дата назначения курсозанятия'),
        default=None,
        null=True,
        blank=True,
    )
    date_completed = models.DateTimeField(
        verbose_name=_('Дата завершения курсозанятия'),
        default=None,
        null=True,
        blank=True,
    )

    @property
    def date_assignment_passed(self):
        """
        Настало ли время выдачи курсозанятия
        """
        return (self.date_assignment is not None and
                self.date_assignment < timezone.now())


@python_2_unicode_compatible
class BaseMixin(TimeStampMixin):
    lesson_block_slug = models.SlugField(
        verbose_name=_('Слаг группы занятий'),
        null=True,
        blank=True,
    )
    course = models.ForeignKey(
        'courses.Course',
        verbose_name=_('Курс'),
    )
    order = models.PositiveIntegerField(
        verbose_name=_('Сортировочный номер занятия в курсе'),
    )
    accessible_to_teacher = models.DateTimeField(
        verbose_name=_('Дата доступности занятия учителю'),
        default=None,
        null=True,
        blank=True,
    )
    lesson_editable = models.BooleanField(
        verbose_name=_('Можно редактировать занятие'),
        default=True,
    )
    copy_of = models.ForeignKey(
        'self',
        verbose_name=_('Копией какого сценария является'),
        related_name='copies',
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )
    journal = models.FileField(
        verbose_name=_('Файл с csv журналом'),
        upload_to='lesson_journal',
        null=True,
        blank=True,
    )

    @property
    def can_accept_results(self):
        """
        Принимать результаты от ученика по этому
        курсо-занятию.

        1. Нет, если недостаточно времени, чтобы пройти контрольную

        """
        # 1
        if self.finish_date and self.duration and self.can_be_evaluated:
            if (
                    timezone.now() >
                    self.finish_date - timedelta(minutes=self.duration)
            ):
                return False
        return True

    @property
    def can_view_results(self):
        """
        Настало ли время показывать результаты (для не контрольных всегда True)
        """
        return (
            self.mode not in self.EVALUATION_LESSON_MODES or self.evaluation_date < timezone.now()
        )

    def __str__(self):
        return 'Курс: {0}, Урок: {1} (id: {2})'.format(
            self.course_id, self.lesson_id, self.id)

    class Meta(object):
        abstract = True
        verbose_name = _('Сценарий занятия в курсе')
        verbose_name_plural = _('Сценарии занятий в курсах')
        ordering = ['order', 'date_created']
        unique_together = ['course', 'lesson', 'order', 'date_created']


class BaseCourseLessonLink(
    LessonScenarioMixin,
    ProgressIndicatorMixin,
    DateAssignmentMixin,
    BaseMixin,
):
    class Meta(BaseMixin.Meta):
        abstract = True


class CourseLessonLink(BaseCourseLessonLink, AvailableForSupportMixin, UserBlameMixin):
    access_code = models.CharField(
        verbose_name=_('Код занятия'),
        max_length=7,
        blank=True,
        null=True,
    )
    journal_resource = models.ForeignKey(
        'resources.Resource',
        verbose_name=_('Файл с журналом'),
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )

    ACCESS_CODE_LENGTH = 4

    tracker = FieldTracker(fields=[
        'available_for_support',
    ])

    def save(self, *args, **kwargs):
        # INTLMS-1086 - если дата выдачи уже ранее заполнялась, то не даем ее увеличивать,
        # потому что это приводит к потерям в журналах
        # TODO: src/server/express/api/defs/post-publish-clesson.js - здесь есть код фронтбека, в которм явно
        # присылается новая дата назначения. Возможно, это надо выпилить во фронтбеке для случая перевыдачи.
        # Сравнение с now нужно для того, чтобы оставить валидацию случая, когда нам прислали дату в будущем
        if self.pk and self.date_assignment and self.date_assignment + timedelta(seconds=1) < timezone.now():
            self.date_assignment = self.date_created

        if not self.order:
            self.order = (
                CourseLessonLink.objects.filter(
                    course_id=self.course_id,
                ).aggregate(
                    models.aggregates.Max('order')
                )['order__max'] or 0
            ) + 1

        super(CourseLessonLink, self).save(*args, **kwargs)
