from builtins import object

from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _


@python_2_unicode_compatible
class CoursePermission(models.Model):
    """ ROLES """
    NONE = 0
    OWNER = 1
    CONTENT_MANAGER = 2
    ANALYST = 4
    REVIEWER = 8
    CURATOR = 16

    """ ACTION """
    VIEW = "view"
    EDIT = "edit"
    STATS = "stats"
    ROLES = "roles"
    REVIEW = "review"

    ACTIONS = set([VIEW, EDIT, STATS, ROLES, REVIEW])

    '''
                      view    edit    view stats   manage roles   review
    owner              +       +         +            +             +
    content_manager    +       +         +            -             +
    analyst            +       -         +            -             -
    reviewer           +       -         -            -             +
    curator            +       -         +            -             -
    '''
    PERMISSION_MATRIX = {
        OWNER: {
            VIEW: 1,
            EDIT: 1,
            STATS: 1,
            ROLES: 1,
            REVIEW: 1,
        },
        CONTENT_MANAGER: {
            VIEW: 1,
            EDIT: 1,
            STATS: 1,
            ROLES: 0,
            REVIEW: 1,
        },
        ANALYST: {
            VIEW: 1,
            EDIT: 0,
            STATS: 1,
            ROLES: 0,
            REVIEW: 0,
        },
        REVIEWER: {
            VIEW: 1,
            EDIT: 0,
            STATS: 0,
            ROLES: 0,
            REVIEW: 1,
        },
        CURATOR: {
            VIEW: 1,
            EDIT: 0,
            STATS: 1,
            ROLES: 0,
            REVIEW: 0,
        }
    }

    course = models.ForeignKey(
        'courses.Course',
        verbose_name=_('Курс'),
    )

    user = models.ForeignKey(
        'accounts.User',
        verbose_name=_('Пользователь'),
    )

    permission = models.IntegerField(
        verbose_name=_('Уровень доступа'),
        help_text=_('Битовая маска доступных ролей')
    )

    def has_role(self, role):
        return bool(self.permission & role)

    def set_role(self, role):
        self.permission |= role

    def del_role(self, role):
        self.permission &= ~role

    @property
    def is_owner(self):
        return self.has_role(self.OWNER)

    @is_owner.setter
    def is_owner(self, value):
        if value:
            self.set_role(self.OWNER)
        else:
            self.del_role(self.OWNER)

    @property
    def is_content_manager(self):
        return self.has_role(self.CONTENT_MANAGER)

    @is_content_manager.setter
    def is_content_manager(self, value):
        if value:
            self.set_role(self.CONTENT_MANAGER)
        else:
            self.del_role(self.CONTENT_MANAGER)

    @property
    def is_analyst(self):
        return self.has_role(self.ANALYST)

    @is_analyst.setter
    def is_analyst(self, value):
        if value:
            self.set_role(self.ANALYST)
        else:
            self.del_role(self.ANALYST)

    @property
    def is_reviewer(self):
        return self.has_role(self.REVIEWER)

    @is_reviewer.setter
    def is_reviewer(self, value):
        if value:
            self.set_role(self.REVIEWER)
        else:
            self.del_role(self.REVIEWER)

    @property
    def is_curator(self):
        return self.has_role(self.CURATOR)

    @is_curator.setter
    def is_curator(self, value):
        if value:
            self.set_role(self.CURATOR)
        else:
            self.del_role(self.CURATOR)

    def can_do_action(self, action):
        if self.user.is_superuser:
            return True
        for role, role_actions in self.PERMISSION_MATRIX.items():
            if role_actions[action] and self.has_role(role):
                return True
        return False

    def can_view(self):
        return self.can_do_action(self.VIEW)

    def can_edit(self):
        return self.can_do_action(self.EDIT)

    def can_view_stats(self):
        return self.can_do_action(self.STATS)

    def can_manage_roles(self):
        return self.can_do_action(self.ROLES)

    def can_review(self):
        return self.can_do_action(self.REVIEW)

    @classmethod
    def get_default_actions_dict(self, default=0):
        return {x: default for x in self.ACTIONS}

    def get_actions_dict(self):
        """
        для данной записи CoursePermission возвращает
        словарь действий в human-readable формате
        """
        default = 1 if self.user.is_superuser else 0
        result = self.get_default_actions_dict(default=default)

        for role in list(self.PERMISSION_MATRIX.keys()):
            if self.has_role(role):
                for action_key, action_value in self.PERMISSION_MATRIX[role].items():
                    result[action_key] |= action_value

        return result

    def __str__(self):
        return "Курс: {} Пользователь: {} Роль: {}".format(self.course.id, self.user.id, self.permission)

    class Meta(object):
        verbose_name = _('Право доступа пользователя к курсу')
        verbose_name_plural = _('Права доступа пользователей к курсам')
        unique_together = ('course', 'user',)
