import logging
from builtins import object
from functools import wraps

from django_bulk_update.helper import bulk_update
from jsonschema import validate
from jsonschema.exceptions import ValidationError as JsonschemaValidationError

from django.core.exceptions import ValidationError
from django.db import transaction
from django.utils.dateparse import parse_date

from kelvin.achievery.models import ACHIEVEMENT_LEVEL_REGEXP, AchievementTeam, UserAchievement
from kelvin.achievery.signals import grant_achievement
from kelvin.courses.models import CourseLessonGraphFactory, CourseStudent, Criterion
from kelvin.idm.models import UserIDMRoleRequest

logger = logging.getLogger(__name__)
BULK_BATCH_SIZE = 1_000


def debug(func):
    @wraps(func)
    def wrapper(*args, **kwargs):
        logger.debug('{} action started'.format(args[0].TYPE))
        func(*args, **kwargs)
        logger.debug('{} action finished'.format(args[0].TYPE))

    return wrapper


class Action(object):
    TYPE = None

    def __init__(self):
        if not self.TYPE:
            raise RuntimeError('Action.TYPE class variable is required')

    def do(self, user, criterion):
        """Выполнить Action."""
        raise NotImplemented


class MyAction(Action):
    TYPE = 'MYACT'

    SCHEMA_NAME = TYPE

    SCHEMA = {
        "type": "object",
        "properties": {
            "type": {"type": "string", "enum": [SCHEMA_NAME]},
        },
        "required": ["type"],
    }

    def __init__(self, text=''):
        super(MyAction, self).__init__()
        self.text = text

    @debug
    def do(self, user, criterion):
        pass


class RequestAchievementActionBase(Action):
    def __init__(
        self,
        achievement_id,
        level,
        comment,
        is_active=True,
        request_if_exists=False,
        *args,
        **kwargs
    ):
        super().__init__()
        self.achievement_id = achievement_id
        self.level = level
        self.comment = comment
        self.request_if_exists = request_if_exists
        self.is_active = is_active

    def grant_achievements(self, users, criterion):
        with transaction.atomic():
            user_achievements_to_update = []
            users_update_ids = set()
            existing_user_achievements = (
                UserAchievement.objects
                .select_for_update()
                .filter(user__in=users, achievement_id=self.achievement_id, level=self.level)
            )
            for user_achievement in existing_user_achievements:
                user_achievement.is_active = self.is_active
                user_achievement.comment = self.comment
                user_achievement.criterion = criterion
                user_achievement.request_if_exists = self.request_if_exists

                user_achievements_to_update.append(user_achievement)
                users_update_ids.add(user_achievement.user_id)

            user_achievements_to_create = [
                UserAchievement(
                    user=user,
                    achievement_id=self.achievement_id,
                    is_active=self.is_active,
                    comment=self.comment,
                    criterion=criterion,
                    request_if_exists=self.request_if_exists,
                    level=self.level,
                )
                for user in users if user.id not in users_update_ids
            ]

            bulk_update(
                objs=user_achievements_to_update,
                batch_size=BULK_BATCH_SIZE,
            )

            UserAchievement.objects.bulk_create(user_achievements_to_create)

        # Вызываем пост-обработку при сохранении вручную, так как bulk-команды не генерируют сигналы
        for user_achievement in user_achievements_to_create + user_achievements_to_update:
            grant_achievement(user_achievement=user_achievement)


class RequestAchievementAction(RequestAchievementActionBase):
    TYPE = 'REQUEST_ACHIEVEMENT_ACTION'

    SCHEMA_NAME = TYPE

    SCHEMA = {
        "type": "object",
        "properties": {
            "type": {"type": "string", "enum": [SCHEMA_NAME]},
            "achievement_id": {"type": "integer", "minimum": 1},
            "level": {"type": "string", "pattern": ACHIEVEMENT_LEVEL_REGEXP},
            "max_level": {"type": "integer"},
            "min_level": {"type": "integer"},
            "comment": {"type": "string", "maxLength": 255},
            "is_active": {"type": "boolean"},
            "request_if_exists": {"type": "boolean"},
        },
        "required": ["type", "achievement_id", "level", "comment"],
    }

    @debug
    def do(self, user, criterion):
        self.grant_achievements(users=[user], criterion=criterion)


class RequestTeamAchievementAction(RequestAchievementActionBase):
    TYPE = 'REQUEST_TEAM_ACHIEVEMENT_ACTION'

    SCHEMA_NAME = TYPE

    SCHEMA = {
        "type": "object",
        "properties": {
            "type": {"type": "string", "enum": [SCHEMA_NAME]},
            "achievement_id": {"type": "integer", "minimum": 1},
            "clesson_id": {"type": "integer", "minimum": 1},
            "level": {"type": "string", "pattern": ACHIEVEMENT_LEVEL_REGEXP},
            "comment": {"type": "string", "maxLength": 255},
            "is_active": {"type": "boolean"},
            "request_if_exists": {"type": "boolean"},
        },
        "required": ["type", "achievement_id", "level", "comment", "clesson_id"],
    }

    def __init__(
        self,
        clesson_id,
        *args,
        **kwargs
    ):
        self.clesson_id = clesson_id
        super().__init__(*args, **kwargs)

    @debug
    def do(self, user, criterion):
        achievement_team = (
            AchievementTeam.objects
            .filter(captain=user, clesson_id=self.clesson_id, status=AchievementTeam.STATUS_READY)
            .first()
        )
        if achievement_team is None:
            users = [user]
        else:
            users = list(achievement_team.participants.all()) + [user]
        self.grant_achievements(users=users, criterion=criterion)


class RequestIDMRoleAction(Action):
    TYPE = 'REQUEST_IDM_ROLE_ACTION'

    SCHEMA_NAME = TYPE

    SCHEMA = {
        "type": "object",
        "properties": {
            "type": {"type": "string", "enum": [SCHEMA_NAME]},
            "system": {"type": "string", "maxLength": 255},
            "path": {"type": "string", "maxLength": 1024},
            "comment": {"type": "string", "maxLength": 1024},
            "simulate": {"type": "boolean"},
            "fields_data": {"type": "object"},
            "deprive_at": {"type": "string", "format": "date-time"},
            "review_at": {"type": "string", "format": "date-time"},
            "deprive_after_days": {"type": "integer"},
            "silent": {"type": "boolean"},
        },
        "required": ["type", "system", "path", "comment"],
    }

    def __init__(
        self,
        system,
        path,
        comment,
        simulate=False,
        fields_data=None,
        deprive_at=None,
        review_at=None,
        deprive_after_days=None,
        silent=False,
    ):
        super(RequestIDMRoleAction, self).__init__()
        self.system = system
        self.path = path
        self.comment = comment
        self.simulate = simulate
        self.fields_data = fields_data or {}
        self.deprive_at = deprive_at
        self.review_at = review_at
        self.deprive_after_days = deprive_after_days
        self.silent = silent

    @debug
    def do(self, user, criterion):
        obj_kwargs = {
            'user': user,
            'system': self.system,
            'path': self.path,
            'fields_data': self.fields_data,
        }
        with transaction.atomic():
            role_request = UserIDMRoleRequest.objects.select_for_update().filter(**obj_kwargs).first()

            if role_request is None:
                role_request = UserIDMRoleRequest(**obj_kwargs)
            elif role_request.requested_at is not None:
                return

            role_request.comment = self.comment
            role_request.deprive_at = self.deprive_at
            role_request.review_at = self.review_at
            role_request.deprive_after_days = self.deprive_after_days
            role_request.criterion = criterion

            role_request.save()  # чтобы сработал сигнал


class CLessonCompletionAction(Action):
    """
    Action для обработки логики завершения курсо-зянятия ( модуля ).
    По завершению хотим сделать следующее:
     - пытаемся сделать пользователю (user) доступными для прохождения модули, зависящие от модуля,
       критерий (criterion) которого сработал, в результате чего был вызван текущий Action.
     - сам модуль пытаемся достать из criterion
    """
    TYPE = 'CLESSON_COMPLETION'

    SCHEMA_NAME = TYPE

    SCHEMA = {
        "type": "object",
        "properties": {
            "type": {"type": "string", "enum": [SCHEMA_NAME]},
        },
        "required": ["type"],
    }

    __already_visited_parents_cache = {}

    def __all_parents_are_available(self, graph, sub_clesson_id):
        """ проверяет, что все родители переданного модуля - открыты """
        all_available = True
        parent_clessons = graph.get_parent_clessons(clesson_id=sub_clesson_id)

        for parent_clesson_id in parent_clessons:
            parent_clesson_available = self.__already_visited_parents_cache.get(parent_clesson_id, None)
            if parent_clesson_available is None:
                self.__already_visited_parents_cache[parent_clesson_id] = parent_clesson_available = \
                    graph.get_clesson_available(clesson_id=parent_clesson_id)

            if not parent_clesson_available:
                # не можем пока открыть модуль, потому что есть неоткрытые родительские модули
                all_available = False
                logger.info("Cannot open clesson {} for now because parent clesson {} is not available yet".format(
                    sub_clesson_id,
                    parent_clesson_id,
                ))
                graph.set_clesson_available(clesson_id=sub_clesson_id, available=False)

        return all_available

    def _open_submodules(self, user_id, assignment_rule_id, clesson_id, criterion_id):
        # пытаемся десериализовать граф
        graph = CourseLessonGraphFactory.deserialize_for_user(user_id=user_id, assignment_rule_id=assignment_rule_id)

        # проверяем, есть ли в графе модулей курса вообще модуль, с которым мы работаем
        if not graph.has_node(clesson_id=clesson_id):
            raise RuntimeError("Course graph misconfiguraion: no such clesson {} for assignment_rule {} graph".format(
                clesson_id,
                assignment_rule_id,
            ))

        sub_clessons = graph.get_sub_clessons(clesson_id=clesson_id, criterion_id=criterion_id)

        self.__already_visited_parents_cache = {}

        for sub_clesson_id in sub_clessons:
            all_available = self.__all_parents_are_available(graph, sub_clesson_id)
            if all_available:
                # если все родители зависимого модуля открыты, то помечаем его открытым
                graph.set_clesson_available(clesson_id=sub_clesson_id, available=True)
                logger.info("CLesson {} is available".format(sub_clesson_id))

            CourseLessonGraphFactory.serialize_for_user(graph)

    @debug
    def do(self, user, criterion):
        clesson_id = criterion.clesson_id
        if not clesson_id:
            # ситуация не нормальная, потому что Action предназначен именно для модулей
            # и обязательно должен быть связан с модулем
            raise RuntimeError("Cannot call clesson criterion action without clesson linked")

        # открываем подмодули
        self._open_submodules(
            user_id=user.id,
            assignment_rule_id=criterion.assignment_rule_id,
            clesson_id=clesson_id,
            criterion_id=criterion.id
        )


class CourseCompletionAction(Action):
    """
    Action для обработки завершения курса
    По завершению мы ставим в CourseStudent признак completed=True
    """
    TYPE = 'COURSE_COMPLETION'
    SCHEMA_NAME = TYPE
    SCHEMA = {
        "type": "object",
        "properties": {
            "type": {"type": "string", "enum": [SCHEMA_NAME]},
            "extra_courses": {"type": "array", "items": {"type": "number"}},
            "student_from_date": {"type": "string", "format": "date"},
            "student_until_date": {"type": "string", "format": "date"},
        },
        "required": ["type"],
    }

    def __init__(self, extra_courses=None, student_from_date=None, student_until_date=None, *args, **kwargs):
        super().__init__()
        self.extra_courses = extra_courses or []
        self.student_from_date = parse_date(student_from_date or "")
        self.student_until_date = parse_date(student_until_date or "")

    def complete_courses(self, user, courses) -> None:
        queryset = CourseStudent.objects.filter(student=user, course_id__in=courses, completed=False)

        if self.student_from_date:
            queryset = queryset.filter(date_created__gte=self.student_from_date)

        if self.student_until_date:
            queryset = queryset.filter(date_created__lt=self.student_until_date)

        for student in queryset:
            if not student.completed:
                student.completed = True
                student.save()

    @debug
    def do(self, user, criterion: "Criterion"):
        assignment_rule = getattr(criterion, 'assignment_rule')
        if not assignment_rule:
            raise RuntimeError("Cannot call criterion action without assignment rule linked")
        course_id = assignment_rule.course_id

        with transaction.atomic():
            courses = [course_id] + self.extra_courses
            self.complete_courses(user, courses)


ACTIONS_AVAILABLE = [
    MyAction,
    CLessonCompletionAction,
    CourseCompletionAction,
    RequestAchievementAction,
    RequestTeamAchievementAction,
    RequestIDMRoleAction,
]


class ActionAdapter(object):
    def __init__(self):
        pass

    @classmethod
    def get_action_by_type(cls, action_type):
        for action in ACTIONS_AVAILABLE:
            if action_type == action.TYPE:
                return action

        raise Exception(f"Action {action_type} not found")


def collect_actions_schema():
    return {
        "type": "array",
        "items": {
            "oneOf": [
                action_schema.SCHEMA for action_schema in ACTIONS_AVAILABLE
            ],
        },
    }


ACTION_SCHEMA = collect_actions_schema()


def validate_actions(actions):
    try:
        validate(actions, ACTION_SCHEMA)
    except JsonschemaValidationError as exc:
        raise ValidationError(str(exc))
