import operator
from builtins import map, object
from functools import reduce

from attr import attrib, attrs, validators
from jsonschema import validate
from jsonschema.exceptions import ValidationError as JsonschemaValidationError

from django.core.exceptions import ValidationError
from django.db.models import Count, IntegerField, OuterRef, Subquery, Sum, Value
from django.db.models.functions import Cast, Coalesce

from kelvin.courses.models import CourseLessonLink
from kelvin.lessons.models import LessonProblemLink
from kelvin.results.models import CourseLessonResult


@attrs
class Condition(object):
    """Условие - часть формулы критерия."""
    TYPE = None
    VISIBLE = False

    COMPARE_MAP = {
        '>': lambda a, b: a > b,
        '>=': lambda a, b: a >= b,
        '<=': lambda a, b: a <= b,
        '==': lambda a, b: a == b,
        '!=': lambda a, b: a != b,
        '<': lambda a, b: a < b,
        '<=': lambda a, b: a <= b,
    }

    AGGREGATE_MAP = {
        'all': (all, True),
        'any': (any, True),
        'sum': (sum, False),
        'mul': (lambda iterable: reduce(operator.mul, iterable, 1), False),
        'max': (max, False),
        'min': (min, False),
    }

    SCHEMA_NAME = "PREDICATE_BASE"

    SCHEMA = {
        "type": "object",
        "properties": {
            "type": {"type": "string"},
            "aggregator": {
                "type": "string",
                "enum": list(AGGREGATE_MAP.keys()),
            },
            "comparator": {
                "type": "string",
                "enum": list(COMPARE_MAP.keys()),
            },
            "value": {"type": "number"},
        },
        "required": ["type", "aggregator", "comparator", "value"],
    }

    value = attrib(default='')
    comparator = attrib(default='', validator=validators.in_(COMPARE_MAP))
    aggregator = attrib(default='', validator=validators.in_(AGGREGATE_MAP))

    @comparator.validator
    def check(self, attribute, comparator_value):
        """Валидатор для компаратора."""
        if comparator_value not in self.COMPARE_MAP:
            raise RuntimeError(
                f'comparator "{comparator_value}" must be in {list(self.COMPARE_MAP.keys())}'
            )

    @aggregator.validator
    def check(self, attribute, aggregator_value):
        """Валидатор для агрегатора."""
        if aggregator_value not in self.AGGREGATE_MAP:
            raise RuntimeError(
                f'aggregator_value "{aggregator_value}" must be in {list(self.AGGREGATE_MAP.keys())}'
            )

    def get_source_values(self, user, criterion):
        raise NotImplemented

    @staticmethod
    def cast(value):
        """Привести значение к формату, который принимают на вход операции."""
        return value

    @classmethod
    def get_compare_func(cls, comparator):
        return cls.COMPARE_MAP[comparator]

    @classmethod
    def get_aggregate_func(cls, aggregator):
        return cls.AGGREGATE_MAP[aggregator]

    def eval(self, criterion, user):
        """Вычисление условия."""
        self.validate(criterion)
        source_values = self.get_source_values(user, criterion)
        compare_func = self.get_compare_func(self.comparator)
        aggregate_func, aggregate_first = self.get_aggregate_func(self.aggregator)

        if aggregate_first:
            return aggregate_func(
                compare_func(source_value, self.cast(self.value))
                for source_value in map(self.cast, source_values)
            )

        return compare_func(
            aggregate_func(map(self.cast, source_values)),
            self.cast(self.value)
        )

    def validate(self, criterion):
        if not (criterion.assignment_rule and criterion.assignment_rule.course_id):
            raise RuntimeError('Invalid condition type for the criterion')


@attrs
class ConditionCourse(Condition):
    """Условие на курс."""
    TYPE = 'COURSE_BASE'

    SCHEMA_NAME = f"PREDICATE_{TYPE}"

    SCHEMA = {
        "allOf": [
            {"$ref": f"#/definitions/{Condition.SCHEMA_NAME}"},
        ],
    }


@attrs
class ConditionClesson(Condition):
    """Условие на модуль/урок."""
    TYPE = 'COURSE_LESSON_BASE'

    SCHEMA_NAME = f"PREDICATE_{TYPE}"

    SCHEMA = {
        "allOf": [
            {"$ref": f"#/definitions/{Condition.SCHEMA_NAME}"},
            {
                "properties": {
                    "clesson_id": {
                        "type": "integer",
                        "minimum": 1,
                    }
                },
                "required": ["clesson_id"],
            },
        ],
    }

    clesson_id = attrib(default='')


class ConditionClessonPoints(ConditionClesson):
    """Условие на модуль, проверяющее количество баллов."""
    TYPE = 'ClessonPoints'

    SCHEMA_NAME = f"PREDICATE_{TYPE}"

    SCHEMA = {
        "allOf": [
            {"$ref": f"#/definitions/{ConditionClesson.SCHEMA_NAME}"},
            {
                "properties": {
                    "type": {"type": "string", "enum": [TYPE]},
                    "value": {
                        "type": "number",
                        "minimum": 0,
                        "maximum": 100,
                    },
                },
            },
        ],
    }

    def get_source_values(self, user, criterion):
        latest_clesson_result = CourseLessonResult.objects.filter(
            summary__student_id=user.id,
            summary__clesson_id=self.clesson_id,
        ).order_by('-id').only('points').first()

        if latest_clesson_result:
            return [latest_clesson_result.points, ]

        return [0]


class ConditionClessonPointsPercentage(ConditionClesson):
    TYPE = 'ClessonPointsPercentage'

    SCHEMA_NAME = f"PREDICATE_{TYPE}"

    SCHEMA = {
        "allOf": [
            {"$ref": f"#/definitions/{ConditionClesson.SCHEMA_NAME}"},
            {
                "properties": {
                    "type": {"type": "string", "enum": [TYPE]},
                    "value": {
                        "type": "number",
                        "minimum": 0,
                        "maximum": 100,
                    },
                },
            },
        ],
    }

    def get_source_values(self, user, criterion):
        latest_clesson_result = CourseLessonResult.objects.filter(
            summary__student_id=user.id,
            summary__clesson_id=self.clesson_id,
        ).order_by('-id').only('points', 'max_points').first()

        if latest_clesson_result:
            return [calculate_percentage(latest_clesson_result.points, latest_clesson_result.max_points)]

        return [0]


class ConditionClessonProgressPercentage(ConditionClesson):
    TYPE = 'ClessonProgressPercentage'

    SCHEMA_NAME = f"PREDICATE_{TYPE}"

    SCHEMA = {
        "allOf": [
            {"$ref": f"#/definitions/{ConditionClesson.SCHEMA_NAME}"},
            {
                "properties": {
                    "type": {"type": "string", "enum": [TYPE]},
                    "value": {
                        "type": "number",
                        "minimum": 0,
                        "maximum": 100,
                    },
                },
            },
        ],
    }

    @staticmethod
    def get_progress_max_points(clesson_result):
        lesson_problem_links = LessonProblemLink.objects.filter(lesson_id=clesson_result.summary.clesson.lesson_id)
        return lesson_problem_links.count()

    @staticmethod
    def get_progress_points(clesson_result):
        answers = clesson_result.answers or {}
        return len(answers)

    def get_source_values(self, user, criterion):
        latest_clesson_result = CourseLessonResult.objects.filter(
            summary__student_id=user.id,
            summary__clesson_id=self.clesson_id,
        ).select_related(
            'summary', 'summary__clesson',
        ).only(
            'id', 'answers', 'summary__clesson__lesson_id',
        ).order_by('-id').first()

        if latest_clesson_result:
            return [
                calculate_percentage(
                    self.get_progress_points(latest_clesson_result),
                    self.get_progress_max_points(latest_clesson_result),
                ),
            ]

        return [0]


class ConditionCoursePoints(ConditionCourse):
    """Условие на курс, проверяющее количество баллов."""
    TYPE = 'CoursePoints'

    SCHEMA_NAME = f"PREDICATE_{TYPE}"

    SCHEMA = {
        "allOf": [
            {"$ref": f"#/definitions/{ConditionCourse.SCHEMA_NAME}"},
            {
                "properties": {
                    "type": {"type": "string", "enum": [TYPE]},
                    "value": {
                        "type": "number",
                        "minimum": 0,
                        "maximum": 100,
                    },
                },
            },
        ],
    }

    def get_source_values(self, user, criterion):
        latest_clesson_points = CourseLessonResult.objects.filter(
            summary__student_id=user.id,
            summary__clesson=OuterRef("id"),
        ).order_by('-id').values_list('points')[:1]

        latest_results = CourseLessonLink.objects.filter(
            course_id=criterion.assignment_rule.course_id
        ).annotate(
            points=Coalesce(
                Cast(Subquery(latest_clesson_points), IntegerField()),
                Value(0),
            )
        ).values_list("points", flat=True)

        return latest_results


class ConditionCoursePointsPercentage(ConditionCourse):
    TYPE = 'CoursePointsPercentage'

    SCHEMA_NAME = f"PREDICATE_{TYPE}"

    SCHEMA = {
        "allOf": [
            {"$ref": f"#/definitions/{ConditionCourse.SCHEMA_NAME}"},
            {
                "properties": {
                    "type": {"type": "string", "enum": [TYPE]},
                    "value": {
                        "type": "number",
                        "minimum": 0,
                        "maximum": 100,
                    },
                },
            },
        ],
    }

    def get_source_values(self, user, criterion):
        latest_clesson_points = CourseLessonResult.objects.filter(
            summary__student_id=user.id,
            summary__clesson=OuterRef("id"),
        ).order_by('-id').values_list('points')[:1]

        latest_clesson_max_points = CourseLessonResult.objects.filter(
            summary__student_id=user.id,
            summary__clesson=OuterRef("id"),
        ).order_by('-id').values_list('max_points')[:1]

        latest_results = CourseLessonLink.objects.filter(
            course_id=criterion.assignment_rule.course_id
        ).annotate(
            points=Coalesce(
                Cast(Subquery(latest_clesson_points), IntegerField()),
                Value(0),
            ),
            max_points=Coalesce(
                Cast(Subquery(latest_clesson_max_points), IntegerField()),
                Cast(Sum("lesson__problems__max_points"), IntegerField()),
                Value(0),
            ),
        ).aggregate(
            sum_points=Sum("points"),
            sum_max_points=Sum("max_points"),
        )

        return [
            calculate_percentage(
                points=latest_results["sum_points"],
                max_points=latest_results["sum_max_points"],
            ),
        ]


class ConditionCourseProgressPercentage(ConditionCourse):
    TYPE = 'CourseProgressPercentage'

    SCHEMA_NAME = f"PREDICATE_{TYPE}"

    SCHEMA = {
        "allOf": [
            {"$ref": f"#/definitions/{ConditionCourse.SCHEMA_NAME}"},
            {
                "properties": {
                    "type": {"type": "string", "enum": [TYPE]},
                    "value": {
                        "type": "number",
                        "minimum": 0,
                        "maximum": 100,
                    },
                },
            },
        ],
    }

    @staticmethod
    def get_progress_points(clesson_result_answers):
        answers = clesson_result_answers or {}
        return len(answers)

    def get_source_values(self, user, criterion):
        latest_clesson_answers = CourseLessonResult.objects.filter(
            summary__student_id=user.id,
            summary__clesson=OuterRef("id"),
        ).order_by('-id').values_list('answers')[:1]

        latest_results = CourseLessonLink.objects.filter(
            course_id=criterion.assignment_rule.course_id
        ).annotate(
            answers=Subquery(latest_clesson_answers),
            count_problems=Count("lesson__problems"),
        )

        sum_answers = 0
        sum_problems = 0
        for latest_result in latest_results:
            sum_answers += self.get_progress_points(latest_result.answers)
            sum_problems += latest_result.count_problems

        return [
            calculate_percentage(
                sum_answers,
                sum_problems,
            )
        ]


CONDITIONS_AVAILABLE = [
    ConditionClessonPoints,
    ConditionClessonPointsPercentage,
    ConditionClessonProgressPercentage,
    ConditionCoursePoints,
    ConditionCoursePointsPercentage,
    ConditionCourseProgressPercentage,
]


class ConditionAdapter(object):
    def __init__(self):
        pass

    @classmethod
    def get_condition_by_type(cls, condition_type):
        for condition in CONDITIONS_AVAILABLE:
            if condition_type == condition.TYPE:
                return condition

        raise Exception(f'Condition {condition_type} not found')


def calculate_percentage(points, max_points):
    if not points or not max_points:
        return 0
    return int(points * 100 / max_points)


def collect_formula_schema():
    conditions_classes = set()
    for condition_avaliable in CONDITIONS_AVAILABLE:
        conditions_classes |= {
            base_condition
            for base_condition in condition_avaliable.__mro__
            if issubclass(base_condition, Condition)
        }

    conditions_schemas = {
        condition_class.SCHEMA_NAME: condition_class.SCHEMA
        for condition_class in conditions_classes
    }

    predicate = {
        "oneOf": [
            {"$ref": f"#/definitions/{condition_class.SCHEMA_NAME}"}
            for condition_class in CONDITIONS_AVAILABLE
        ],
    }

    return {
        "definitions": {
            **conditions_schemas,
            **{
                "PREDICATE": predicate,
                "CONDITION_ITEM": {
                    "oneOf": [
                        {"$ref": "#/definitions/PREDICATE"},
                        {"$ref": "#/definitions/AND_OR"},
                    ]
                },
                "CONDITION_ARRAY": {
                    "type": "array",
                    "minItems": 1,
                    "items": {
                        "$ref": "#/definitions/CONDITION_ITEM",
                    },
                },
                "AND_OR": {
                    "type": "object",
                    "properties": {
                        "AND": {
                            "$ref": "#/definitions/CONDITION_ARRAY",
                        },
                        "OR": {
                            "$ref": "#/definitions/CONDITION_ARRAY",
                        },
                    },
                    "oneOf": [
                        {"required": ["AND"]},
                        {"required": ["OR"]},
                    ],
                },
            },
        },
        "$ref": "#/definitions/CONDITION_ITEM"
    }


FORMULA_SCHEMA = collect_formula_schema()


def validate_formula(formula):
    try:
        validate(formula, FORMULA_SCHEMA)
    except JsonschemaValidationError as exc:
        raise ValidationError(str(exc))
