from builtins import object, str

from django.conf import settings
from django.contrib.auth import get_user_model
from django.db import models
from django.utils.translation import ugettext_lazy as _

from kelvin.common.fields import JSONField

User = get_user_model()


class Criterion(models.Model):
    """Критерий."""
    clesson = models.ForeignKey(
        to='courses.CourseLessonLink', verbose_name=_('Курсозанятие'), null=True, blank=True
    )  # deprecated (clesson_id is in formula)
    name = models.CharField(verbose_name=_('Название'), max_length=255, default='')
    priority = models.PositiveSmallIntegerField(verbose_name=_('Приоритет'), default=0)
    formula = JSONField(verbose_name=_('Формула условий'), default=[])
    actions = JSONField(verbose_name=_('Действия'), default=[])
    assignment_rule = models.ForeignKey(to='courses.AssignmentRule', verbose_name=_('Правило назначения'),
                                        null=True, blank=True)

    class Meta(object):
        verbose_name = _('Критерий')
        verbose_name_plural = _('Критерии')

    def __str__(self):
        return self.name if self.name else str(self.id)

    def save(self, *args, **kwargs):
        if not self.name:
            assignment_rule_name = str(self.assignment_rule) if self.assignment_rule else ''
            action_names = {
                action.get('type', '') for action in (self.actions or [])
            }
            self.name = f'{assignment_rule_name}, actions: {action_names}'

        return super().save(*args, **kwargs)

    def _evaluate_formula_item(self, formula_item, user):
        if "OR" in formula_item:
            return any(self._evaluate_formula_item(or_item, user) for or_item in formula_item["OR"])
        if "AND" in formula_item:
            return all(self._evaluate_formula_item(or_item, user) for or_item in formula_item["AND"])

        return condition_factory(**formula_item).eval(criterion=self, user=user)

    def evaluate(self, user):
        """
        Вычисление формулы критерия для пользователя.

        Формула удовлетворяет схеме FORMULA_SCHEMA"
        """
        return self._evaluate_formula_item(formula_item=self.formula, user=user)

    def do_actions(self, user: User, force: bool = False):
        user_criterion = UserCriterion.objects.filter(user=user, criterion=self).first()
        if user_criterion and user_criterion.done and not force:
            return

        if self.evaluate(user):
            for action_kwargs in self.actions:
                action = action_factory(**action_kwargs)
                action.do(user=user, criterion=self)

            UserCriterion.objects.update_or_create(user=user, criterion=self, defaults={'done': True})


class UserCriterion(models.Model):
    criterion = models.ForeignKey(to=Criterion, verbose_name=_('Критерий'))
    user = models.ForeignKey(to=settings.AUTH_USER_MODEL, verbose_name=_('Пользователь'))
    done = models.BooleanField(verbose_name=_('Действия выполнены'), default=False)

    class Meta(object):
        verbose_name = _('Критерий пользователя')
        verbose_name_plural = _('Критерии пользователей')

    def __str__(self):
        return '{}: {} {}'.format(self.id, self.criterion, self.user)


def condition_factory(**kwargs):
    """Фабрика условий."""
    from .condition import ConditionAdapter

    condition_type = kwargs.pop('type', '')
    condition_class = ConditionAdapter.get_condition_by_type(condition_type)
    condition = condition_class(**kwargs)

    return condition


def action_factory(**kwargs):
    """Фабрика действий."""
    from .action import ActionAdapter

    action_type = kwargs.pop('type', '')
    action_class = ActionAdapter.get_action_by_type(action_type)
    action = action_class(**kwargs)

    return action
