from simple_history.models import HistoricalRecords

from django.contrib.auth import get_user_model
from django.core.validators import MaxValueValidator, MinValueValidator
from django.db import models
from django.db.models import Q
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _

from kelvin.common.fields import JSONField

from ...common.model_mixins import TimeStampMixin
from .course import Course, CourseStudent

User = get_user_model()


class PeriodicCourse(TimeStampMixin):
    """
    Периодические курсы

    """
    course = models.OneToOneField(
        Course,
        verbose_name=_("Текущий курс"),
        related_name="periodic",
        on_delete=models.PROTECT,
    )

    previous = models.ManyToManyField(
        Course,
        verbose_name=_("прошедшие курсы"),
        blank=True,
    )

    period = models.PositiveSmallIntegerField(
        verbose_name=_("период"),
        help_text=_("период зачисления на курс, в днях"),
        default=365,
        validators=[
            MinValueValidator(1),
        ],
    )

    class Meta:
        verbose_name = _("периодический курс")
        verbose_name_plural = _("периодические курсы")

    def __str__(self):
        return "Курс: {}| Период: {}".format(self.course_id, self.period)


class PeriodicNotification(TimeStampMixin):
    periodic_course = models.ForeignKey(
        PeriodicCourse,
        verbose_name=_("курс"),
        related_name="notifications",
        on_delete=models.CASCADE,
    )

    NOTIFY_TYPE_EMAIL = 'email'
    NOTIFY_TYPE_TRACKER = 'tracker'

    NOTIFY_TYPE_CHOICES = (
        (NOTIFY_TYPE_EMAIL, _("Почта")),
        (NOTIFY_TYPE_TRACKER, _("Тикет в Трекере")),
    )

    notify_type = models.CharField(
        _("тип уведомления"),
        max_length=20,
        choices=NOTIFY_TYPE_CHOICES,
        default=NOTIFY_TYPE_EMAIL,
    )

    delay = models.PositiveSmallIntegerField(
        _("кол-во дней до уведомления"),
        help_text=_("период в днях между зачислением на курс и отправкой уведомления"),
        default=0,
    )

    priority = models.PositiveSmallIntegerField(
        _("приоритет"),
        help_text=_("уведомления будут отправляться с учетом приоритета, от большего к меньшему"),
        default=0,
    )

    parameters = JSONField(
        _("параметры"),
        default=dict,
    )

    class Meta:
        ordering = ("periodic_course", "delay")
        verbose_name = _("настройка уведомлений")
        verbose_name_plural = _("настройки уведомлений")

    def __str__(self):
        return "[{}] {}: {}d".format(
            self.periodic_course_id,
            self.get_notify_type_display(),
            self.delay,
        )


class PeriodicStudentNotification(TimeStampMixin):
    notification = models.ForeignKey(
        PeriodicNotification,
        verbose_name=_("уведомление"),
        related_name="notified",
        on_delete=models.PROTECT,
    )

    student = models.ForeignKey(
        CourseStudent,
        verbose_name=_("студент"),
        related_name="periodic_notifications",
        on_delete=models.CASCADE,
    )

    course = models.ForeignKey(
        Course,
        verbose_name=_("курс"),
        related_name="periodic_student_notifications",
        blank=True,
        null=True,
        on_delete=models.CASCADE,
    )

    STATUS_PENDING = 'pending'
    STATUS_SENT = 'sent'
    STATUS_ERROR = 'error'

    STATUS_CHOICES = (
        (STATUS_PENDING, _("Ожидает")),
        (STATUS_SENT, _("Отправлено")),
        (STATUS_ERROR, _("Ошибка")),
    )
    status = models.CharField(_("статус"), max_length=20, choices=STATUS_CHOICES, default=STATUS_PENDING)
    errors = models.TextField(_("сообщения об ошибках"), blank=True)
    result_data = JSONField(
        _("результат"),
        default=dict,
        null=True,
        blank=True,
    )

    class Meta:
        verbose_name = _("отправленное уведомление по периодическому курсу")
        verbose_name_plural = _("отправленные уведомления по периодическим курсам")

    def __str__(self):
        return str(self.pk)


class ExcludedUser(TimeStampMixin):
    login = models.CharField(_("логин"), max_length=255)
    exclude_from_courses = models.BooleanField(_("не зачислять в курсы"), default=True)
    exclude_from_issues = models.BooleanField(_("не создавать тикеты"), default=True)

    class Meta:
        verbose_name = _("пользователь, исключенный из периодических курсов")
        verbose_name_plural = _("пользователи, исключенные из периодических курсов")

    def __str__(self):
        return str(self.login)


class NotifyDayOffManager(models.Manager):
    def allowed(self, current=None) -> bool:
        """
        проверяет доступна ли дата
        """
        current = current or timezone.now()
        queryset = self.get_queryset().filter(
            day=current.day,
            month=current.month,
            is_active=True,
        ).filter(
            Q(valid_from__lte=current, valid_to__isnull=True) |
            Q(valid_from__lte=current, valid_to__gt=current)
        )
        return not queryset.exists()


class NotifyDayOff(TimeStampMixin):
    name = models.CharField(_("название"), max_length=255, blank=True)
    comments = models.TextField(_("комментарии"), blank=True)
    day = models.PositiveSmallIntegerField(_("день"), validators=[MinValueValidator(1), MaxValueValidator(31)])
    month = models.PositiveSmallIntegerField(_("месяц"), validators=[MinValueValidator(1), MaxValueValidator(12)])
    is_active = models.BooleanField(_("активен"), default=True)
    valid_from = models.DateTimeField(_("действует с"), default=timezone.now)
    valid_to = models.DateTimeField(_("действует до"), blank=True, null=True)

    objects = NotifyDayOffManager()

    class Meta:
        verbose_name = _("исключение для уведомлений")
        verbose_name_plural = _("исключения для уведомлений")

    def __str__(self):
        now_year = timezone.now().year
        label = "{}-{:02d}-{:02d}".format(now_year, self.month, self.day)
        return label if not self.name else "[{}] {}".format(label, self.name)


class PeriodicRoleDigest(TimeStampMixin):
    ROLE_CHIEF = 'chief'
    ROLE_HRBP = 'hrbp'

    ROLE_CHOICES = (
        (ROLE_CHIEF, _("руководитель")),
        (ROLE_HRBP, _("hr-партнер")),
    )

    periodic_course = models.ForeignKey(
        PeriodicCourse,
        verbose_name=_("периодический курс"),
        on_delete=models.CASCADE,
        related_name='role_digests',
    )
    first_run = models.DateField(
        verbose_name=_("дата первого запуска дайджеста"),
        help_text=_("дата, с которой запускаются проверки на формирование дайджеста"),
        db_index=True,
    )
    next_run = models.DateField(
        verbose_name=_("дата следующего запуска дайджеста"),
        db_index=True,
    )
    is_active = models.BooleanField(
        verbose_name=_("активен"),
        default=True,
    )
    interval = models.PositiveSmallIntegerField(
        verbose_name=_("минимальный интервал между дайджестами"),
        help_text=_("дайджесты не формируются чаще, чем этот интервал"),
        validators=[MinValueValidator(1)],
    )
    role = models.CharField(
        max_length=10,
        verbose_name=_("тип дайджеста"),
        choices=ROLE_CHOICES,
        db_index=True,
    )
    delay = models.PositiveSmallIntegerField(
        verbose_name=_("период до попадания в дайджест"),
        help_text=_("период (в днях) между датой назначения курса и попаданием в дайджест руководителю"),
        validators=[MinValueValidator(1)],
    )
    can_reopen_issue = models.BooleanField(
        verbose_name=_("переоткрывать тикет для нового дайджеста"),
        help_text=_("для нового дайджеста переоткрывается старый тикет (иначе создается новый)"),
        default=False,
    )
    parameters = JSONField(
        _("параметры"),
        default=dict,
    )

    class Meta:
        verbose_name=_("периодический дайджест")
        verbose_name_plural = _("периодические дайджесты")

    def __str__(self):
        return _("Периодический курс: {}| Роль: {}").format(self.periodic_course_id, self.role)

    def save(self, *args, **kwargs):
        if self.next_run is None:
            self.next_run = self.first_run
        return super().save(*args, **kwargs)


class RoleDigest(TimeStampMixin):
    PROCESS_STATUS_PENDING = 'pending'
    PROCESS_STATUS_DONE = 'done'
    PROCESS_STATUS_ERROR = 'error'

    PROCESS_STATUS_CHOICES = (
        (PROCESS_STATUS_PENDING, _("обрабатывается")),
        (PROCESS_STATUS_DONE, _("создан")),
        (PROCESS_STATUS_ERROR, _("ошибка")),
    )

    periodic_role_digest = models.ForeignKey(
        PeriodicRoleDigest,
        verbose_name=_("периодический дайджест"),
        on_delete=models.CASCADE,
        related_name='digests',
    )
    user = models.ForeignKey(
        User,
        verbose_name=_("пользователь"),
        help_text=_("пользователь, для которого создается дайджест"),
        on_delete=models.CASCADE,
        related_name='digests',
    )
    tracker_issue_key = models.CharField(
        max_length=255,
        verbose_name=_("ключ тикета"),
        help_text=_("ключ тикета в трекере"),
        blank=True,
        db_index=True,
    )
    target_issue_status = models.CharField(
        max_length=100,
        verbose_name=_("статус тикета"),
        help_text=_("статус тикета в трекере, который нужно установить"),
    )
    process_status = models.CharField(
        max_length=10,
        verbose_name=_("статус обработки тикета"),
        choices=PROCESS_STATUS_CHOICES,
        default=PROCESS_STATUS_PENDING,
    )
    errors = models.TextField(_("сообщения об ошибках"), blank=True)

    history = HistoricalRecords(excluded_fields=['errors', 'process_status'])

    class Meta:
        verbose_name = _("дайджест для сотрудника")
        verbose_name_plural = _("дайджесты для сотрудников")

    def __str__(self):
        return _("Дайджест: {}| Пользователь: {}").format(self.periodic_role_digest_id, self.user_id)
