from django.conf import settings
from django.utils import timezone

from rest_framework import permissions
from rest_framework.exceptions import ValidationError

from kelvin.courses.models import (
    Course, CourseLessonLink, CoursePermission, CourseStudent, CourseTVMService, UserCLessonState,
)
from kelvin.lessons.models import LessonProblemLink
from kelvin.switcher import get_feature_bool_option as enabled_feature


class BaseCoursePermission(permissions.BasePermission):
    def check_permission(self, request, view, obj, permission_checker_func):
        if not enabled_feature('E7N_ENABLE_NEW_COURSE_PERMISSIONS') or request.user.is_superuser:
            return True
        if request.user.is_anonymous:
            return False
        if obj.available_for_support and request.user.is_support:
            return True
        course_permission = CoursePermission.objects.filter(
            user=request.user,
            course=obj,
        ).first()

        if course_permission is None:
            return False

        checker = getattr(course_permission, permission_checker_func)

        return checker()


class CanViewCourse(BaseCoursePermission):
    """
    Может видеть курс в лабе
    """
    def has_object_permission(self, request, view, obj):
        return self.check_permission(
            request,
            view,
            obj,
            "can_view"
        )


class CanEditCourse(BaseCoursePermission):
    """
    Может редактировать курс в лабе
    """

    def has_object_permission(self, request, view, obj):
        return self.check_permission(
            request,
            view,
            obj,
            "can_edit"
        )


class CanEditClesson(BaseCoursePermission):
    """
    Может редактировать модули курса в лабе
    """

    def has_object_permission(self, request, view, obj):
        if request.method in permissions.SAFE_METHODS:
            return True

        return self.check_permission(
            request,
            view,
            obj.course,
            "can_edit"
        )


class CanViewClesson(BaseCoursePermission):
    """
    Может видеть модули курса в лабе
    """

    def has_object_permission(self, request, view, obj):
        return self.check_permission(
            request, view, obj.course, "can_view"
        )


class CanEditProblem(BaseCoursePermission):
    """
    Может редактировать задания курса в лабе
    """

    def has_object_permission(self, request, view, obj):
        if request.method in permissions.SAFE_METHODS:
            return True

        if not enabled_feature('E7N_ENABLE_NEW_COURSE_PERMISSIONS') or request.user.is_superuser:
            return True

        if obj.available_for_support and request.user.is_support:
            return True

        lesson_ids = LessonProblemLink.objects.filter(problem=obj).values_list('lesson_id')
        clls = CourseLessonLink.objects.filter(
            lesson_id__in=lesson_ids,
        ).select_related('course')

        return any(
            self.check_permission(
                request,
                view,
                cll.course,
                "can_edit"
            )
            for cll in clls
        )


class CanViewCourseStats(BaseCoursePermission):
    """
    Может видеть статистику по курсу в лабе
    """

    def has_object_permission(self, request, view, obj):
        return self.check_permission(
            request,
            view,
            obj,
            "can_view_stats"
        )


class CanManageCourseRoles(BaseCoursePermission):
    """
    Может управлять ролями в курсе
    """

    def has_object_permission(self, request, view, obj):
        return self.check_permission(
            request,
            view,
            obj,
            "can_manage_roles"
        )


class ObjectDenyEditForAssigned(permissions.BasePermission):
    """
    Базовый пермишен запрещает редактировать курсы с непустым date_assignment.
    Этот пермишен его расширяет, позволяя явно установить data_assignment в None
    для возврата курса на редактирование
    """

    def has_object_permission(self, request, view, obj):
        """
        Позволяет явно установить date_assignment в None
        для отзыва курса на редактирование
        """
        if (
                'clesson' in request.data and
                'date_assignment' in request.data['clesson'] and
                request.data['clesson']['date_assignment'] is None
        ):
            return True

        """
        Дает возможность редактировать только объекты, у которых
        не проставлено поле `date_assignment`
        """
        if request.method in permissions.SAFE_METHODS:
            return True
        return not obj.date_assignment


class CanViewCuratorStats(permissions.BasePermission):
    """
        Проверяет, что пользователь имеет право смотреть статистику куратора
        по ВСЕМ предоставленным курсам
    """
    def has_permission(self, request, view):
        if request.user.is_superuser:
            return True

        raw_course_ids = request.GET.get('course_ids')

        if not raw_course_ids:
            return True

        course_ids = raw_course_ids.split(',')

        available_permissions = CoursePermission.objects.filter(
            course_id__in=course_ids,
            user=request.user,
        )

        if len(available_permissions) < len(course_ids):
            return False

        for permission in available_permissions:
            if not permission.can_view_stats():
                return False

        return True


class CanViewStudentStats(permissions.BasePermission):
    """
        Проверяет, что пользователь имеет право смотреть статистику куратора
        по ВСЕМ предоставленным курсам
    """
    def has_permission(self, request, view):
        raw_course_ids = request.GET.get('course_ids')

        if not raw_course_ids:
            return True

        course_ids = raw_course_ids.split(',')

        ever_assigned_courses_count = CourseStudent.objects.filter(
            student=request.user,
            course_id__in=course_ids,
        ).count()

        if ever_assigned_courses_count < len(course_ids):
            return False

        return True


class ObjectCLessonAssigned(permissions.BasePermission):
    """
    CourseLessonLink is allowed if date_assignment is in the past
    """
    def has_object_permission(self, request, view, obj):
        return bool(
            obj.date_assignment and
            obj.date_assignment < timezone.now()
        )


class ObjectCLessonAssignedToStudent(ObjectCLessonAssigned):
    """
    Позволяет смотреть курсо-занятия только если наступило время назначения
    `date_assignment` и занятие в курсе, которое назначено пользователю
    """
    def has_object_permission(self, request, view, obj):
        return (
            super(ObjectCLessonAssignedToStudent, self).has_object_permission(
                request, view, obj,
            )
        ) and (
            request.user  # check if it is real user
        ) and (
            CourseStudent.objects.filter(
                course=obj.course, student=request.user, deleted=False,
            ).exists()
        )


class ObjectCLessonAvailableToStudent(permissions.BasePermission):
    """
    Проверяет, доступен ли модуль ученику согласно настройкам траекторий прохождения модулей курса
    """
    def has_object_permission(self, request, view, obj):
        try:
            is_available = UserCLessonState.objects.get(
                user=request.user,
                clesson=obj
            ).available
        except UserCLessonState.DoesNotExist:
            is_available = settings.COURSES_DEFAULT_CLESSON_IS_AVAILABLE

        return is_available


def course_tvmservice_permission_factory(course_id_attr_name='course_pk', attr_path='kwargs'):
    class CourseTVMServicePermission(permissions.BasePermission):
        def get_course_id(self, view):
            obj = view
            for attr in attr_path.split('.'):
                obj = getattr(obj, attr)
            course_id = obj.get(course_id_attr_name)
            try:
                return int(course_id)
            except (ValueError, TypeError) as exc:
                raise ValidationError(exc)

        def _has_permission(self, tvm_service_id, view):
            course_id = self.get_course_id(view)
            return CourseTVMService.objects.filter(course_id=course_id, tvm_service_id=tvm_service_id).exists()

        def has_permission(self, request, view):
            return self._has_permission(
                tvm_service_id=getattr(request, 'tvm_service_id', 0) or 0,
                view=view,
            )

        def has_object_permission(self, request, view, obj):
            return self._has_permission(
                tvm_service_id=getattr(request, 'tvm_service_id', 0) or 0,
                view=view,
            )

    return CourseTVMServicePermission


class ObjectCourseAssignedToStudent(permissions.BasePermission):
    """
    Позволяет смотреть курс, который назначен пользователю
    """
    def has_object_permission(self, request, view, obj):
        return CourseStudent.objects.filter(course=obj, student=request.user.id).exists()


class ObjectForContentManager(permissions.BasePermission):
    """
    Доступ только контент-менеджерам
    """
    def has_object_permission(self, request, view, obj):
        return getattr(request.user, 'is_content_manager', False)


class ObjectCourseForAnonymous(permissions.BasePermission):
    """
    Позволяет смотреть курс, который доступен анонимным пользователям
    """
    def has_object_permission(self, request, view, obj):
        return obj.allow_anonymous


class ObjectCourseSourceAccessibleForTeacher(permissions.BasePermission):
    """
    Проверка, что пользователь является учителем и
    либо владельцем курса (если курс платный, бесплатный курс доступен
    только для чтения), либо у него есть курс для которого данный - книга.
    """
    def has_object_permission(self, request, view, obj):
        is_owner = request.user.id == obj.course.owner_id
        is_free = (
            request.method in permissions.SAFE_METHODS and obj.course.free
        )
        is_course_book = Course.objects.filter(
            source_courses__pk=obj.course_id,
            owner_id=request.user.id,
        ).exists()
        return request.user.is_teacher and (
            is_owner or is_free or is_course_book
        )


class ObjectAccessibleForContentManager(permissions.BasePermission):
    """
    Контент-менеджерам нет ограничения
    """
    def has_object_permission(self, request, view, obj):
        return (
            request.user.is_content_manager or
            (
                request.user.is_support and
                getattr(obj, 'available_for_support', False)
            )
        )


class ObjectCLessonAssignedToChildren(ObjectCLessonAssigned):
    """
    Позволяет смотреть курсо-занятия только если наступило время назначения
    `date_assignment` и занятие в курсе, которое назначено ребенку родителя
    """
    def has_object_permission(self, request, view, obj):
        return (
            super(ObjectCLessonAssignedToChildren, self).has_object_permission(
                request, view, obj,
            )
        ) and (
            request.user.is_authenticated and request.user.is_parent
        ) and (
            CourseStudent.objects.filter(
                course=obj.course,
                student__parents__parent=request.user.id,
            ).exists()
        )


class ObjectCLessonForAnonymous(permissions.BasePermission):
    """
    Позволяет смотреть курсозанятие в курсе для анонимных пользователей
    """
    def has_object_permission(self, request, view, obj):
        return obj.course.allow_anonymous


class ObjectCourseAssignedToChildren(permissions.BasePermission):
    """
    Позволяет смотреть курс, который назначен ребенку родителя
    """
    def has_object_permission(self, request, view, obj):
        return (
            request.user.is_authenticated and request.user.is_parent
        ) and (
            CourseStudent.objects.filter(
                course=obj,
                student__parents__parent=request.user.id,
            ).exists()
        )


class ObjectCourseLessonOwner(permissions.BasePermission):
    """
    Считаем, что владелец курсозанятия - это владелец курса, в котором
    находится курсозанятие
    """
    def has_object_permission(self, request, view, obj):
        return request.user.id == obj.course.owner_id
