from builtins import object, range
from datetime import timedelta
from typing import Optional

from past.utils import old_div

from django.conf import settings
from django.utils import timezone

from kelvin.accounts.models import User
from kelvin.common.staff_reader import staff_reader
from kelvin.courses.models import CourseLessonLink, CourseStudent
from kelvin.result_stats.models import StudentCourseStat
from kelvin.result_stats.serializers import StudentCourseStatInReportSerializer


class BaseStaffReport(object):
    """
    Базовый отчет по курсу для произвольного подмножества сотрудников из Staff
    """

    STATUS_DO_NOT_JOIN_MOE = 'do_not_join_moebius'
    STATUS_DO_NOT_JOIN_COURSE = 'do_not_join_course'
    STATUS_JOIN_COURSE = 'join_course'

    def __init__(self, courses, user=None, offset=0, limit=0):
        self.courses = courses
        self.user = user
        self.offset = offset
        self.limit = limit
        self._data = None
        self._course_stats = None
        self._course_stats = None
        self._staff_users = None
        self._students_map = None
        self._clesson_ids = None
        self._clessons = None
        self._clessons_map = None
        self._course_student_data = None
        self._course_clessons_map = None

    @property
    def staff_users(self):
        raise NotImplementedError(
            'Getting users from staff is not implemented'
        )

    @property
    def students_map(self):
        if self._students_map is None:
            usernames = [
                staff_user['login']
                for staff_user in self.staff_users
            ]
            self._students_map = {
                user.username: user.id
                for user in User.objects.filter(username__in=usernames)
            }
        return self._students_map

    @property
    def course_stats(self):
        if self._course_stats is None:
            self._course_stats = StudentCourseStat.objects.filter(
                course__in=self.courses,
                student_id__in=list(self.students_map.values()),
            ).select_related('course')
        return self._course_stats

    @property
    def course_student_data(self):
        if self._course_student_data is None:
            self._course_student_data = {
                course_student.student_id: {
                    'date_created': course_student.date_created,
                    'completed': course_student.completed,
                    'date_completed': course_student.date_completed,
                }
                for course_student in CourseStudent.objects.filter(
                    student_id__in=list(self.students_map.values()),
                    course_id__in=self.courses,
                )
            }
        return self._course_student_data

    @property
    def clessons(self):
        if self._clessons is None:
            self._clessons = CourseLessonLink.objects.filter(
                course__in=self.courses,
            ).select_related('lesson')
        return self._clessons

    @property
    def clesson_ids(self):
        if self._clesson_ids is None:
            self._clesson_ids = CourseLessonLink.objects.filter(
                course__in=self.courses,
            ).values_list('id', flat=True)
        return self._clesson_ids

    @property
    def course_clessons_map(self):
        if self._course_clessons_map is None:
            self._course_clessons_map = {}
            for clesson in self.clessons:
                if clesson.course_id in self._course_clessons_map:
                    self._course_clessons_map[clesson.course_id].append(clesson.id)
                else:
                    self._course_clessons_map[clesson.course_id] = [clesson.id]
        return self._course_clessons_map

    @property
    def clessons_map(self):
        if self._clessons_map is None:
            self._clessons_map = {clesson.id: clesson.lesson.name for clesson in self.clessons}
        return self._clessons_map

    def get_serializer(self, course_id):
        return StudentCourseStatInReportSerializer(
            context={'clessons': self.course_clessons_map.get(course_id, [])},
            many=True,
        )

    @property
    def data(self):
        completed = getattr(self, 'completed', None)
        min_delay = getattr(self, 'min_delay', None)
        max_date_created = None if min_delay is None else timezone.now() - timedelta(days=min_delay)
        if self._data is None:
            self._data = []
            for course in self.courses:
                course_stats = [stat for stat in self.course_stats if stat.course_id == course.id]
                stats = self.get_serializer(course.id).to_representation(course_stats)
                stats_map = {
                    stat.student_id: {
                        'course': stat.course,
                        'lessons': stats[i].get('lessons', []),
                    }
                    for i, stat in enumerate(course_stats)
                }
                for staff_user in self.staff_users:
                    student_data = {
                        'username': staff_user['login'],
                        'first_name': staff_user['name']['first']['ru'],
                        'last_name': staff_user['name']['last']['ru'],
                        'position': staff_user['official']['position']['ru'],
                        'department_group_name': (
                            staff_user['department_group']['name']
                        ),
                        'started_datetime': None,
                        'date_completed': None,
                        'completed': False,
                    }
                    student_id = self.students_map.get(staff_user['login'])
                    student_data['id'] = student_id
                    if not student_id:
                        student_data['status'] = self.STATUS_DO_NOT_JOIN_MOE
                    elif student_id not in stats_map:
                        student_data['status'] = self.STATUS_DO_NOT_JOIN_COURSE
                    else:
                        student_stats = stats_map[student_id]
                        clesson_results = []
                        if 'lessons' in student_stats:
                            for clesson_id, score in sorted(student_stats['lessons'].items()):
                                clesson_results.append({
                                    'score': score or 0,
                                    'course_name': student_stats['course'].name,
                                    'lesson_name': self.clessons_map[clesson_id],
                                    'clesson_id': clesson_id,
                                    'course_id': clesson_id,
                                })

                        student_data.update({
                            'status': self.STATUS_JOIN_COURSE,
                            'clessons_results': clesson_results,
                            'clessons_results_avg': (
                                old_div(sum(r['score'] for r in clesson_results), float(len(clesson_results)))
                                if clesson_results else 0
                            ),
                        })
                        if student_id in self.course_student_data:
                            course_student_data_item = self.course_student_data[student_id]
                            student_data['started_datetime'] = (
                                course_student_data_item['date_created'].strftime(settings.DATETIME_FORMAT)
                                if course_student_data_item['date_created'] else None
                            )
                            student_data['started_datetime_original'] = course_student_data_item['date_created']
                            student_data['date_completed'] = (
                                course_student_data_item['date_completed'].strftime(settings.DATETIME_FORMAT)
                                if course_student_data_item['date_completed'] else None
                            )
                            student_data['date_completed_original'] = course_student_data_item['date_completed']
                            student_data['completed'] = course_student_data_item['completed']
                    if (
                        (completed is None or student_data['completed'] == completed) and
                        (
                            max_date_created is None or (
                                not student_data.get('completed', False) and (
                                    student_data.get('started_datetime_original') is None or (
                                        student_data.get('started_datetime_original') and
                                        student_data['started_datetime_original'] < max_date_created
                                    )
                                )
                            )
                        )
                    ):
                        self._data.append(student_data)

        return self._data

    def table(self):
        if not self.data:
            return []
        clessons_count = len(self.clesson_ids)
        tab = [
            [
                u'ФИ', u'Логин', u'Должность', u'Подразделение',
            ] + [
                u'Оценка модуля {}'.format(i + 1) for i in range(clessons_count)
            ] + [
                u'Средняя успеваемость', u'Дата начала',
            ]
        ]
        for user in self.data:
            clesson_results = user.get('clessons_results') or ([''] * clessons_count)
            tab.append(
                [
                    u' '.join((user['first_name'], user['last_name'])),
                    user['username'],
                    user['position'],
                    user['department_group_name'],
                ] + clesson_results + [
                    user.get('clessons_results_avg', ''),
                    user.get('started_datetime', ''),
                ]
            )

        return tab


class FilteredBaseStaffReport(BaseStaffReport):
    def __init__(
        self,
        completed: Optional[bool] = None,
        with_nested: Optional[bool] = None,
        min_delay: Optional[int] = None,
        *args, **kwargs,
    ):
        self.completed = completed
        self.with_nested = with_nested
        self.min_delay = min_delay
        super().__init__(*args, **kwargs)


class HRUsersStaffReport(FilteredBaseStaffReport):
    """
    Отчет по курсу для подопечных HR-партнера
    """

    @property
    def staff_users(self):
        """
        Возращает подопечных HR-партнера
        из StaffReader для построения отчета
        """
        if self._staff_users is None:
            self._staff_users = staff_reader.get_hrusers(
                self.user.username,
                limit=self.limit,
                offset=self.offset,
                with_nested=getattr(self, 'with_nested', None),
            )


        return self._staff_users

    @property
    def data(self):
        if self._data is None:
            self._data = super(HRUsersStaffReport, self).data
            for user_data in self._data:
                if 'clessons_results' in user_data:
                    user_data['clessons_results'] = [
                        clesson_result['score']
                        for clesson_result in user_data.get('clessons_results', [])
                    ]
        return self._data


class ChiefUsersStaffReport(FilteredBaseStaffReport):
    """
    Отчет по курсу для подчиненных руководителя
    """

    @property
    def staff_users(self):
        """
        Возращает подчиненных для руководителя
        из StaffReader для построения отчета
        """
        if self._staff_users is None:
            self._staff_users = staff_reader.get_chiefusers(
                self.user.username,
                limit=self.limit,
                offset=self.offset,
                with_nested=getattr(self, 'with_nested', None),
            )

        return self._staff_users

    @property
    def data(self):
        if self._data is None:
            self._data = super(ChiefUsersStaffReport, self).data
            for user_data in self._data:
                user_data['clessons_results'] = [
                    clesson_result['score']
                    for clesson_result in user_data.get('clessons_results', [])
                ]
        return self._data


class CuratorReport(BaseStaffReport):
    """
    Отчет по курсу для куратора
    """

    def __init__(self, usernames, *args, **kwargs):
        super(CuratorReport, self).__init__(*args, **kwargs)
        self.usernames = usernames

    @property
    def staff_users(self):
        if self._staff_users is None:
            self.usernames = sorted(self.usernames)
            if self.usernames:
                self._staff_users = staff_reader.get_suggestuser_many(usernames=self.usernames)
            else:
                self._staff_users = []

        return self._staff_users

    @property
    def data(self):
        if self._data is None:
            super_data = super(CuratorReport, self).data
            self._data = []
            for user_data in super_data:
                for clesson_result in user_data.get('clessons_results', []):
                    self._data.append({
                        'id': user_data['id'],
                        'username': user_data['username'],
                        'first_name': user_data['first_name'],
                        'last_name': user_data['last_name'],
                        'score': clesson_result['score'],
                        'course_name': clesson_result['course_name'],
                        'lesson_name': clesson_result['lesson_name'],
                    })
        if self.offset:
            self._data = self._data[self.offset:]
        if self.limit:
            self._data = self._data[:self.limit]
        return self._data

    def table(self):
        tab = [
            [u'ФИ', u'Логин', u'Курс', u'Модуль', u'Успеваемость, %']
        ]

        for item in self.data:
            tab.append([
                u' '.join((item['first_name'], item['last_name'])),
                item['username'],
                item['course_name'],
                item['lesson_name'],
                item['score'],
            ])

        return tab
