from builtins import object
from functools import partial

from django.conf import settings
from django.utils import timezone

from rest_framework import serializers

from kelvin.common.serializer_mixins import DateFieldsYearLimitMixin, DateUpdatedFieldMixin, SerializerForeignKeyMixin
from kelvin.common.utils import generate_safe_code
from kelvin.courses.models import CourseLessonLink, UserCLessonState
from kelvin.lessons.serializers import LessonInCLessonSerializer

from .course_name import CourseNameSerializer
from .progress_indicator import ProgressIndicatorSerializer


class BaseCLessonSerializer(
    DateFieldsYearLimitMixin,
    DateUpdatedFieldMixin,
    SerializerForeignKeyMixin,
    serializers.ModelSerializer,
):
    """
    Сериализатор, который представляет связь курс-занятие аналогично занятию
    Используется во вьюхе.
    """
    lesson = LessonInCLessonSerializer()
    course = CourseNameSerializer()
    progress_indicator = ProgressIndicatorSerializer(required=False, allow_null=True, read_only=True)

    fk_update_fields = ['lesson']

    def update(self, instance, validated_data):
        """
        При апдейте занятия, при выдаче, в контекст
        складываем флаг just_assigned
        """
        if (
            instance.date_assignment is None and
            validated_data.get('date_assignment') is not None
        ):
            self.context['just_assigned'] = True
        return super(BaseCLessonSerializer, self).update(
            instance, validated_data)

    def to_representation(self, instance):
        """
        Преобразует представление в аналогичное занятию,
        возвращает только назначенные ученику задачи
        """
        ret = super(BaseCLessonSerializer, self).to_representation(instance)
        lesson = ret.pop('lesson')
        lesson['clesson'] = ret

        assignment = self.context.get('assignment')
        user = self.context['request'].user
        if assignment and not (user.is_teacher or user.is_content_manager):
            lesson['problems'] = [
                link for link in lesson['problems'] if link['id'] in assignment
            ]
        return lesson

    def transform_data(self, data):
        clesson = data.pop('clesson', {})
        if not isinstance(clesson, dict):
            raise serializers.ValidationError(
                'wrong value for `clesson`: {0}'.format(clesson))
        clesson['lesson'] = data
        return clesson

    def to_internal_value(self, data):
        """
        Преобразует в представление сериализатора
        """
        data = self.transform_data(data)
        return super(BaseCLessonSerializer, self).to_internal_value(data)

    def save(self, **kwargs):
        """
        При создании курсозанятия дает доступ к нему
        """
        if self.instance is None:
            kwargs['accessible_to_teacher'] = timezone.now()
        return super(BaseCLessonSerializer, self).save(**kwargs)

    class Meta(object):
        model = CourseLessonLink
        validators = [
            partial(CourseLessonLink.get_validator().validate, exception_class=serializers.ValidationError)
        ]
        read_only_fields = (
            'id',
            'accessible_to_teacher',
            'date_updated',
            'date_completed',
            'progress_indicator',
        )
        fields = (
            'course',
            'lesson',
            'mode',
            'duration',
            'show_answers_in_last_attempt',
            'date_assignment',
            'evaluation_date',
            'finish_date',
            'url',
            'start_date',
            'visual_mode',
            'lesson_block_slug',
        ) + read_only_fields


class CLessonSerializer(BaseCLessonSerializer):
    """
    Сериализатор, который представляет связь курс-занятие аналогично занятию
    Используется во вьюхе.
    """
    lesson = LessonInCLessonSerializer()
    available = serializers.SerializerMethodField()

    def get_available(self, obj):
        user = self.context['request'].user

        if not user.is_authenticated:
            return settings.COURSES_DEFAULT_CLESSON_IS_AVAILABLE and obj.course.allow_anonymous

        try:
            # при добавлении курса в "Мои" состояния пользователя в графе курса должно быть уже заполнено
            available = UserCLessonState.objects.get(user_id=self.context['request'].user.id, clesson=obj).available
        except UserCLessonState.DoesNotExist:
            # сюда попадаем, когда курс не добавлен в "Мои"
            available = settings.COURSES_DEFAULT_CLESSON_IS_AVAILABLE

        return available

    class Meta(BaseCLessonSerializer.Meta):
        additional_fields = (
            'access_code',
            'max_attempts_in_group',
            'show_all_problems',
            'comment',
            'available',
        )
        fields = BaseCLessonSerializer.Meta.fields + additional_fields

    def transform_data(self, data):
        data = super(CLessonSerializer, self).transform_data(data)

        if data.get('access_code') is True:
            # запрос на генерацию кода для занятия
            data['access_code'] = generate_safe_code(
                CourseLessonLink.ACCESS_CODE_LENGTH
            )
        return data
