from builtins import object

from django.apps import apps
from django.conf import settings

from rest_framework import serializers

from kelvin.accounts.serializers import StudentSerializer
from kelvin.accounts.utils import get_user_projects
from kelvin.common.serializer_mixins import DateUpdatedFieldMixin, ExcludeForStudentMixin, InfoFieldMixin
from kelvin.courses.models import AssignmentRule, Course, CoursePermission, Criterion
from kelvin.courses.serializers.course_lesson_link import CourseLessonLinkSerializer
from kelvin.group_levels.serializers import GroupLevelBasicSerializer
from kelvin.resources.serializers import ResourceSerializerCorp as ResourceSerializer

from .progress_indicator import ProgressIndicatorSerializer


class BaseCourseSerializer(
    DateUpdatedFieldMixin,
    InfoFieldMixin,
    ExcludeForStudentMixin,
    serializers.ModelSerializer,
):
    """
    Сериализатор для модели курса
    """
    class ClessonsOrdering(object):
        """
        Выды сортировки курсозанятий в представлении
        """
        RAW = 'raw'  # Не меняет сортировку

    lessons = CourseLessonLinkSerializer(
        source="courselessonlink_set",
        many=True,
        required=False,
    )
    group_levels = GroupLevelBasicSerializer(many=True)
    cover = ResourceSerializer(read_only=True)
    subject = serializers.CharField(source='subject.slug')
    closed = serializers.SerializerMethodField()
    mailing_list = serializers.SerializerMethodField()
    progress_indicator = ProgressIndicatorSerializer(
        required=False,
        allow_null=True,
        read_only=True,
    )

    student_serializer = StudentSerializer

    class Meta(object):
        model = Course
        read_only_fields = (
            'id',
            'cover',
            'date_updated',
            'free',
            'code',
            'subject',
            'mailing_list',
            'progress_indicator',
        )
        fields = (
            'name',
            'description',
            'lessons',
            'mode',
            'source_courses',
            'author',
            'group_levels',
            'color',
            'closed',
            'news',
            'info',
        ) + read_only_fields
        exclude_for_student = (
            'source_courses',
        )

    def get_mailing_list(self, course):
        """
        Значение поля mailing_list может быть либо None либо
        mailing list slug из рассылятора.

        :type course: Course
        """
        if hasattr(course, 'mailing_list') and course.mailing_list.slug:
            return course.mailing_list.slug
        return None

    def add_course_students(self, ret, course):
        students = course.students.all().order_by('id')

        student_serializer = self.student_serializer(
            many=True, context=self.context,
        )

        count = students.count()
        students = student_serializer.to_representation(
            students[:settings.MAX_COURSE_JOURNAL_STUDENTS]
        )
        if self.context.get('paginated'):
            ret['students'] = {
                'count': count,
                'items': students,
            }
        else:
            ret['students'] = student_serializer.to_representation(
                students
            )
        return ret

    @staticmethod
    def get_code(course):
        """
        Возвращает код курса если он бесплатный
        """
        if course.free:
            return course.code

        return None

    @staticmethod
    def get_closed(course):
        """
        Галочка закрытости курса
        """
        return course.date_closed is not None

    def sort_lessons(self, ret):
        # Без 'date_assignment' идут позже, поэтому
        # `not x['date_assignment']`
        # True > False, поэтому `not x['accessible_to_teacher']`
        ret['lessons'].sort(
            key=lambda link: (
                not link['date_assignment'],
                link['date_assignment'],
                not link['accessible_to_teacher'],
                link['accessible_to_teacher'],
            )
        )
        return ret

    def to_representation_with_students(self, ret, course):
        if self.context.get('with_students') == 'true':
            ret = self.add_course_students(ret, course)
        return ret

    def to_representation_with_clessons_ordering(self, ret):
        if self.context.get('clessons_ordering') != self.ClessonsOrdering.RAW:
            ret = self.sort_lessons(ret)
        return ret

    def to_representation(self, course):
        """
        1. Добавляет учеников курса в представление, если передан параметр
           `with_students`
        2. Если параметр `clessons_ordering` не равен `'raw'`,
           сортирует ссылки на занятия следующим образом:
           cначала назначенные учителем по дате назначения от ранней к поздней
           затем доступные учителю с сохранением исходного порядка
           затем недоступные учителю с сохранением исходного порядка

        Сортировка происходит после сериализации
        """
        ret = super(BaseCourseSerializer, self).to_representation(course)
        ret = self.to_representation_with_students(ret, course)
        ret = self.to_representation_with_clessons_ordering(ret)
        return ret


class CourseSimpleSuggestSerializer(serializers.ModelSerializer):
    """
    CourseSimpleSuggestSerializer ...
    """

    class Meta(object):
        model = Course
        fields = (
            'id',
            'name',
            'subject',
        )

    subject = serializers.CharField(
        source="subject.name",
        max_length=255
    )


class CourseSerializer(BaseCourseSerializer):
    """
    Сериализатор для модели курса
    """
    is_assigned = serializers.SerializerMethodField()
    permissions = serializers.SerializerMethodField()
    cover = ResourceSerializer()

    def get_is_assigned(self, obj):
        """
        Узнаем, назначени ли курс (obj) текущему пользователю,
        которого мы пытаемся узнать из request
        """
        user_id = self.context.get('user_id', None) or self.context['request'].user.id
        return obj.is_assigned(user_id)

    def get_permissions(self, course):
        request = self.context.get('request', None)

        if request is None or not getattr(self.context['request'], 'user'):
            return CoursePermission.get_default_actions_dict(default=0)

        user = self.context['request'].user

        try:
            return CoursePermission.objects.get(
                user=user,
                course=course
            ).get_actions_dict()
        except CoursePermission.DoesNotExist as e:
            return CoursePermission.get_default_actions_dict(default=int(user.is_superuser))

    class Meta(BaseCourseSerializer.Meta):
        model = Course
        additional_read_only_fields = (
            'supports_web',
            'supports_ios',
            'supports_android',
            'is_assigned',
            'permissions',
            'subject_id',
            'project',
        )
        subject_id = serializers.IntegerField(source="subject.id")
        fields = BaseCourseSerializer.Meta.fields + additional_read_only_fields
        read_only_fields = (
            BaseCourseSerializer.Meta.read_only_fields +
            additional_read_only_fields
        )

    def to_representation_with_credentials(self, ret, *_, **__):
        return ret


class CourseEditSerializer(serializers.ModelSerializer):
    """ Сериализатор для поддержки write-операций с курсом """
    """ TODO: сделать альяс так, чтобы фронт передавал не cover, а cover_id"""
    source_courses = serializers.PrimaryKeyRelatedField(
        many=True,
        queryset=Course.objects.all(),
    )

    class Meta(object):
        model = Course
        fields = [
            'id',
            'name',
            'description',
            'subject',
            'owner',
            'free',
            'cover',
            'source_courses',
            'project',
        ]

    def validate_project(self, value):
        if not value:
            raise serializers.ValidationError("Project should be not empty")
        user = self.context['request'].user
        if value not in get_user_projects(user):
            raise serializers.ValidationError("Project {} is not among the projects allowed for user {}".format(
                value,
                user.username
            ))
        return value


class AssignmentRuleSerializer(serializers.ModelSerializer):
    class Meta(object):
        model = AssignmentRule
        fields = [
            'id',
            'course',
            'title',
            'formula',
            'mandatory'
        ]


class AssignmentRuleTriggerSerializer(serializers.ModelSerializer):
    priority = serializers.IntegerField(required=False)
    name = serializers.CharField(required=False)

    class Meta(object):
        model = Criterion
        fields = [
            'id',
            'clesson',
            'name',
            'priority',
            'formula',
            'actions',

        ]
        read_only_fields = [
            'assignment_rule',
        ]
