from builtins import object

from django.utils import timezone

from rest_framework import serializers

from kelvin.accounts.serializers.owner import CourseOwnerSerializer
from kelvin.courses.models import Course, CourseLessonLink
from kelvin.lessons.models import LessonProblemLink
from kelvin.resources.serializers import ResourceSerializerCorp as ResourceSerializer
from kelvin.results.models import CourseLessonResult

from .progress_indicator import ProgressIndicatorSerializer


class CourseDetailSerializer(serializers.ModelSerializer):
    """
    Сериализатор информации о курсе
    """
    points = serializers.SerializerMethodField()
    code = serializers.SerializerMethodField()
    subject = serializers.CharField(source='subject.name')
    closed = serializers.SerializerMethodField()
    cover_details = serializers.SerializerMethodField()
    is_assigned = serializers.SerializerMethodField()
    mandatory = serializers.SerializerMethodField()
    owner = CourseOwnerSerializer()

    progress_indicator = ProgressIndicatorSerializer(
        required=False,
        allow_null=True,
    )

    average_score = serializers.DecimalField(
        decimal_places=1,
        max_digits=2,
    )
    feedback = serializers.SerializerMethodField()
    mandatory = serializers.SerializerMethodField()

    class Meta(object):
        model = Course
        fields = (
            'id',
            'name',
            'description',
            'points',
            'cover',
            'cover_details',
            'info',
            'free',
            'code',
            'subject',
            'closed',
            'progress_indicator',
            'is_assigned',
            'average_score',
            'feedback',
            'mandatory',
            'project_id',
            'owner',
            'author',
        )

    def get_is_assigned(self, obj):
        """
        Узнаем, назначени ли курс (obj) текущему пользователю,
        которого мы пытаемся узнать из request
        """
        user_id = self.context.get('user_id', None) or self.context['request'].user.id
        return obj.is_assigned(user_id)

    def get_mandatory(self, obj):
        """
        Узнаем, назначени ли курс (obj) текущему пользователю в обязательном порядке,
        которого мы пытаемся узнать из request
        """
        user_id = self.context.get('user_id', None) or self.context['request'].user.id
        return obj.is_assigned_mandatory(user_id)

    @staticmethod
    def get_code(course):
        """
        Возвращает код курса если он бесплатный
        """
        if course.free:
            return course.code

        return None

    def get_cover_details(self, course):
        # TODO: remove this field and serialize dict in 'cover' (INTLMS-345)
        return ResourceSerializer(course.cover, read_only=True).data

    def get_points(self, course):
        """
        Возвращает число максимальных для курса и
        набранных пользователем баллов
        """
        max_points = 0

        lesson_problem_link_qs = LessonProblemLink.objects.filter(
            problem__isnull=False,
            lesson_id__in=CourseLessonLink.objects.filter(
                course=course,
                date_assignment__isnull=False,
            ).values_list('lesson_id', flat=True)
        ).select_related(
            'problem',
            'lesson',
        )

        for lpl in lesson_problem_link_qs:
            if lpl.options is None or (
                getattr(lpl, 'options', {}).get('max_points')
            )is None:
                max_points += lpl.problem.max_points
            else:
                max_points += lpl.options.get('max_points')

        results = CourseLessonResult.objects.filter(
            summary__clesson__course=course.pk,
            summary__student_id=self.context.get('user_id')
        ).values_list(
            'points',
            'summary__clesson__finish_date',
            'summary__clesson__evaluation_date',
            'summary__clesson__mode',

        )

        points_sum = 0
        for result_item in results:
            [points, finish_date, evaluation_date, mode] = result_item
            points = int(points)  # for a case
            if points == 0:
                continue

            # тренировки считаем сразу как получен результат
            if mode not in CourseLessonLink.EVALUATION_LESSON_MODES:
                points_sum += points
                continue

            # получаем дату доступности результата для контрольной
            publish_results_date = (
                finish_date if evaluation_date is None else evaluation_date
            )
            if (
                publish_results_date is None or
                publish_results_date < timezone.now()
            ):
                points_sum += points

        return {
            'points': points_sum,
            'max_points': max_points
        }

    @staticmethod
    def get_closed(course):
        """
        Галочка закрытости курса
        """
        return course.date_closed is not None

    def get_feedback(self, course):

        feedback = course.feedback_set.filter(
            user=self.context['request'].user,
        ).first()

        if not feedback:
            return {}

        return {"score": feedback.score, "comment": feedback.comment}

    def get_mandatory(self, obj):
        """
        Узнаем, назначени ли курс (obj) текущему пользователю в обязательном порядке,
        которого мы пытаемся узнать из request
        """
        user_id = self.context.get('user_id', None)
        return obj.is_assigned_mandatory(user_id)
