from builtins import object

from rest_framework import serializers

from kelvin.common.serializer_mixins import DateUpdatedFieldMixin
from kelvin.common.serializers import ManyToManyListSerializer
from kelvin.lessons.serializer_fields import ExpandableLessonField

from ..models import CourseLessonLink
from .progress_indicator import ProgressIndicatorSerializer


class BaseCourseLessonLinkSerializer(DateUpdatedFieldMixin, serializers.ModelSerializer):
    """
    Сериализатор связи курс-занятие
    Используется только для сериализации уроков в CourseSerializer.
    Снаружи используется только метод control_work_data
    """
    lesson = ExpandableLessonField()
    progress_indicator = ProgressIndicatorSerializer(
        required=False,
        read_only=True,
    )

    class Meta(object):
        model = CourseLessonLink
        fields = (
            'id',
            'lesson',
            'mode',
            'duration',
            'show_answers_in_last_attempt',
            'accessible_to_teacher',
            'date_assignment',
            'date_updated',
            'date_completed',
            'finish_date',
            'url',
            'start_date',
            'progress_indicator',
            'visual_mode',
            'lesson_block_slug',
            'max_attempts_in_group',
            'show_all_problems',
            'comment',
        )
        read_only_fields = (
            'accessible_to_teacher',
            'date_updated',
            'date_completed',
            'progress_indicator',
        )
        list_serializer_class = ManyToManyListSerializer
        extra_kwargs = {
            'order': {'required': False, 'write_only': True},
        }

    @staticmethod
    def control_work_data(clesson, result, now):
        """
        Дополнительные данные по контрольным
        """
        if not clesson:
            return {}
        date_format = serializers.DateTimeField().to_representation
        return {
            'started_at': date_format(result.date_created) if result else None,
            'time_left': 0 if result and result.completed else max(
                0,
                (result.quiz_time_limit(clesson) - now).total_seconds()
            ) if result else None,
            'finish_date': date_format(clesson.finish_date),
            'evaluation_date': (date_format(clesson.evaluation_date)
                                if clesson.evaluation_date else ''),
            'is_finished': now >= clesson.finish_date,
            'is_evaluated': (now >= clesson.evaluation_date
                             if clesson.evaluation_date else False),
            'viewed': result.viewed if result else None,
        }


class CourseLessonLinkSerializer(BaseCourseLessonLinkSerializer):
    """
    Сериализатор связи курс-занятие
    Используется только для сериализации уроков в CourseSerializer.
    Снаружи используется только метод control_work_data
    """
    class Meta(BaseCourseLessonLinkSerializer.Meta):
        additional_fields = (
            'access_code',
        )
        fields = BaseCourseLessonLinkSerializer.Meta.fields + additional_fields
