from builtins import object

from rest_framework import serializers

from kelvin.common.serializer_mixins import DateUpdatedFieldMixin
from kelvin.courses.models import Course, CoursePermission
from kelvin.resources.serializers import ResourceSerializerCorp as ResourceSerializer

from .progress_indicator import ProgressIndicatorSerializer


class BaseCourseShortSerializer(DateUpdatedFieldMixin, serializers.ModelSerializer):
    cover = ResourceSerializer(read_only=True)
    lessons_count = serializers.IntegerField(read_only=True)
    closed = serializers.SerializerMethodField()
    progress_indicator = ProgressIndicatorSerializer(
        required=False,
        read_only=True,
    )

    class Meta(object):
        model = Course
        read_only_fields = (
            'id',
            'cover',
            'date_updated',
            'free',
            'progress_indicator',
        )
        fields = (
            'name',
            'mode',
            'author',
            'color',
            'description',
            'lessons_count',
            'closed',
        ) + read_only_fields

    def get_closed(self, course):
        """Галочка закрытости курса"""
        return course.date_closed is not None


class CourseShortSerializer(BaseCourseShortSerializer):
    """
    Краткий сериализатор модели `Course`
    """
    is_assigned = serializers.SerializerMethodField()
    permissions = serializers.SerializerMethodField()
    cover = ResourceSerializer(read_only=True)

    def get_is_assigned(self, obj):
        """
        Узнаем, назначени ли курс (obj) текущему пользователю,
        которого мы пытаемся узнать из request
        """
        user_id = self.context.get('user_id', None) or self.context['request'].user.id
        return obj.is_assigned(user_id)

    def get_permissions(self, course):
        request = self.context.get('request', None)

        if request is None or not getattr(self.context['request'], 'user'):
            return CoursePermission.get_default_actions_dict(default=0)

        user = self.context['request'].user

        try:
            return CoursePermission.objects.get(
                user=user,
                course=course
            ).get_actions_dict()
        except CoursePermission.DoesNotExist as e:
            return CoursePermission.get_default_actions_dict(default=int(user.is_superuser))

    class Meta(BaseCourseShortSerializer.Meta):
        additional_read_only_fields = (
            'code',
            'supports_web',
            'supports_ios',
            'supports_android',
            'is_assigned',
            'permissions',
            'cover',
        )
        additional_fields = (
            'copyright',
        )
        fields = (
            BaseCourseShortSerializer.Meta.fields +
            additional_read_only_fields + additional_fields
        )
        read_only_fields = (
            BaseCourseShortSerializer.Meta.fields +
            additional_read_only_fields
        )
