from builtins import object

from rest_framework import serializers

from kelvin.courses.models import Course, CourseLessonLink
from kelvin.resources.serializers import ResourceSerializerCorp as ResourceSerializer

from .progress_indicator import ProgressIndicatorSerializer


class CoursesAssignedSerializer(serializers.ModelSerializer):
    """
    Сериализатор назначенных пользователю курсов
    """
    cover = ResourceSerializer(read_only=True)
    subject = serializers.CharField(source='subject.slug')
    subject_name = serializers.CharField(source='subject.name')
    controlworks = serializers.SerializerMethodField()
    average_score = serializers.DecimalField(
        decimal_places=1,
        max_digits=2,
    )
    progress_indicator = ProgressIndicatorSerializer(
        required=False,
        allow_null=True,
    )
    mandatory = serializers.SerializerMethodField()

    class Meta(object):
        model = Course
        fields = (
            'id',
            'name',
            'description',
            'code',
            'color',
            'cover',
            'date_updated',
            'subject',
            'subject_name',
            'news',
            'info',
            'controlworks',
            'progress_indicator',
            'average_score',
            'mandatory',
        )
        read_only_fields = (
            'id',
            'name',
            'description',
            'code',
            'color',
            'cover',
            'date_updated',
            'subject',
            'subject_name',
            'news',
            'info',
            'controlworks',
            'progress_indicator',
            'average_score',
            'mandatory',
        )

    @staticmethod
    def get_controlworks(course):
        """
        Возвращает список имеющихся контрольных работ курса
        """
        result = (item.id
                  for item in course.courselessonlink_set.all()
                  if item.mode == CourseLessonLink.CONTROL_WORK_MODE)

        return result

    def get_mandatory(self, obj):
        """
        Узнаем, назначени ли курс (obj) текущему пользователю в обязательном порядке,
        которого мы пытаемся узнать из request
        """
        user_id = self.context.get('user_id', None)
        return obj.is_assigned_mandatory(user_id)
