from builtins import object

from rest_framework import serializers

from kelvin.accounts.serializers.student import StudentSerializer
from kelvin.courses.models import Course
from kelvin.courses.serializers.sirius_course_lesson_link import SiriusCourseLessonLinkSerializer
from kelvin.resources.serializers import ResourceSerializerCorp as ResourceSerializer


class CoursesListSerializer(serializers.ModelSerializer):
    """
    .. Сериализатор для модели `Course`

    .. _object:
    .. csv-table:: Response Object
        :header: Name, Type, Required, Description
        :widths: 15, 15, 10, 60

        "id", "int", "yes", "(ro) ID учебного курса"
        "name", "string", "yes", "Название учебного курса"
        "subject", "Subject", "yes", "Учебный предмет курса"
        "lessons", "array", "no", "Уроки курса"

    """
    lessons = SiriusCourseLessonLinkSerializer(
        source='courselessonlink_set',
        many=True,
        required=False,
    )
    cover = ResourceSerializer(read_only=True)
    subject = serializers.CharField(source='subject.slug')
    closed = serializers.SerializerMethodField()

    class Meta(object):
        model = Course
        fields = (
            'id',
            'name',
            'description',
            'lessons',
            'color',
            'cover',
            'date_updated',
            'free',
            'code',
            'subject',
            'supports_web',
            'supports_ios',
            'supports_android',
            'closed',
            'prices',
            'news',
            'info',
        )
        read_only_fields = (
            'id',
            'cover',
            'date_updated',
            'free',
            'code',
            'subject',
            'supports_web',
            'supports_ios',
            'supports_android',
            'prices',
        )

    def to_representation(self, course):
        """
        Сортирует ссылки на занятия следующим образом:
        Сначала назначенные учителем по дате назначения от ранней к поздней
        Затем доступные учителю с сохранением исходного порядка
        Затем недоступные учителю с сохранением исходного порядка

        Сортировка происходит после сериализации
        """
        ret = super(CoursesListSerializer, self).to_representation(course)

        # добавляет учеников из курсе
        if self.context.get('with_students'):
            ret['students'] = StudentSerializer(many=True).to_representation(
                course.students.all())

        # Без 'date_assignment' идут позже, поэтому `not x['date_assignment']`
        # True > False, поэтому `not x['accessible_to_teacher']`
        ret['lessons'].sort(
            key=lambda link: (
                not link['date_assignment'],
                link['date_assignment'],
                not link['accessible_to_teacher'],
                link['accessible_to_teacher'],
            )
        )
        return ret

    def get_closed(self, course):
        """Галочка закрытости курса"""
        return course.date_closed is not None
