from builtins import object

import dateutil.parser

from django.conf import settings
from django.utils import timezone

from rest_framework import serializers

from kelvin.common.serializer_fields import JSONField
from kelvin.courses.models import CourseLessonGraphFactory, CourseLessonLink, CourseStudent, UserCLessonState
from kelvin.courses.serializers import ProgressIndicatorSerializer
from kelvin.lessons.models import LessonProblemLink


def get_max_points_from_problem_list(problem_list):
    """
    Get max points from list of problems
    :param problem_list:
    :return:
    """
    max_points = 0
    for item in problem_list:
        if item['options'] is not None:
            max_points += (
                item['problem__max_points']
                if item['options'].get('max_points') is None else
                item['options'].get('max_points')
            )
        else:
            max_points += item['problem__max_points']

    return max_points


class SiriusCourseLessonWithResultSerializer(serializers.ModelSerializer):
    """
    Сериализатор курсозанятий с результатами
    """
    _result_data = {}

    lesson_id = serializers.IntegerField(source='lesson.id')
    course_id = serializers.IntegerField(source='course.id')
    name = serializers.CharField(source='lesson.name')
    info = JSONField(source='lesson.info')
    problems_count = serializers.SerializerMethodField()
    problems_completed = serializers.SerializerMethodField()
    points = serializers.SerializerMethodField()
    max_points = serializers.SerializerMethodField()
    progress_points = serializers.SerializerMethodField()
    progress_max_points = serializers.SerializerMethodField()
    progress = serializers.SerializerMethodField()
    started_at = serializers.SerializerMethodField()
    completed = serializers.SerializerMethodField()
    progress_indicator = ProgressIndicatorSerializer(required=False)
    available = serializers.SerializerMethodField()

    class Meta(object):
        model = CourseLessonLink
        fields = (
            'id',
            'points',
            'max_points',
            'progress_points',
            'progress_max_points',
            'progress',
            'lesson_id',
            'name',
            'date_assignment',
            'date_completed',
            'course_id',
            'finish_date',
            'evaluation_date',
            'start_date',
            'mode',
            'completed',
            'duration',
            'problems_count',
            'problems_completed',
            'comment',
            'started_at',
            'progress_indicator',
            'allow_time_limited_problems',
            'lesson_block_slug',
            'info',
            'available',
        )

    def get_result_data(self, clesson):
        """
        Get result data from context from
        SiriusCourseLessonResultsGroupSerializer
        :param clesson:
        :return:
        """
        clesson_results = [x for x in self.context.get('course_lesson_results').data if x['clesson_id'] == clesson.pk]

        self._result_data[clesson.pk] = sorted(
            clesson_results,
            key=lambda result: result['date_created'],
            reverse=True
        )

        return self._result_data[clesson.pk]

    def get_completed(self, clesson):
        """
        Узнаем завершен модуль или нет по последней попытки
        """
        sorted_results = self.get_result_data(clesson)

        return sorted_results[0]['completed'] if len(sorted_results) else None

    def get_available(self, obj):
        for userclessonstate in obj.userclessonstate_set.all():
            # при добавлении курса в "Мои" состояния пользователя в графе курса должно быть уже заполнено
            return userclessonstate.available
        # сюда попадаем, когда курс не добавлен в "Мои"
        return settings.COURSES_DEFAULT_CLESSON_IS_AVAILABLE

    def get_problems_count(self, clesson):
        """
        Возвращает количество заданий в курсозанятии (с учетом того,
        что в курсе может быть много модулей/курсозанятий
        с одним и тем же lesson + variate_problems_count)
        """
        lesson_problem_links_common = [x for x in self.context.get('lesson_problem_link_values') if (
            x['type'] == LessonProblemLink.TYPE_COMMON and
            x['lesson_id'] == clesson.lesson.pk
        )]

        variate_problems = set(
            [
                p['group']
                for p in lesson_problem_links_common
                if p['group'] is not None
            ]
        )
        not_variate_problems = [
            p for p in lesson_problem_links_common if p['group'] is None
        ]
        return len(variate_problems) + len(not_variate_problems)

    def get_started_at(self, clesson):
        """
        Возвращает время и дату начала выполнения курсозанятия
        """
        sorted_results = self.get_result_data(clesson)

        return (
            sorted_results[0]['date_created'] if len(sorted_results) else None
        )

    def get_problems_completed(self, clesson):
        """
        Возвращает количество пройденных задач
        (с учетом того, что в курсе может быть много модулей/курсозанятий
        с одним и тем же lesson)
        """
        sorted_results = self.get_result_data(clesson)
        if len(sorted_results) == 0:
            return 0

        answered_count = 0
        for answer in sorted_results[0]['answers']:
            answered_count += 1 if [x for x in sorted_results[0]['answers'][answer] if x['answered']] else 0

        return answered_count

    def get_points(self, clesson):
        """
        Возвращает количество набранных балов
        """
        results = self.get_result_data(clesson)

        points_sum = 0
        for result in results:
            points = int(result['points'])  # for a case
            if points == 0:
                continue

            # тренировки считаем сразу как получен результат
            if clesson.mode not in CourseLessonLink.EVALUATION_LESSON_MODES:
                points_sum += points
                continue

            # получаем дату доступности результата для контрольной
            publish_results_date = (
                result['finish_date']
                if result['evaluation_date'] is None else
                result['evaluation_date']
            )
            publish_results_date = dateutil.parser.parse(publish_results_date)
            if (
                publish_results_date is None or
                publish_results_date < timezone.now()
            ):
                points_sum += points

        return points_sum

    def get_max_points(self, clesson):
        lesson_problem_links = [x for x in self.context.get('lesson_problem_link_values') if (
            x['problem_id'] is not None and
            x['lesson_id'] == clesson.lesson.pk
        )]

        not_variated_problems = [x for x in lesson_problem_links if x['group'] is None]

        max_points = get_max_points_from_problem_list(not_variated_problems)

        if clesson.mode in CourseLessonLink.EVALUATION_LESSON_MODES:
            # учитываем вариативность для контрольных работ
            # (получаем max_points из назначенных problem's)
            assignements = self.context.get('lesson_assignments_problems')
            clesson_assignements = (
                assignements[clesson.pk]
                if clesson.pk in assignements else
                None
            )

            # тот случай, когда пользователь опоздал
            # на контрольную с вариативностью
            unique_group = {}
            variate_problems = []
            for p in lesson_problem_links:
                if (
                    p['group'] is not None and
                    unique_group.get(p['group']) is None
                ):
                    unique_group[p['group']] = True
                    variate_problems.append(p)

            if clesson_assignements is not None:
                # тот случай, когда проходил контрольную
                # и есть назначенные задания
                variate_problems = [x for x in lesson_problem_links if (
                    x['group'] is not None and
                    x['id'] in clesson_assignements
                )]

            max_points += get_max_points_from_problem_list(variate_problems)

        return max_points

    def get_progress_points(self, clesson):
        """
        Возвращает количество набранных балов за просмотр теории
        """
        results = self.get_result_data(clesson)

        points_sum = 0
        for result in results:
            points = len(result.get('answers', {}) or {})
            points_sum += points

        return points_sum

    def get_progress_max_points(self, clesson):
        lesson_problem_links = [x for x in self.context.get('lesson_problem_link_values') if (
            (
                x['theory_id'] is not None or
                x['problem_id'] is not None
            ) and
            x['lesson_id'] == clesson.lesson.id
        )]
        return len(lesson_problem_links)

    def get_progress(self, clesson):
        progress_max_points = self.get_progress_max_points(clesson)
        progress_points = self.get_progress_points(clesson)

        progress = 0
        if progress_max_points:
            progress = int(progress_points * 100 / progress_max_points)

        return progress
