import logging
from builtins import object

from rest_framework import serializers
from rest_framework.fields import empty

from kelvin.common.serializer_fields import JSONField
from kelvin.common.serializer_mixins import DateUpdatedFieldMixin
from kelvin.common.serializers import ManyToManyListSerializer
from kelvin.courses.models import CourseLessonLink
from kelvin.lessons.serializers import LessonNameSerializer
from kelvin.results.models import CourseLessonResult
from kelvin.sapi.serializers import SiriusCourseLessonResultSerializer

logger = logging.getLogger(__name__)


class SiriusCourseLessonLinkSerializer(
    DateUpdatedFieldMixin,
    serializers.ModelSerializer,
):
    """
    Сериализатор связи курс-занятие
    """
    lesson = LessonNameSerializer()
    url = JSONField(required=False)
    problems_count = serializers.SerializerMethodField()
    problems = serializers.SerializerMethodField()
    course_name = serializers.CharField(source='course.name')
    course_id = serializers.CharField(source='course.id')

    def __init__(
        self,
        instance=None,
        data=empty,
        include_problems=True,
        **kwargs
    ):
        super(SiriusCourseLessonLinkSerializer, self).__init__(
            instance, data, **kwargs
        )

        self.include_problems = include_problems

        self._cached_problems = {}

    class Meta(object):
        model = CourseLessonLink
        fields = (
            'id',
            'course_name',
            'course_id',
            'lesson',
            'order',  # только для импорта из Учебника; потом удалить
            'mode',
            'duration',
            'max_attempts_in_group',
            'show_answers_in_last_attempt',
            'show_all_problems',
            'accessible_to_teacher',
            'date_assignment',
            'date_updated',
            'date_completed',
            'url',
            'start_date',
            'finish_date',
            'comment',
            'problems_count',
            'problems',
        )
        read_only_fields = (
            'accessible_to_teacher',
            'date_updated',
            'date_completed',
        )
        list_serializer_class = ManyToManyListSerializer
        extra_kwargs = {
            'order': {'required': False, 'write_only': True},
        }

    def get_problems_count(self, clesson):
        """
        Возвращает количество задач курсозанятия
        """
        if not self.include_problems:
            return None

        return len(self.get_problems(clesson))

    def get_problems(self, clesson):
        """
        Возвращает данные по задачам
        """
        if not self.include_problems:
            return []

        if clesson.pk in self._cached_problems:
            return self._cached_problems[clesson.pk]

        self._cached_problems[clesson.pk] = SiriusCourseLessonResultSerializer(
            instance=CourseLessonResult.objects.filter(
                summary__clesson=clesson
            ).select_related(),
            many=True
        ).data

        return self._cached_problems[clesson.pk]

    @staticmethod
    def control_work_data(clesson, result, now):
        """
        Дополнительные данные по контрольным и диагностикам
        """
        if not clesson:
            return {}
        date_format = serializers.DateTimeField().to_representation
        return {
            'started_at': date_format(result.date_created) if result else None,
            'time_left': 0 if result and result.completed else max(
                0,
                (result.quiz_time_limit(clesson) - now).total_seconds()
            ) if result else None,
            'finish_date': date_format(clesson.finish_date),
            'evaluation_date': (date_format(clesson.evaluation_date)
                                if clesson.evaluation_date else ''),
            'is_finished': now >= clesson.finish_date,
            'is_evaluated': (now >= clesson.evaluation_date
                             if clesson.evaluation_date else False),
        }
