import logging
from datetime import datetime, timedelta
from typing import Iterable, List, NewType, Optional
from yandex_tracker_client.exceptions import NotFound as TrackerIssueNotFound

from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.contenttypes.models import ContentType
from django.contrib.postgres.aggregates import BoolOr
from django.db import models, transaction
from django.db.models import Avg, Q
from django.db.models.functions import Cast
from django.utils import timezone

from kelvin.common.startrek.client import startrek_api
from kelvin.lessons.models import Lesson
from kelvin.lessons.services import update_available_for_support_on_lessons

from ..common.utils import generate_safe_code
from ..lesson_assignments.models import LessonAssignment
from .mails import PeriodicNotificationEmail
from .models import (
    AssignmentRule, Course, CourseFeedback, CourseLessonLink, CourseStudent, Criterion, ExcludedUser, NotifyDayOff,
    PeriodicCourse, PeriodicNotification, PeriodicRoleDigest, PeriodicStudentNotification, RoleDigest, UserCriterion,
)
from .tracker_templates.notification import TrackerNotificationIssue
from .tracker_templates.role_digest import TrackerRoleDigestIssue

__all__ = ["copy_course", "add_student", "apply_criterion", "prepare_periodic_course_notifications"]
UserId = NewType('UserId', int)


logger = logging.getLogger(__name__)
User = get_user_model()

NOTIFICATION_BATCH_SIZE = settings.COURSES_PERIODIC_NOTIFICATION_BATCH_SIZE


@transaction.atomic()
def copy_course(course, owner=None, reset_clessons=True):
    """
    Copies the course with course lessons and change current instance.

    :param owner: owner of copied course
    :param reset_clessons: need to edit clessons fields or not
    """
    clessons = list(course.courselessonlink_set.all())
    source_courses = list(course.source_courses.all())

    course.copy_of_id = course.id
    course.color = None
    course.pk = None

    if owner:
        course.owner = owner

    if course.code:
        course.code = generate_safe_code(course.CODE_LENGTH)

    course.journal = None
    course.save()

    course.source_courses = source_courses

    for clesson in clessons:
        clesson.copy_of_id = clesson.pk
        clesson.pk = None
        clesson.course = course

        if reset_clessons:
            clesson.lesson_editable = False
            clesson.date_assignment = None
        clesson.journal = None

    CourseLessonLink.objects.bulk_create(clessons)


def add_student(course, student):
    CourseStudent.objects.get_or_create(
        course=course,
        student=student,
    )
    LessonAssignment.ensure_student_assignments(
        course=course,
        student=student,
    )


def update_course_average_score(course_id) -> None:
    average_score = CourseFeedback.objects.filter(
        course_id=course_id,
    ).aggregate(avg_score=Avg('score'))['avg_score']

    average_score = min(max(average_score, CourseFeedback.MIN_SCORE), CourseFeedback.MAX_SCORE)

    Course.objects.filter(pk=course_id).update(average_score=average_score)


def update_available_for_support_on_course_lesson_links(course_lesson_links: Iterable[CourseLessonLink]) -> None:
    lessons_to_update = Lesson.objects.filter(
        id__in=CourseLessonLink.objects.filter(
            id__in=[cll.id for cll in course_lesson_links],
        ).values_list('lesson_id', flat=True),
    )
    lessons_to_update.update(available_for_support=True)

    update_available_for_support_on_lessons(lessons=lessons_to_update, available_for_support=True)


def update_unavailable_for_support_on_course_lesson_links(course_lesson_links: Iterable[CourseLessonLink]) -> None:
    lessons_to_update = Lesson.objects.filter(
        id__in=CourseLessonLink.objects.filter(
            id__in=[cll.id for cll in course_lesson_links],
        ).values_list('lesson_id', flat=True),
    ).annotate(
        max_available_for_support=BoolOr('courselessonlink__available_for_support'),
    ).filter(
        max_available_for_support=False,
    )
    lessons_to_update.update(available_for_support=False)

    update_available_for_support_on_lessons(lessons=lessons_to_update, available_for_support=False)


def update_available_for_support_on_courses(courses: Iterable[Course], available_for_support: bool) -> None:
    course_lesson_link_to_update = CourseLessonLink.objects.filter(course_id__in=courses)
    course_lesson_link_to_update.update(available_for_support=available_for_support)

    if available_for_support:
        update_available_for_support_on_course_lesson_links(course_lesson_link_to_update)
    else:
        update_unavailable_for_support_on_course_lesson_links(course_lesson_link_to_update)


def update_available_for_support_on_course(course: Course) -> None:
    if course.tracker.has_changed('available_for_support'):
        update_available_for_support_on_courses(courses=[course], available_for_support=course.available_for_support)


def update_available_for_support_on_course_lesson_link(course_lesson_link: CourseLessonLink) -> None:
    if course_lesson_link.tracker.has_changed('available_for_support'):
        if course_lesson_link.available_for_support:
            update_available_for_support_on_course_lesson_links([course_lesson_link])
        else:
            update_unavailable_for_support_on_course_lesson_links([course_lesson_link])


def on_update_assignment_rule_make_available_for_support(assignment_rule: AssignmentRule):
    RULES_TO_MAKE_AVAILABLE_FOR_SUPPORT = [
        {
            'value': 'Коммерческий департамент (Поддержка бизнеса)',
            'operation': '==',
            'semantic_type': 'StaffGroup'
        },
        {
            'value': 'Внешние консультанты',
            'operation': '==',
            'semantic_type': 'StaffGroup'
        },
    ]

    for disjunction in assignment_rule.formula:
        for predicate in disjunction:
            if predicate in RULES_TO_MAKE_AVAILABLE_FOR_SUPPORT:
                course = assignment_rule.course
                course.available_for_support = True
                course.save()
                return


def apply_criterion(criterion: Criterion, force: bool):
    from kelvin.results.models import CourseLessonResult

    students = User.objects.filter(
        id__in=(
            CourseLessonResult.objects
                .filter(summary__clesson__course_id=criterion.assignment_rule.course_id)
                .select_related('summary')
                .values_list('summary__student', flat=True)
        )
    )
    if not force:
        students = students.exclude(
            id__in=UserCriterion.objects.filter(criterion=criterion, done=True).values_list('user', flat=True),
        )

    for student in students:
        criterion.do_actions(user=student, force=force)


def prepare_periodic_course_notifications() -> None:
    """
    формирует список получателей по периодическим курсам
    """
    now = timezone.now()

    # проверяем можно ли сейчас отправлять уведомления
    if not NotifyDayOff.objects.allowed(now):
        logger.info("periodic notification is not allowed today")
        return

    # выбираем периодические курсы
    active_courses = PeriodicCourse.objects.filter(
        Q(course__date_closed__isnull=True) |
        Q(course__date_closed__gt=now)
    )

    excluded_user_ids = User.objects.filter(
        username__in=ExcludedUser.objects.filter(
            exclude_from_courses=True
        ).values_list('login', flat=True)
    ).values_list('pk', flat=True)

    for periodic_course in active_courses:
        notifications = PeriodicNotification.objects.filter(
            periodic_course=periodic_course,
        ).order_by('-priority', 'delay')

        previous_notification = None
        for notification in notifications:
            queryset = CourseStudent.objects.filter(
                course_id=periodic_course.course_id,
                completed=False,
                date_created__lte=now.date() - timedelta(days=notification.delay),
                student__is_dismissed=False,
            )

            if previous_notification:
                # проверяем наличие отправленных уведомлений
                # по предыдущему уведомлению
                previously_notified = PeriodicStudentNotification.objects.filter(
                    course_id=periodic_course.course_id,
                    notification=previous_notification,
                    status=PeriodicStudentNotification.STATUS_SENT,
                ).values_list('student_id', flat=True)
                queryset = queryset.filter(id__in=previously_notified)

            # исключаем тех, кому уже отправили уведомление
            already_sent = PeriodicStudentNotification.objects.filter(
                course=periodic_course.course_id,
                notification=notification,
            ).values_list('student_id', flat=True)
            queryset = queryset.exclude(id__in=already_sent)

            # исключаем тех, кому нельзя отправлять уведомления
            if excluded_user_ids:
                queryset = queryset.exclude(student_id__in=excluded_user_ids)

            # создаем уведомления, ожидающие отправки
            PeriodicStudentNotification.objects.bulk_create([
                PeriodicStudentNotification(
                    notification=notification,
                    student=student,
                    course_id=periodic_course.course_id,
                ) for student in queryset
            ], batch_size=NOTIFICATION_BATCH_SIZE)

            previous_notification = notification


def create_periodic_email(student_notification: PeriodicStudentNotification):
    """
    Отправляет уведомление на почту
    """
    PeriodicNotificationEmail(student_notification).send()


def create_periodic_tracker_issue(student_notification: PeriodicStudentNotification):
    """
    Создает тикет в трекере
    """
    issue = TrackerNotificationIssue(student_notification=student_notification)
    user = student_notification.student.student
    tracker_issue = startrek_api.issues.create(**issue.as_dict())
    if tracker_issue.status.name != student_notification.notification.parameters['issue_status']:
        issue_transition = student_notification.notification.parameters['issue_transition']
        tracker_issue.transitions[issue_transition].execute()
    student_notification.result_data = str(tracker_issue.key)
    open_comment = issue.get_open_comment()
    if open_comment:
        tracker_issue.comments.create(text=open_comment, summonees=[user.username])


PERIODIC_NOTIFY_MAP = {
    PeriodicNotification.NOTIFY_TYPE_EMAIL: create_periodic_email,
    PeriodicNotification.NOTIFY_TYPE_TRACKER: create_periodic_tracker_issue,
}


def periodic_notify_student(student_notification: PeriodicStudentNotification):
    notification = student_notification.notification
    func = PERIODIC_NOTIFY_MAP.get(notification.notify_type)
    if func:
        func(student_notification)


def create_open_comment(tracker_role_digest_issue: TrackerRoleDigestIssue, tracker_issue, summonees=None):
    open_comment = tracker_role_digest_issue.get_open_comment()
    if open_comment:
        comment_kwargs = {'text': open_comment}
        if summonees:
            comment_kwargs['summonees'] = summonees
        tracker_issue.comments.create(**comment_kwargs)


def create_close_comment(tracker_role_digest_issue: TrackerRoleDigestIssue, tracker_issue, summonees=None):
    close_comment = tracker_role_digest_issue.get_close_comment()
    if close_comment:
        comment_kwargs = {'text': close_comment}
        if summonees:
            comment_kwargs['summonees'] = summonees
        tracker_issue.comments.create(**comment_kwargs)


def create_issue(role_digest: RoleDigest):
    tracker_role_digest_issue = TrackerRoleDigestIssue(role_digest=role_digest)
    tracker_issue = startrek_api.issues.create(**tracker_role_digest_issue.as_dict())
    tracker_issue_key = str(tracker_issue.key)

    create_open_comment(
        tracker_role_digest_issue=tracker_role_digest_issue,
        tracker_issue=tracker_issue,
        summonees=[role_digest.user.username],
    )

    role_digest.tracker_issue_key = tracker_issue_key


def open_issue(role_digest: RoleDigest):
    tracker_role_digest_issue = TrackerRoleDigestIssue(role_digest=role_digest)
    tracker_issue = startrek_api.issues[role_digest.tracker_issue_key]

    open_status = role_digest.periodic_role_digest.parameters['open_status']
    if tracker_issue.status.name != open_status:
        open_transition = role_digest.periodic_role_digest.parameters['open_transition']
        tracker_issue.transitions[open_transition].execute()

    create_open_comment(
        tracker_role_digest_issue=tracker_role_digest_issue,
        tracker_issue=tracker_issue,
        summonees=[role_digest.user.username],
    )


def close_issue(role_digest: RoleDigest):
    tracker_role_digest_issue = TrackerRoleDigestIssue(role_digest=role_digest)
    if role_digest.tracker_issue_key:
        tracker_issue = startrek_api.issues[role_digest.tracker_issue_key]
        create_close_comment(tracker_role_digest_issue=tracker_role_digest_issue, tracker_issue=tracker_issue)

        close_status = role_digest.periodic_role_digest.parameters['close_status']
        if tracker_issue.status.name != close_status:
            close_transition = role_digest.periodic_role_digest.parameters['close_transition']
            close_resolution = role_digest.periodic_role_digest.parameters['close_resolution']
            tracker_issue.transitions[close_transition].execute(resolution=close_resolution)


def create_role_digest_tracker_issue(role_digest: RoleDigest):
    if role_digest.tracker_issue_key:
        close_role_digest_tracker_issue(role_digest=role_digest)

    create_issue(role_digest=role_digest)


def close_role_digest_tracker_issue(role_digest: RoleDigest):
    close_issue(role_digest=role_digest)


def update_role_digest_tracker_issue(role_digest: RoleDigest):
    if role_digest.tracker_issue_key:
        open_issue(role_digest=role_digest)
    else:
        create_role_digest_tracker_issue(role_digest=role_digest)


def process_role_digest(role_digest: RoleDigest):
    if role_digest.target_issue_status == role_digest.periodic_role_digest.parameters['open_status']:
        if role_digest.periodic_role_digest.can_reopen_issue and role_digest.tracker_issue_key:
            update_role_digest_tracker_issue(role_digest=role_digest)
        else:
            create_role_digest_tracker_issue(role_digest=role_digest)
    else:
        close_role_digest_tracker_issue(role_digest=role_digest)


def user_to_role_qs(
    periodic_role_digest: PeriodicRoleDigest,
    course_student_ids: Iterable[int],
) -> Iterable[int]:
    from kelvin.tags.models import TaggedObject, TagTypeStaffChief,  TagTypeStaffHRBP

    DIGEST_ROLE_TAG_MAP = {
        PeriodicRoleDigest.ROLE_CHIEF: TagTypeStaffChief,
        PeriodicRoleDigest.ROLE_HRBP: TagTypeStaffHRBP,
    }

    return (
        TaggedObject.objects
        .annotate(
            tag_value_int=Cast('tag__value', output_field=models.IntegerField())
        )
        .filter(
            tag__type=DIGEST_ROLE_TAG_MAP[periodic_role_digest.role].get_db_type(),
            content_type=ContentType.objects.get(app_label='accounts', model='user'),
            object_id__in=course_student_ids,
        )
        .exclude(
            tag_value_int__in=(
                User.objects
                .filter(
                    username__in=ExcludedUser.objects.filter(exclude_from_issues=True).values_list('login', flat=True)
                )
                .values_list('id', flat=True)
            ),
        )
        .distinct()
        .values_list('tag_value_int', flat=True)
    )


def get_users_with_students_with_expired_delay(
    periodic_role_digest: PeriodicRoleDigest,
    for_users: Optional[Iterable[User]] = None,
) -> Iterable[UserId]:
    """
    Получить список пользователей, котором нужно отправлять дайджест по periodic_role_digest
    Если указан for_users, то возвращаются пользователи только из этого списка
    """
    now = timezone.now()

    course_students = CourseStudent.objects.filter(
        course_id=periodic_role_digest.periodic_course.course_id,
        completed=False,
        date_created__lt=now - timedelta(days=periodic_role_digest.delay)
    )

    course_student_ids = course_students.only('student_id').distinct().values_list('student_id', flat=True)

    users = user_to_role_qs(
        periodic_role_digest=periodic_role_digest,
        course_student_ids=course_student_ids,
    )

    if for_users:
        users = users.filter(tag_value_int__in=[user.id for user in for_users])

    return users


def update_role_digests(periodic_role_digest: PeriodicRoleDigest):
    """
    Обновление дайджестов пользователей
    """

    # существующие открытые дайджесты
    # отображение: id_пользователя - дайджест
    existing_opened_digests_map: dict[UserId, RoleDigest] = {}

    # существующие дайджесты
    # отображение: id_пользователя - дайджест
    existing_digests_map: dict[UserId, RoleDigest] = {}

    for role_digest in RoleDigest.objects.filter(periodic_role_digest=periodic_role_digest):
        if role_digest.target_issue_status != periodic_role_digest.parameters['close_status']:
            existing_opened_digests_map[role_digest.user_id] = role_digest
        existing_digests_map[role_digest.user_id] = role_digest

    # пользователи, у которых должен быть дайджест
    users_to_create_digests = set(get_users_with_students_with_expired_delay(periodic_role_digest=periodic_role_digest))

    # закрытие дайджестов, которые должны быть закрыты (существующие открытые минус те, которые нужно создать)
    users_digests_to_close: set[UserId] = set(existing_opened_digests_map.keys()) - users_to_create_digests
    for user_digest_to_close in users_digests_to_close:
        digest = existing_opened_digests_map[user_digest_to_close]
        digest.target_issue_status = periodic_role_digest.parameters['close_status']
        digest.errors = ""
        digest.process_status = RoleDigest.PROCESS_STATUS_PENDING
        digest.save()

    # создание дайджестов, которые должны быть созданы (которые нужно создать минус уже существующие)
    users_digests_to_create: set[UserId] = users_to_create_digests - set(existing_digests_map.keys())
    for user_digest_to_create in users_digests_to_create:
        RoleDigest.objects.create(
            periodic_role_digest=periodic_role_digest,
            user_id=user_digest_to_create,
            target_issue_status=periodic_role_digest.parameters['open_status'],
        )

    # обновление существующих дайджестов (которые нужно создать и уже существующие)
    users_digests_to_update: set[UserId] = users_to_create_digests & set(existing_digests_map.keys())
    for user_digest_to_update in users_digests_to_update:
        digest = existing_digests_map[user_digest_to_update]
        digest.target_issue_status = periodic_role_digest.parameters['open_status']
        digest.errors = ""
        digest.process_status = RoleDigest.PROCESS_STATUS_PENDING
        digest.save()


def process_periodic_role_digest(periodic_role_digest: PeriodicRoleDigest):
    today = datetime.today().date()
    if not NotifyDayOff.objects.allowed(today):
        logger.info("periodic role digest is not allowed today")
        return

    if periodic_role_digest.next_run > today:
        return

    with transaction.atomic():
        update_role_digests(periodic_role_digest=periodic_role_digest)
        next_run = max(
            periodic_role_digest.next_run + timedelta(days=periodic_role_digest.interval),
            today + timedelta(days=1),
        )
        periodic_role_digest.next_run = next_run
        periodic_role_digest.save()


def close_role_digest(periodic_role_digest: PeriodicRoleDigest, user: User) -> RoleDigest:
    """
    Закрытие дайджеста конкретного пользователя
    Когда студент завершил курс
    """
    role_digest = RoleDigest.objects.filter(periodic_role_digest=periodic_role_digest, user=user).first()

    # если нет дайджеста, то нечего закрывать
    if role_digest is None:
        return

    # если дайджест уже закрыт, то повторно не закрываем
    if role_digest.target_issue_status == periodic_role_digest.parameters['close_status']:
        return

    users_to_create_digests = get_users_with_students_with_expired_delay(
        periodic_role_digest=periodic_role_digest,
        for_users=[user],
    )

    # если еще есть должники, то не закрываем дайджест
    if user.id in users_to_create_digests:
        return

    role_digest.target_issue_status = periodic_role_digest.parameters['close_status']
    role_digest.errors = ""
    role_digest.process_status = RoleDigest.PROCESS_STATUS_PENDING
    role_digest.save()

    return role_digest


def process_role_digest_on_complete_course_student(
    course_student: CourseStudent,
    for_periodic_role_digest: Optional[PeriodicRoleDigest] = None,
):
    if not course_student.completed:
        return

    periodic_role_digests = (
        [for_periodic_role_digest] if for_periodic_role_digest
        else (
            PeriodicRoleDigest.objects
            .filter(periodic_course__course_id=course_student.course_id)
            .select_related('periodic_course')
        )
    )

    for periodic_role_digest in periodic_role_digests:
        for_users = User.objects.filter(
            id__in=user_to_role_qs(
                periodic_role_digest=periodic_role_digest,
                course_student_ids=[course_student.student_id],
            )
        )

        for for_user in for_users:
            close_role_digest(periodic_role_digest=periodic_role_digest, user=for_user)


def close_student_notification_tracker_issues(course_student_id: int):
    """
    Закрытие созданных на студента тикетов с уведомлениями
    """
    # находим призывы пользователя в курс через трекер
    student_notifications = PeriodicStudentNotification.objects.filter(
        student_id=course_student_id,
        status=PeriodicStudentNotification.STATUS_SENT,
        notification__notify_type=PeriodicNotification.NOTIFY_TYPE_TRACKER,
    ).select_related('notification')

    # закрываем незакрытые тикеты
    for student_notification in student_notifications:
        tracker_issue_key = student_notification.result_data
        if not tracker_issue_key:
            continue
        try:
            tracker_issue = startrek_api.issues[tracker_issue_key]
        except TrackerIssueNotFound:
            continue
        close_status = student_notification.notification.parameters['close_status']
        if tracker_issue.status.name != close_status:
            close_transition = student_notification.notification.parameters['close_transition']
            close_resolution = student_notification.notification.parameters['close_resolution']
            tracker_issue.transitions[close_transition].execute(resolution=close_resolution)


def delete_pending_periodic_student_notifications_for_student(course_student_id: int):
    """
    Удаление необработанных уведомлений для студента
    """
    PeriodicStudentNotification.objects.filter(
        student_id=course_student_id,
        status=PeriodicStudentNotification.STATUS_PENDING,
    ).delete()
