import logging

from django.conf import settings
from django.db import models, transaction
from django.db.models import F
from django.db.models.signals import post_delete, post_save
from django.dispatch import receiver

from kelvin.accounts.update_user_courses_task import set_update_user_courses_task
from kelvin.accounts.utils import invalidate_user_courses_cache
from kelvin.courses.models import Course, CourseFeedback, CourseLessonLink, CoursePermission
from kelvin.courses.models.assignment_rule import AssignmentRule
from kelvin.courses.tasks import add_user_to_revisor_group

from .models.periodic import CourseStudent, RoleDigest
from .services import (
    on_update_assignment_rule_make_available_for_support, process_role_digest_on_complete_course_student,
    update_available_for_support_on_course, update_available_for_support_on_course_lesson_link,
    update_course_average_score,
)
from .tasks import (
    stop_course_student_notifications_task, process_role_digest_on_complete_course_student_task,
    process_role_digest_task,
)

logger = logging.getLogger(__name__)


def recompute_curator_role(sender, instance, **kwargs):
    course_permission = instance
    user = course_permission.user

    curated_courses_count = CoursePermission.objects.annotate(
        only_curator_role=F('permission').bitand(CoursePermission.CURATOR)
    ).filter(
        user=user,
        only_curator_role__gt=0,
    ).count()

    user.is_curator = (curated_courses_count > 0)
    user.save()


def create_user_course_permission(sender, instance, **kwargs):
    """
    Добавляем создателю курса соответствующую роль при сохранении курса
    """
    course_permission = None

    try:
        course_permission = CoursePermission.objects.get(user=instance.owner, course=instance)
        course_permission.permission |= CoursePermission.OWNER

        course_permission.save()
    except CoursePermission.DoesNotExist:
        CoursePermission.objects.get_or_create(
            user=instance.owner,
            course=instance,
            permission=CoursePermission.OWNER
        )


def add_reviewer_to_revisor_group(sender, instance, **kwargs):
    add_user_to_revisor_group.delay(instance.id)


def update_course_average_score_handler(sender, instance, **kwargs):
    update_course_average_score(instance.course_id)


post_save.connect(update_course_average_score_handler, CourseFeedback)
post_save.connect(create_user_course_permission, Course)
# post_save.connect(add_reviewer_to_revisor_group, CoursePermission)
post_save.connect(recompute_curator_role, CoursePermission)
post_delete.connect(recompute_curator_role, CoursePermission)


@receiver(signal=models.signals.post_save, sender=AssignmentRule)
def assignment_rule_post_save_handler(instance, **kwargs):
    # обновим кэш владельца курса
    if settings.USER_COURSE_CACHE_ENABLE:
        invalidate_user_courses_cache(instance.course.owner)
        set_update_user_courses_task(user_id=instance.course.owner.id, force_update=True)


@receiver(signal=models.signals.post_save, sender=Course)
def course_post_save_handler(instance, **kwargs):
    # обновим кэш владельца курса
    if settings.USER_COURSE_CACHE_ENABLE:
        invalidate_user_courses_cache(instance.owner)
        set_update_user_courses_task(user_id=instance.owner.id, force_update=True)


@receiver(signal=models.signals.post_delete, sender=AssignmentRule)
def assignment_rule_post_delete_handler(instance, **kwargs):
    # обновим кэш владельца курса
    if settings.USER_COURSE_CACHE_ENABLE:
        invalidate_user_courses_cache(instance.course.owner)
        set_update_user_courses_task(user_id=instance.course.owner.id, force_update=True)


@receiver(signal=models.signals.post_delete, sender=Course)
def course_post_delete_handler(instance, **kwargs):
    # обновим кэш владельца курса
    if settings.USER_COURSE_CACHE_ENABLE:
        invalidate_user_courses_cache(instance.owner)
        set_update_user_courses_task(user_id=instance.owner.id, force_update=True)


@receiver(signal=models.signals.post_save, sender=Course)
def update_available_for_support_on_course_handler(sender, instance, **kwargs):
    update_available_for_support_on_course(instance)


@receiver(signal=models.signals.post_save, sender=CourseLessonLink)
def update_available_for_support_on_course_lesson_link_handler(sender, instance, **kwargs):
    update_available_for_support_on_course_lesson_link(instance)


@receiver(signal=models.signals.post_save, sender=AssignmentRule)
def on_update_assignment_rule_make_available_for_support_handler(sender, instance, **kwargs):
    on_update_assignment_rule_make_available_for_support(instance)


@receiver(signal=models.signals.post_save, sender=RoleDigest)
def role_digest_post_save_handler(sender, instance: RoleDigest, **kwargs):
    transaction.on_commit(lambda: process_role_digest_task.delay(role_digest_id=instance.id))


@receiver(signal=models.signals.post_save, sender=CourseStudent)
def course_student_post_save_handler(sender, instance: CourseStudent, **kwargs):
    if instance.tracker.has_changed('completed') and instance.completed:
        process_role_digest_on_complete_course_student_task.delay(course_student_id=instance.id)
        stop_course_student_notifications_task.delay(course_student_id=instance.id)
