var DashboardApp = DashboardApp || {data: {}};

$(function(){
    // Add trailing slash to backbone model views
    var _sync = Backbone.sync;
    Backbone.sync = function(method, model, options){

        var _url = _.isFunction(model.url) ?  model.url() : model.url;
        if (_url.indexOf("?") == -1) {
            _url += _url.charAt(_url.length - 1) == '/' ? '' : '/';
        }

        options = _.extend(options, {
            url: _url
        });

        return _sync(method, model, options);
    };

    // Models

    DashboardApp.CLesson = Backbone.Model.extend({
        urlRoot: "/api/v2/clessons/",
        defaults: {
            clesson: {
                id: null
            }
        }
    });

    DashboardApp.Course = Backbone.Model.extend({
        urlRoot: "/api/v2/courses/",
        defaults: {
            lessons: []
        }
    });

    // модель задачи
    // создаем эти модели сами; состоят из идентификатора и имени
    // TODO удалить модель, добавить название задачи в ProblemStat
    DashboardApp.Problem = Backbone.Model.extend({});
    DashboardApp.ProblemCollection = Backbone.Collection.extend({
        model: DashboardApp.Problem,
    })

    // модель статистики, сколько студентов онлайн
    DashboardApp.OnlineStudentStat = Backbone.Model.extend({
        defaults: {
            online: 0,
            total: 0
        },
        url: function(){
            return "/api/v2/users_status/course_stat/?course=" + DashboardApp.data.courseId;
        }
    });

    // статистика по задаче
    DashboardApp.ProblemStat = Backbone.Model.extend({
        defaults: {
            problem: null,
            right: 0,
            wrong: 0,
            no_answer: 0,
            first_attempt: 0,
            solving: 0
        }
    });
    DashboardApp.ProblemStatCollection = Backbone.Collection.extend({
        model: DashboardApp.ProblemStat,
        url: function() {
            return "/api/v2/clesson_results/origin_course_stat/?clesson=" + this.clessonId;
        },
        initialize: function(attributes, options) {
            this.clessonId = options.clessonId;
        }
    });

    // Views

    // вид выбора занятия в курсе
    DashboardApp.CourseView = Backbone.View.extend({
        courseTemplate: _.template($("#course_select_template").html()),
        events: {
            "click #show-lesson-button": 'showLesson'
        },
        initialize: function(){
            this.listenTo(this.model, "change", this.render);
            this.render();
        },
        render: function() {
            // рендерим и вставляем шаблон
            this.$el.html(this.courseTemplate(this.model.toJSON()));

            // проставляем текущее занятие в селекторе
            try {
                var route = Backbone.history.getFragment();
                if (route) {
                    clessonId = route.split("/")[0]
                    this.$el.find("#lesson-selector").val(clessonId);
                }
            } catch (e){
                // FIXME `Backbone.history.getFragment` кидает исключение
                // вообще надо что-то переделать, в модели должен быть
                // идентификатор выбираемого занятия
                // убрать `render` из `initialize`?
            }

            // отображение числа онлайн пользователей
            if (!this.statView) {
                var onlineStat = new DashboardApp.OnlineStudentStat();
                this.statView = new DashboardApp.OnlineStudentStatView({model: onlineStat});
                onlineStat.fetch()

                // обновление статистики
                setInterval(function(){onlineStat.fetch();}, 15000);
            }
            this.$el.append(this.statView.el);

            return this;
        },

        showLesson: function() {
            var clessonId = this.$el.find('select').val();
            DashboardApp.router.navigate(clessonId + "/problems", {trigger: true});
        }
    });

    // отображение задачи в iframe и статистики по задаче
    DashboardApp.ProblemStatView = Backbone.View.extend({
        className: "row",
        problemStatTemplate: _.template($("#problem_stat_template").html()),
        problemTemplate: _.template($("#problem_template").html()),
        initialize: function(attributes){
            this.listenTo(this.model, "change", this.render);
            this.$el.html(this.problemTemplate({
                src: DashboardApp.data.frontendUrl + this.model.get("problem") +
                    "/?show_solution=1&show_answers=1",
                id: this.model.get("problem"),
                index: attributes.index
            }));
            this.render();
        },
        render: function(){
            this.$el.find("div.problem-stat").html(this.problemStatTemplate(this.model.toJSON()));
            return this
        }
    });

    // вкладка статистики по задачам
    DashboardApp.ProblemsView = Backbone.View.extend({
        initialize: function(attributes, options) {
            this.statCollection = options.statCollection;
            this.listenTo(this.statCollection, "update", this.renderStats);
            this.listenTo(this.statCollection, "reset", this.resetStats);
            this.listenTo(this.collection, "reset", this.renderProblemButtons);
            this.render();
        },
        events: {
            "click .problem-button-container .btn": "scrollToProblem"
        },
        problemButtonTemplate: _.template($("#problem_button_template").html()),
        render: function() {
            this.renderProblemButtons();
            this.renderStats();
            return this;
        },
        renderProblemButtons(){
            // кнопки задач
            var problemButtons = this.$el.find(".problem-button-container");
            problemButtons.empty();
            this.collection.each(function(problem){
                problemButtons.append(this.problemButtonTemplate(problem.toJSON()));
            }, this);
        },
        renderStats: function(){
            // задача и статистика по ним
            if (!this.statViews) {
                this.statViews = {}
                var problemList = this.$el.find(".problem-list-container");
                problemList.empty();
                this.statCollection.each(function(problemStat, index){
                    var problemStatView = new DashboardApp.ProblemStatView(
                        {model: problemStat, index: index + 1});
                    problemList.append(problemStatView.el);
                    this.statViews[problemStat.get("problem")] = problemStatView;
                }, this);

                var that = this;
                this.statCollectionIntervalId = setInterval(
                    function(){that.statCollection.fetch();}, 10000);
            } else {
                _.each(this.statCollection.models, function(problemStat){
                    this.statViews[problemStat.get("problem")].model.set(problemStat.toJSON());
                }, this);
            }
        },
        resetStats: function(){
            _.each(this.statViews, function(view){view.remove();});
            this.statViews = null;
            if (this.statCollectionIntervalId) {
                clearInterval(this.statCollectionIntervalId);
                this.statCollectionIntervalId = null;
            }
            this.renderStats();
        },
        scrollToProblem: function(event){
            var $button = $(event.target);
            $button.siblings().removeClass("btn-primary");
            $button.addClass("btn-primary");
            var problemId = $button.attr("data-problem-id");
            $('.problem-list-container').animate({
                scrollTop: (
                    $(".problem-iframe[data-problem-id=" + problemId+ "]").offset().top -
                    $(".problem-iframe").first().offset().top
                )
            }, 1000);
        }
    })

    // главный вид с занятием
    DashboardApp.LessonView = Backbone.View.extend({
        initialize: function() {
            this.listenTo(this.model, "change", this.render);
            this.$el.find("#problems-content").hide();
            this.$el.find("#results-content").hide();
            this.$el.find("#groups-content").hide();
            this.render();
        },
        events: {
            "click #problems-tab": "showProblemsTab",
            "click #results-tab": "showResultsTab",
            "click #groups-tab": "showGroupsTab",
        },
        render: function() {
            // отображение задач
            var problemCollection = new DashboardApp.ProblemCollection(
                _.filter(
                    _.map(
                        this.model.get("problems"),
                        function (problemLink, index) {
                            if (problemLink.problem != null) {
                                return {
                                    id: problemLink.problem.id,
                                    name: "Задача " + (index + 1)
                                };
                            } else {
                                return null;
                            }
                        }
                    ),
                    function (x) {return x != null;}
                )
            )
            var statCollection = new DashboardApp.ProblemStatCollection(
                _.filter(
                    _.map(
                        this.model.get("problems"),
                        function (problemLink, index) {
                            if (problemLink.problem) {
                                return {
                                    problem: problemLink.problem.id
                                }
                            } else {
                                return null;
                            }
                        }
                    ),
                    function (x) {return x != null;}
                ),
                {clessonId: this.model.get("clesson").id}
            )
            if (!this.problemsView) {
                this.problemsView = new DashboardApp.ProblemsView(
                    {
                        el: "#problems-content",
                        collection: problemCollection
                    },
                    {
                        statCollection: statCollection
                    }
                );
            } else {
                this.problemsView.collection.reset(problemCollection.models);
                this.problemsView.statCollection.reset(statCollection.models);
                this.problemsView.statCollection.clessonId = statCollection.clessonId;
            }
            return this;
        },
        showProblemsTab: function() {
            DashboardApp.router.navigate(this.model.get("id") + "/problems", {trigger: true});
        },
        showResultsTab: function() {
            DashboardApp.router.navigate(this.model.get("id") + "/results", {trigger: true});
        },
        showGroupsTab: function() {
            DashboardApp.router.navigate(this.model.get("id") + "/groups", {trigger: true});
        },
        setTab: function(tabName) {
            this.$el.find("div.app-content").hide();
            this.$el.find("#" + tabName + "-content").show();
            this.$el.find("a").removeClass("active");
            this.$el.find("#" + tabName+ "-tab").addClass("active");
        }
    });

    // отображение числа учеников онлайн
    DashboardApp.OnlineStudentStatView = Backbone.View.extend({
        tagName: "h4",
        onlineTemplate: _.template($("#online_student_stat_template").html()),
        initialize: function() {
            this.listenTo(this.model, "change", this.render);
            this.render();
        },
        render: function(){
            $(this.el).html(this.onlineTemplate(this.model.toJSON()));
            return this;
        }
    });

    // Router
    DashboardApp.Router = Backbone.Router.extend({
        routes:{
            "": "home",
            ":id": "showLesson",
            ":id/problems": "showProblems",
            ":id/results": "showResults",
            ":id/groups": "showGroups"
        },
        // общая функция инициализации приложения
        init: function(clessonId) {
            if (clessonId) {
                if (!DashboardApp.lessonView) {
                    var clesson = new DashboardApp.CLesson({id: clessonId});
                    DashboardApp.lessonView = new DashboardApp.LessonView({
                        model: clesson,
                        el: "#lesson-stat-container"
                    })
                    clesson.fetch({data: {expand_problems: 1}});
                } else if (DashboardApp.lessonView.model.get("id") != clessonId){
                    DashboardApp.lessonView.model.set("id", clessonId);
                    DashboardApp.lessonView.model.fetch({data: {expand_problems: 1}});
                }
                $("#lesson-stat-container").show();
            } else {
                $("#lesson-stat-container").hide();
            }
        },

        // страница выбора занятия
        home: function(){
            this.init();
        },

        // навигация на страницу задач, для поддержки всех урлов
        showLesson: function(clessonId) {
            DashboardApp.router.navigate(clessonId + "/problems", {trigger: true});
        },

        // страница задач
        showProblems: function(clessonId) {
            this.init(clessonId);
            DashboardApp.lessonView.setTab("problems");
        },

        // страница результатов по занятию
        showResults: function(clessonId) {
            this.init(clessonId);
            DashboardApp.lessonView.setTab("results");
        },

        // страница групп
        showGroups: function(clessonId) {
            this.init(clessonId);
            DashboardApp.lessonView.setTab("groups");
        }
    });

    DashboardApp.start = function start(authToken, frontendUrl, originalId) {
        $.ajaxSetup({
            headers: {"Authorization": "Bearer " + authToken}
        });
        DashboardApp.data.frontendUrl = frontendUrl;
        DashboardApp.data.courseId = originalId;
        var course = new DashboardApp.Course({id: originalId });
        new DashboardApp.CourseView({
            el: "#course-select-container",
            model: course
        });
        course.fetch({
            data: {expand_lessons: 1}
        });

        DashboardApp.router = new DashboardApp.Router();
        Backbone.history.start();
    }
});
