import traceback
from builtins import range, str
from datetime import timedelta

from celery.utils.log import get_task_logger
from past.utils import old_div

from django.apps import apps
from django.conf import settings
from django.db import transaction
from django.db.models import F, Q, Sum
from django.utils import timezone

from kelvin.accounts.models import User
from kelvin.accounts.utils import (
    get_user_projects, put_rules_to_user_rules_cache, put_user_to_course_rules_cache,
    remove_course_from_user_courses_cache, remove_rule_from_user_rules_cache, remove_user_from_course_users_cache,
    remove_user_from_rule_users_cache, user_matched_to_course_by_single_rule,
)
from kelvin.celery import app
from kelvin.common.decorators.celery_throttler import throttled_task
from kelvin.common.decorators.task_logger import logged_task
from kelvin.common.revisor import client as revisor
from kelvin.common.sender import SenderHTTPError
from kelvin.courses.mails import MandatoryCourseEmail
from kelvin.lessons.models import LessonProblemLink

from .models import CourseStudent, PeriodicRoleDigest
from .models.criterion import Criterion
from .services import (
    apply_criterion, close_student_notification_tracker_issues,
    delete_pending_periodic_student_notifications_for_student, periodic_notify_student,
    prepare_periodic_course_notifications, process_periodic_role_digest, process_role_digest,
    process_role_digest_on_complete_course_student,
)

logger = get_task_logger(__name__)

MINUTE = 60
NUM_BATCHES_MANDATORY = settings.COURSES_USERS_BATCH_NUM_FOR_MANDATORY_RULE
NUM_BATCHES_OPTIONAL = settings.COURSES_USERS_BATCH_NUM_FOR_OPTIONAL_RULE


@logged_task
@app.task()
def add_user_to_revisor_group(course_permission_id):
    course_permission_model = apps.get_model('courses', 'CoursePermission')
    course_permission = course_permission_model.objects.get(id=course_permission_id)
    if course_permission.can_review():
        review_lesson_problem_links = LessonProblemLink.objects.filter(
            lesson_id__in=course_permission.course.lessons.values_list('id', flat=True),
            problem__type='review',
        ).select_related('problem')

        for lesson_problem_link in review_lesson_problem_links:
            revisor_group_id = lesson_problem_link.problem.external_data.get('revisor_group_id')

            if not revisor_group_id:
                logger.error('Review problem has not is not synced to revisor')
                continue

            revisor_user_id = revisor.create_user(
                username=course_permission.user.username,
                data={
                    'kelvin_user_id': course_permission.user.id
                }
            ).get("id", None)
            logger.info('Created revisor user {}/{} for kelvin user {}'.format(
                revisor_user_id,
                course_permission.user.username,
                course_permission.user.id
            ))
            revisor.create_group_membership(
                revisor_group_id=revisor_group_id,
                revisor_user_id=revisor_user_id,
            )
            logger.info('Added revisor user {} to revisor group {}'.format(
                revisor_user_id,
                revisor_group_id
            ))


def __match_users_with_assignment_rule(users, assignment_rule, assign=True):
    """
    Для каждого из переданных пользователей проверяем, матчится ли он с переданным правилом назначения.
    Добавляем курс пользователю ( предполагается, что вызывающий код проверил, возможно ли такое добавление. Например,
    проверил, обязательное ли правило назначения )
    Заодно мы безусловно актуализируем кеш "пользователь-курсы" и "курс-пользователи", чтобы несколько ускорить работу
    "Библиотеки" в API
    """
    course_student_model = apps.get_model('courses', 'CourseStudent')
    periodic_course_model = apps.get_model('courses', 'PeriodicCourse')

    periodic_course = periodic_course_model.objects.filter(
        course_id=assignment_rule.course_id,
    ).first()
    previous_course_ids = set(periodic_course.previous.values_list('id', flat=True)) if periodic_course is not None else set()

    initial_assign = assign

    for user in users:
        assign = initial_assign
        matched = user_matched_to_course_by_single_rule(assignment_rule, user)
        logger.debug("Matching result for user {} by rule {} of course {}: {}".format(
            user.username,
            assignment_rule.id,
            assignment_rule.course_id,
            matched
        ))
        if matched:
            # put_user_to_course_users_cache(user_id=user.id, course_id=assignment_rule.course_id)
            # put_course_to_user_courses_cache(user_id=user.id, course_id=assignment_rule.course_id)
            put_user_to_course_rules_cache(user=user, assignment_rule=assignment_rule)
            put_rules_to_user_rules_cache(user=user, assignment_rules=[assignment_rule])

            # проверяем периодические курсы
            if periodic_course is not None:
                # проверяем прохождения предыдущих курсов,
                # если курс пройдет недавно, то на новый курс не зачисляем

                recent_completions = course_student_model.objects.filter(
                    course_id__in=previous_course_ids,
                    student_id=user.id,
                    completed=True,
                    date_completed__gt=timezone.now().date() - timedelta(days=periodic_course.period),
                )
                if recent_completions.exists():
                    assign = False

            if assign:
                course_student_model.objects.update_or_create(
                    course_id=assignment_rule.course_id,
                    student_id=user.id,
                    defaults={
                        'assignment_rule': assignment_rule,
                        'deleted': False
                    }
                )
                logger.debug("Added/actualized CourseStudent for user {} course {} by rule {}".format(
                    user.username,
                    assignment_rule.course_id,
                    assignment_rule.id
                ))
                try:
                    if not settings.SENDER_EMAIL_ONLY or user.username in settings.SENDER_EMAIL_ONLY:
                        MandatoryCourseEmail(user, assignment_rule.course).send()
                except SenderHTTPError as e:
                    logger.error("Cannot send a letter via sender: {}".format(str(e)))

        else:
            remove_user_from_course_users_cache(user_id=user.id, course_id=assignment_rule.course_id)
            remove_course_from_user_courses_cache(user_id=user.id, course_id=assignment_rule.course_id)

            remove_user_from_rule_users_cache(user_id=user.id, assignment_rule_id=assignment_rule.id)
            remove_rule_from_user_rules_cache(user_id=user.id, assignment_rule_id=assignment_rule.id)


@logged_task
@app.task(soft_time_limit=MINUTE * 30, time_limit=MINUTE * 40)
def add_course_to_every_student_by_assignment_rule_chunked(assignment_rule_id, chunked_user_ids):
    """
        По переданному правилу назначения пытается найти всех пользователей яанка,
        которые соответствуют этому правилу и добавляет по-необходимости связь с курсом этого правила,
        линкуя эту связь с переданным правилом назначения
    """
    assignment_rule_model = apps.get_model('courses', 'AssignmentRule')
    rule = assignment_rule_model.objects.get(id=assignment_rule_id)
    chunked_users = User.objects.filter(id__in=chunked_user_ids)
    __match_users_with_assignment_rule(chunked_users, rule)


@logged_task
@app.task(soft_time_limit=MINUTE * 30, time_limit=MINUTE * 40)
def add_course_to_every_student_by_assignment_rule(assignment_rule_id):
    """
        Выгребаем всех пользователей проекта, бьем на чанки и передаем эти чанки в дочернюю таску
    """
    assignment_rule_model = apps.get_model('courses', 'AssignmentRule')
    rule = assignment_rule_model.objects.get(id=assignment_rule_id)
    current_project = rule.course.project
    project_user_ids = User.objects.filter(userproject__project=current_project).values_list('id', flat=True)
    for chunk_no in range(NUM_BATCHES_MANDATORY):
        chunked_user_ids = [x for x in project_user_ids if x % NUM_BATCHES_MANDATORY == chunk_no]
        logger.info("Processing chunk {} of size {}".format(chunk_no, len(chunked_user_ids)))
        add_course_to_every_student_by_assignment_rule_chunked.delay(assignment_rule_id, chunked_user_ids)


@logged_task
@throttled_task(ack_late=True)
def add_mandatory_courses_to_students_chunk(chunk_no):
    """
    :param chnuk_no -
        Номер чанка пользователей, рассчитывается как user_id MOD NUM_BATCHES_MANDATORY
        Выбирает из базы пользователей, попадающих в chunk_no и добавляет пользователю курс в мо
    """
    logger.info('Processing users chunk: {}'.format(chunk_no))

    assignment_rule_model = apps.get_model('courses', 'AssignmentRule')
    rules = assignment_rule_model.objects.filter(mandatory=True)

    project_users_map = {
        # отображение из идентификатора проекта в пользователей этого проекта
        # нужно, чтобы для одного проекта вытаскивать его пользователей из базы только один раз
        # TODO: подумать о том, чтобы переделать на централизованный кеш project->users , например,  в redis
    }

    chunk_users_count = 0
    for rule in rules:
        current_project = rule.course.project
        if not project_users_map.get(current_project.id, None):
            project_users_map[current_project.id] = User.objects.filter(
                userproject__project=current_project,
                is_dismissed=False,
            ).annotate(id_mod_chunk_no=F('id') % NUM_BATCHES_MANDATORY).filter(id_mod_chunk_no=chunk_no)
            chunk_users_count += project_users_map[current_project.id].count()

        __match_users_with_assignment_rule(project_users_map[current_project.id], rule)

    logger.info("{} users processed in chunk_no {}".format(chunk_users_count, chunk_no))


@logged_task
@app.task(soft_time_limit=MINUTE * 45, time_limit=MINUTE * 60)
def match_optional_courses_to_students_chunk(chunk_no):
    """
    Берем все небобязательные правила и пересчитываем матчинг кеши
    :param chunk_no -
        Номер чанка пользователей, рассчитывается как user_id MOD NUM_BATCHES_OPTIONAL
        Выбирает из базы пользователей, попадающих в chunk_no и добавляет пользователю курс в мо
    """
    logger.info('Processing users chunk: {}'.format(chunk_no))

    assignment_rule_model = apps.get_model('courses', 'AssignmentRule')
    rules = assignment_rule_model.objects.filter(mandatory=False)

    project_users_map = {
        # отображение из идентификатора проекта в пользователей этого проекта
        # нужно, чтобы для одного проекта вытаскивать его пользователей из базы только один раз
        # TODO: подумать о том, чтобы переделать на централизованный кеш project->users , например,  в redis
    }

    chunk_users_count = 0
    for rule in rules:
        current_project = rule.course.project
        if not project_users_map.get(current_project.id, None):
            project_users_map[current_project.id] = User.objects.filter(
                userproject__project=current_project,
            ).annotate(id_mod_chunk_no=F('id') % NUM_BATCHES_OPTIONAL).filter(id_mod_chunk_no=chunk_no)
            chunk_users_count += project_users_map[current_project.id].count()

        __match_users_with_assignment_rule(project_users_map[current_project.id], rule, assign=False)

    logger.info("{} users processed in chunk_no {}".format(chunk_users_count, chunk_no))


@logged_task
@throttled_task(ack_late=True)
def add_mandatory_courses_to_students():
    """
        Запускает отдельные таски на добавление обязательных курсов для каждого чанка
    """
    for chunk_no in range(NUM_BATCHES_MANDATORY):
        add_mandatory_courses_to_students_chunk.delay(chunk_no)


@logged_task
@app.task(soft_time_limit=MINUTE * 45, time_limit=MINUTE * 60)
def refine_match_cache():
    """
        Запускает отдельные таски на добавление обязательных курсов для каждого чанка
    """
    for chunk_no in range(NUM_BATCHES_OPTIONAL):
        match_optional_courses_to_students_chunk.delay(chunk_no)


@logged_task
@app.task(soft_time_limit=MINUTE * 10, time_limit=MINUTE * 12)
def add_mandatory_courses_to_single_student(user_id):
    """
        Для переданного пользователя перебирает все обязательные правила назначения из проектов пользователя,
        и если пользователь матчится с очередным правилом - добавляет пользователю соответствующий курс в "мои"
    """
    user_model = apps.get_model('accounts', 'User')
    user = user_model.objects.get(id=user_id)
    user_projects = get_user_projects(user)
    for project in user_projects:
        assignment_rule_model = apps.get_model('courses', 'AssignmentRule')
        rules = assignment_rule_model.objects.filter(
            mandatory=True,
            course__project=project
        )
        for rule in rules:
            __match_users_with_assignment_rule([user], rule)


@logged_task
@app.task
def recalculate_feedback_score():
    Course = apps.get_model('courses', 'Course')
    CourseFeedback = apps.get_model('courses', 'CourseFeedback')

    for course in Course.objects.all():
        with transaction.atomic():
            score_count = CourseFeedback.objects.filter(course=course).count()
            score_sum = CourseFeedback.objects.filter(course=course).aggregate(sum=Sum('score')).get('sum') or 0
            course.score_count = score_count
            course.average_score = old_div(float(score_sum), score_count) if score_count else 0
            course.save()


@logged_task
@app.task(bind=True, max_retries=10)
def apply_criterion_task(self, criterion_id: int, force: bool):
    try:
        criterion = Criterion.objects.get(id=criterion_id)
        apply_criterion(criterion=criterion, force=force)
    except Exception as exc:
        self.retry(exc=exc)


@logged_task
@app.task
def process_periodic_notifications():
    prepare_periodic_course_notifications()


@logged_task
@app.task
def send_periodic_notifications():
    PeriodicStudentNotification = apps.get_model('courses', 'PeriodicStudentNotification')
    queryset = PeriodicStudentNotification.objects.filter(
        status=PeriodicStudentNotification.STATUS_PENDING,
    ).values_list('pk', flat=True)
    for pk in queryset:
        with transaction.atomic():
            student_notification = (
                PeriodicStudentNotification.objects
                .select_related(
                    'notification',
                    'student',
                    'student__student',
                )
                .select_for_update()
                .get(pk=pk)
            )
            try:
                periodic_notify_student(student_notification)
                student_notification.status = PeriodicStudentNotification.STATUS_SENT
            except Exception as exc:
                student_notification.status = PeriodicStudentNotification.STATUS_ERROR
                student_notification.errors = traceback.format_exc()
            finally:
                student_notification.save()


@logged_task
@app.task
def process_role_digest_task(role_digest_id: int, force=False):
    RoleDigest = apps.get_model('courses', 'RoleDigest')
    with transaction.atomic():
        role_digest = RoleDigest.objects.select_for_update().get(id=role_digest_id)
        if role_digest.process_status != role_digest.PROCESS_STATUS_PENDING and not force:
            logger.warning("RoleDigest with id=%s already processed" % role_digest_id)
            return
        try:
            process_role_digest(role_digest=role_digest)
            role_digest.process_status = role_digest.PROCESS_STATUS_DONE
        except Exception:
            role_digest.process_status = role_digest.PROCESS_STATUS_ERROR
            role_digest.errors = traceback.format_exc()
        finally:
            role_digest.save()


@logged_task
@app.task
def process_role_digest_periodic_task():
    RoleDigest = apps.get_model('courses', 'RoleDigest')
    role_digests = RoleDigest.objects.filter(process_status=RoleDigest.PROCESS_STATUS_PENDING).only('id')
    for role_digest in role_digests:
        process_role_digest_task.delay(role_digest_id=role_digest.id)


@logged_task
@app.task
def process_periodic_role_digests_task(periodic_role_digest_id: int):
    periodic_role_digest = PeriodicRoleDigest.objects.get(id=periodic_role_digest_id)
    process_periodic_role_digest(periodic_role_digest=periodic_role_digest)


@logged_task
@app.task
def process_all_periodic_role_digests():
    periodic_role_digests = PeriodicRoleDigest.objects.only('id')
    for periodic_role_digest in periodic_role_digests:
        process_periodic_role_digests_task(periodic_role_digest_id=periodic_role_digest.id)


@logged_task
@app.task
def process_role_digest_on_complete_course_student_task(course_student_id: int):
    course_student = CourseStudent.objects.get(id=course_student_id)
    process_role_digest_on_complete_course_student(course_student=course_student)


@logged_task
@app.task
def close_role_digest_task(periodic_role_digest_id: int):
    periodic_role_digest = PeriodicRoleDigest.objects.select_related('periodic_course').get(id=periodic_role_digest_id)
    for course_student in (
        CourseStudent.objects.filter(completed=True, course_id=periodic_role_digest.periodic_course.course_id)
    ):
        process_role_digest_on_complete_course_student(
            course_student=course_student, for_periodic_role_digest=periodic_role_digest,
        )


@logged_task
@app.task
def stop_course_student_notifications_task(course_student_id: int):
    delete_pending_periodic_student_notifications_for_student(course_student_id=course_student_id)
    close_student_notification_tracker_issues(course_student_id=course_student_id)
