from collections import defaultdict

from django.conf import settings
from kelvin.common.staff_reader import staff_reader
from kelvin.common.startrek.settings import STARTREK_ROBOT

from ..models.periodic import PeriodicStudentNotification
from .base import TrackerTemplateBase


class NotificationTrackerTemplateBase(TrackerTemplateBase):
    def __init__(self, student_notification: PeriodicStudentNotification):
        self.student_notification = student_notification
        super().__init__()

    def get_ctx(self) -> dict:
        return {
            'student_notification': self.student_notification,
        }


class TestEnrollmentStudentNotification(NotificationTrackerTemplateBase):
    summary_template = "notification/test_enrollment/summary.tmpl"
    description_template = "notification/test_enrollment/chief.tmpl"
    open_comment_template = "notification/test_enrollment/open_comment.tmpl"
    close_comment_template = "notification/test_enrollment/close_comment.tmpl"

    def get_fields(self) -> dict:
        ctx = self.context
        return {
            "author": ctx["student_notification"].student.student.username,
            "graphTaskId": ctx["student_notification"].id,
        }


class LocalEnrollmentStudentNotification(NotificationTrackerTemplateBase):
    summary_template = "notification/local_enrollment/summary.tmpl"
    description_template = "notification/local_enrollment/chief.tmpl"
    open_comment_template = "notification/local_enrollment/open_comment.tmpl"
    close_comment_template = "notification/local_enrollment/close_comment.tmpl"

    def get_fields(self) -> dict:
        ctx = self.context
        return {
            "author": ctx["student_notification"].student.student.username,
            "runId": ctx["student_notification"].id,
        }


class CorporateEthicsStudentNotification(NotificationTrackerTemplateBase):
    summary_template = "notification/corporate_ethics/summary.tmpl"
    description_template = "notification/corporate_ethics/description.tmpl"
    open_comment_template = "notification/corporate_ethics/open_comment.tmpl"
    close_comment_template = "notification/corporate_ethics/close_comment.tmpl"

    def get_fields(self) -> dict:
        ctx = self.context
        username = ctx["student_notification"].student.student.username
        staff_user_info = staff_reader.get_suggestuser(username=username)

        return {
            "createdBy": STARTREK_ROBOT,
            "employee": username,
            "assignee": username,
            "followers": settings.ETHICS_FOLLOWERS + [staff_user_info['chief']['login']],
            "hrbp": [hr_partner['login'] for hr_partner in staff_user_info['hr_partners']],
            "tags": ["Сотрудник"],
            "department": ' / '.join(
                (
                    ancestors['department']['name']['full']['ru']
                    for ancestors
                    in staff_user_info['department_group']['ancestors'] + [staff_user_info['department_group']]
                )
            )
        }


QUEUE_TEMPLATE_MAP = defaultdict(
    lambda: LocalEnrollmentStudentNotification,
    {
        "TESTENROLLMENT": TestEnrollmentStudentNotification,
        "LOCALENROLLMENT": LocalEnrollmentStudentNotification,
        "CORPORATEETHICS": CorporateEthicsStudentNotification,
    }
)


class TrackerNotificationIssue:
    def __init__(self, student_notification: PeriodicStudentNotification):
        self.student_notification = student_notification

    def get_template(self):
        queue = self.student_notification.notification.parameters['queue']
        return QUEUE_TEMPLATE_MAP[queue](student_notification=self.student_notification)

    def get_open_comment(self) -> str:
        return self.get_template().get_open_comment()

    def get_close_comment(self) -> str:
        return self.get_template().get_close_comment()

    def as_dict(self):
        queue = self.student_notification.notification.parameters['queue']
        issue_type = self.student_notification.notification.parameters['issue_type']
        template = self.get_template()

        return {
            **{
                'queue': queue,
                'type': {'name': issue_type},
            },
            **template.as_dict(),
        }
