from collections import defaultdict

from django.template import loader

from kelvin.common.staff_reader import staff_reader
from kelvin.common.startrek.settings import STARTREK_ROBOT

from ..models.periodic import RoleDigest
from .base import TrackerTemplateBase


class RoleDigestTrackerTemplateBase(TrackerTemplateBase):
    def __init__(self, role_digest: RoleDigest):
        self.role_digest = role_digest
        super().__init__()

    def get_ctx(self) -> dict:
        return {
            'role_digest': self.role_digest,
        }


class TestEnrollmentRoleDigestTrackerTemplate(RoleDigestTrackerTemplateBase):
    summary_template = "role_digest/test_enrollment/summary.tmpl"
    description_template = "role_digest/test_enrollment/description.tmpl"
    open_comment_template = "role_digest/test_enrollment/open_comment.tmpl"
    close_comment_template = "role_digest/test_enrollment/close_comment.tmpl"

    def get_fields(self) -> dict:
        ctx = self.context
        return {
            "author": ctx["role_digest"].user.username,
            "graphTaskId": ctx["role_digest"].id,
        }


class LocalEnrollmentRoleDigestTrackerTemplate(RoleDigestTrackerTemplateBase):
    summary_template = "role_digest/local_enrollment/summary.tmpl"
    description_template = "role_digest/local_enrollment/description.tmpl"
    open_comment_template = "role_digest/local_enrollment/open_comment.tmpl"
    close_comment_template = "role_digest/local_enrollment/close_comment.tmpl"

    def get_fields(self) -> dict:
        ctx = self.context
        return {
            "author": ctx["role_digest"].user.username,
            "runId": ctx["role_digest"].id,
        }


class CorporateEthicsRoleDigestTrackerTemplateBase(RoleDigestTrackerTemplateBase):
    summary_template = "role_digest/corporate_ethics/summary.tmpl"
    open_comment_template = "role_digest/corporate_ethics/open_comment.tmpl"
    close_comment_template = "role_digest/corporate_ethics/close_comment.tmpl"

    def get_fields(self) -> dict:
        ctx = self.context
        username = ctx["role_digest"].user.username
        staff_user_info = staff_reader.get_suggestuser(username=username)

        return {
            "createdBy": STARTREK_ROBOT,
            "assignee": username,
            "followers": "shudeyko",
            "department": ' / '.join(
                (
                    ancestors['department']['name']['full']['ru']
                    for ancestors
                    in staff_user_info['department_group']['ancestors'] + [staff_user_info['department_group']]
                )
            )
        }


class CorporateEthicsChiefDigestTrackerTemplate(CorporateEthicsRoleDigestTrackerTemplateBase):
    description_template = "role_digest/corporate_ethics/description/chief.tmpl"

    def get_fields(self) -> dict:
        return {
            **super().get_fields(),
            **{'tags': ['Руководитель']},
        }


class CorporateEthicsHRBPDigestTrackerTemplate(CorporateEthicsRoleDigestTrackerTemplateBase):
    description_template = "role_digest/corporate_ethics/description/hrbp.tmpl"

    def get_fields(self) -> dict:
        return {
            **super().get_fields(),
            **{'tags': ['HRBP']},
        }


TEMPLATE_MAP = defaultdict(
    lambda: LocalEnrollmentRoleDigestTrackerTemplate,
    {
        "TESTENROLLMENT": TestEnrollmentRoleDigestTrackerTemplate,
        "LOCALENROLLMENT": LocalEnrollmentRoleDigestTrackerTemplate,
        "chief": CorporateEthicsChiefDigestTrackerTemplate,
        "hrbp": CorporateEthicsHRBPDigestTrackerTemplate,
    }
)


class TrackerRoleDigestIssue:
    def __init__(self, role_digest: RoleDigest):
        self.role_digest = role_digest

    def get_template(self):
        template_name = self.role_digest.periodic_role_digest.parameters['template']
        return TEMPLATE_MAP[template_name](role_digest=self.role_digest)

    def get_open_comment(self):
        return self.get_template().get_open_comment()

    def get_close_comment(self):
        return self.get_template().get_close_comment()

    def as_dict(self):
        queue = self.role_digest.periodic_role_digest.parameters['queue']
        issue_type = self.role_digest.periodic_role_digest.parameters['issue_type']
        template = self.get_template()

        return {
            **{
                'queue': queue,
                'type': {'name': issue_type},
            },
            **template.as_dict(),
        }
