import re
from builtins import map, object
from datetime import timedelta

import jsonschema

from django.core.exceptions import ValidationError

from kelvin.lessons.validators import AbstractLessonScenarioValidator


class CourseLessonLinkValidator(AbstractLessonScenarioValidator):
    """
    Валидатор курсозанятий
    """
    invalid_duration_message = u'Длительность контрольной слишком велика'
    invalid_date_assignment_draft = (
        u'Планируемая дата выдачи должна быть больше текущей'
    )

    def validate_dates(self):
        """
        Проверяет, что длительность контрольной не больше, чем интервал между
        датой выдачи и датой завершения
        """
        errors = super(CourseLessonLinkValidator, self).validate_dates()
        if self.non_field_errors in errors:
            return errors
        if self.may_have_errors() and self.is_time_invalid():
            self.add_error(self.invalid_duration_message, errors)
        return errors

    def get_assignment_dates(self):
        return (
            self.get('date_assignment'),
        )

    def may_have_errors(self):
        return self.get('finish_date') and self.get('duration')

    def is_time_invalid(self):
        td_duration = timedelta(minutes=self.get('duration'))
        finish_date = self.get('finish_date')
        return any(
            map(
                lambda date_assign: (
                    date_assign and
                    finish_date - date_assign < td_duration
                ),
                self.get_assignment_dates(),
            )
        )

    def add_error(self, error, errors):
        if self.non_field_errors not in errors:
            errors[self.non_field_errors] = []
        errors[self.non_field_errors].append(error)


class ProgressIndicatorPaletteValidator(object):
    SCHEMA = {
        "type": "array",
        "items": {
            "type": "object",
            "properties": {
                "progress": {
                    "type": "integer",
                },
                "color": {
                    "type": "string"
                }
            }
        }
    }

    PALETTE = re.compile('#[0-9a-fA-f]{6}')

    @classmethod
    def validate_json(cls, value):
        try:
            jsonschema.validate(value, cls.SCHEMA)
        except jsonschema.ValidationError as e:
            raise ValidationError(e.message)

    @classmethod
    def validate_values(cls, value):
        points = set()
        zero = False
        hundred = False

        for point in value:
            progress = point["progress"]

            if progress in points:
                raise ValidationError("Duplicates in palette.")
            points.add(progress)

            if progress < 0 or progress > 100:
                raise ValidationError("Point out of progress scope.")

            zero = zero or progress == 0
            hundred = hundred or progress == 100

            if not cls.PALETTE.match(point["color"]):
                raise ValidationError(
                    "{} is not valid palette value".format(point["color"]),
                )

        if not zero:
            raise ValidationError("No zero progress point.")

        if not hundred:
            raise ValidationError("No 100 progress point.")

    @classmethod
    def validate(cls, value):
        cls.validate_json(value)
        cls.validate_values(value)


def validate_progress_indicator_palette(value):
    return ProgressIndicatorPaletteValidator.validate(value)
