import io
from builtins import map, str
from collections import defaultdict

import xlsxwriter
from django_replicated.decorators import use_state
from rest_condition import And, Or

from django.conf import settings
from django.contrib.auth import get_user_model
from django.core.exceptions import ValidationError as DjangoValidationError
from django.db import transaction
from django.db.models import Count, Q
from django.db.models.functions import Length, Lower
from django.db.utils import IntegrityError
from django.http.response import Http404, HttpResponse
from django.shortcuts import get_object_or_404
from django.utils.decorators import method_decorator

from rest_framework import mixins, serializers, status, viewsets
from rest_framework.decorators import detail_route, list_route
from rest_framework.exceptions import APIException, NotFound, PermissionDenied, ValidationError
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.throttling import ScopedRateThrottle
from rest_framework.views import APIView
from rest_framework.viewsets import GenericViewSet

from kelvin.accounts.permissions import (
    IsContentManager, IsCurator, IsParent, IsStaff, IsStaffChief, IsStaffHRBP, IsSuperuser, IsTeacher,
    ObjectForAuthenticated, ObjectForContentManager, ObjectForOwner, ObjectForStaff, ObjectForTeacher,
)
from kelvin.accounts.serializers import UserCourseOwnerSerializer
from kelvin.accounts.utils import UserCourseFilterFactory, get_user_projects, is_student
from kelvin.common.pagination import PaginationWithPageSize
from kelvin.common.permissions import IsDetailRequest
from kelvin.common.staff_reader import StaffReaderError
from kelvin.courses.journal import CourseGroupJournal, StudentJournal
from kelvin.courses.models import (
    AssignmentRule, Course, CourseLessonEdge, CourseLessonGraph, CourseLessonGraphFactory, CourseLessonLink,
    CourseLessonNode, CoursePermission, CourseStudent, Criterion,
)
from kelvin.courses.models.criterion.action import validate_actions
from kelvin.courses.models.criterion.condition import validate_formula
from kelvin.courses.permissions import (
    CanEditCourse, CanManageCourseRoles, CanViewCourse, CanViewCuratorStats, CanViewStudentStats,
    ObjectCourseAssignedToChildren, ObjectCourseAssignedToStudent, ObjectCourseForAnonymous,
)
from kelvin.courses.report import ChiefUsersStaffReport, CuratorReport, HRUsersStaffReport
from kelvin.courses.serializers import (
    AssignmentRuleSerializer, AssignmentRuleTriggerSerializer, CourseEditSerializer, CourseLessonLinkSerializer,
    CoursePermissionSerializer, CourseSerializer, CourseShortSerializer, CourseSimpleSuggestSerializer,
)
from kelvin.courses.services import add_student
from kelvin.courses.views.course.web_view import CourseWebView
from kelvin.lesson_assignments.models import LessonAssignment
from kelvin.result_stats.models import LessonDiagnosticStats, StudentDiagnosticsStat
from kelvin.result_stats.serializers import (
    CourseLessonStatSerializer, StudentCourseStatShortSerializer, StudentDiagnosticsStatSerializers,
)
from kelvin.switcher import get_feature_bool_option as enabled_feature
from kelvin.tags.models import TagTypeAdapter

# fix default django_replicated decorators
use_master = use_state(forced_state='master')

User = get_user_model()


class BaseCourseViewSet(viewsets.ReadOnlyModelViewSet):
    """
    Методы для работы с курсами
    """
    serializer_class = CourseShortSerializer
    serializer_class_detail = CourseSerializer

    permission_classes = [
        IsAuthenticated,
        Or(IsTeacher, IsContentManager, IsStaff, IsParent, IsDetailRequest),
        Or(ObjectForTeacher, ObjectForContentManager, ObjectForStaff,
           ObjectForOwner, ObjectCourseAssignedToStudent,
           ObjectCourseAssignedToChildren),
    ]
    pagination_class = None

    WebView = CourseWebView

    @staticmethod
    def get_control_work_data(clesson_id, results, now):
        return CourseLessonLinkSerializer.control_work_data(clesson_id, results, now)

    def perform_create(self, serializer):
        """
        При создании курса указываем владельца
        """
        serializer.save(owner=self.request.user)

    def get_serializer_class(self, *args, **kwargs):
        if self.action == 'retrieve' or self.action == 'web':
            return self.serializer_class_detail
        return self.serializer_class

    def get_serializer_context(self):
        """
        Добавляем параметры из запроса в контекст сериализатора
        """
        ctx = super(BaseCourseViewSet, self).get_serializer_context()
        flags = [
            'clessons_ordering', 'expand_lessons', 'with_students',
            'with_credentials', 'paginated',
        ]
        for flag in flags:
            ctx[flag] = self.request.query_params.get(flag)

        ctx['for_student'] = is_student(self.request.user)
        return ctx

    @staticmethod
    def get_base_queryset():
        return Course.objects.all()

    @staticmethod
    def add_base_related_and_prefetches(queryset):
        return queryset.select_related(
            'cover',
            'subject',
            'mailing_list',
            'progress_indicator',
        ).prefetch_related(
            'courselessonlink_set',
            'courselessonlink_set__lesson',
            'courselessonlink_set__progress_indicator',
        )

    def modify_queryset_expand_lessons(self, queryset):
        return queryset.prefetch_related(
            'courselessonlink_set__lesson__lessonproblemlink_set',
        )

    def modify_queryset_with_students(self, queryset):
        return queryset.prefetch_related('students')

    def modify_queryset_with_credentials(self, queryset):
        return queryset.select_related('owner__teacher_profile')

    def modify_queryset_with_query_params(self, queryset):
        if self.action != 'stats':
            if self.request.query_params.get('expand_lessons') == 'true':
                queryset = self.modify_queryset_expand_lessons(queryset)
            if self.request.query_params.get('with_students') == 'true':
                queryset = self.modify_queryset_with_students(queryset)
            if self.request.query_params.get('with_credentials') == 'true':
                queryset = self.modify_queryset_with_credentials(queryset)
        return queryset

    def add_annotations(self, queryset):
        if self.action in ['list', 'assigned']:
            queryset = queryset.annotate(
                lessons_count=Count('courselessonlink'),
            )
        return queryset

    def get_diagnostics_queryset(self):
        return self.get_base_queryset()

    def get_list_queryset(self):
        filters = (
            Q(owner=self.request.user) |
            Q(free=True)
        )
        if self.request.user.is_support:
            filters |= (
                Q(available_for_support=True)
            )

        return self.add_base_related_and_prefetches(
            self.get_base_queryset().filter(
                filters,
            ),
        )

    def get_student_stats_queryset(self):
        return self.add_base_related_and_prefetches(
            self.get_base_queryset()
        ).prefetch_related(
            'courselessonlink_set__lesson__lessonproblemlink_set',
        ).select_related('owner')

    def get_find_problems_queryset(self):
        return self.add_base_related_and_prefetches(
            self.get_base_queryset()
        ).prefetch_related(
            'courselessonlink_set__lesson__lessonproblemlink_set',
        )

    def get_stats_queryset(self):
        return self.add_base_related_and_prefetches(
            self.get_base_queryset()
        ).prefetch_related(
            'courselessonlink_set__courselessonstat',
        )

    def get_default_queryset(self):
        return self.add_base_related_and_prefetches(
            self.get_base_queryset()
        )

    def get_action_queryset_builders(self):
        return {
            'list': self.get_list_queryset,
            'student_stats': self.get_student_stats_queryset,
            'find_problems': self.get_find_problems_queryset,
            'stats': self.get_stats_queryset,
        }

    def choose_queryset_builder(self):
        return self.get_action_queryset_builders().get(
            self.action, self.get_default_queryset,
        )

    def get_queryset(self):
        """
        В списке отдаем только курсы, назначенные группам пользователя
        """
        if self.action == 'diagnostics':
            return self.get_diagnostics_queryset()

        query_builder = self.choose_queryset_builder()
        queryset = self.modify_queryset_with_query_params(query_builder())
        queryset = self.add_annotations(queryset)
        return queryset

    def retrieve(self, request, *args, **kwargs):
        """
        Для учеников удаляем занятия, в которых им не назначены задачи
        """
        # переписываем родительский метод
        course = self.get_object()
        serializer = self.get_serializer(course)
        data = serializer.data
        user = request.user

        # удаляем неназначенные занятия, если запрашивает ученик курса
        if CourseStudent.objects.filter(
            course=course,
            student=user,
        ).exists():
            empty_clesson_ids = set(
                LessonAssignment.objects.filter(
                    student=user,
                    clesson__in=course.courselessonlink_set.all(),
                    problems__startswith=[],  # `__exact` не работает
                ).values_list('clesson_id', flat=True)
            )
            if empty_clesson_ids:
                data['lessons'] = [
                    lesson for lesson in data['lessons']
                    if lesson['id'] not in empty_clesson_ids
                ]
        return Response(data)

    @detail_route(permission_classes=[
        Or(ObjectCourseForAnonymous,
           And(ObjectForAuthenticated,
               Or(ObjectForTeacher, ObjectForContentManager, ObjectForStaff,
                  ObjectForOwner, ObjectCourseAssignedToStudent,
                  ObjectCourseAssignedToChildren),
               )
           )
    ])
    def web(self, request, *args, **kwargs):
        return Response(self.WebView(self).build_data())

    def assigned_courses_for(self, user):
        """
        Возвращает назначенные пользователю курсы
        """
        queryset = (
            Course.objects.filter(
                students=user,
            ).select_related('cover')
        )
        serializer = self.get_serializer(queryset, many=True)
        return serializer.data

    @list_route(permission_classes=[IsAuthenticated])
    def assigned(self, request):
        """
        Список назначенных курсов ученику или ребенку родителя
        """
        if 'child' in request.query_params:
            try:
                user = User.objects.get(
                    id=int(request.query_params['child']),
                    parents=request.user.id,
                )
            except (TypeError, ValueError):
                raise ValidationError('wrong child parameter')
            except User.DoesNotExist:
                raise Http404
        else:
            user = request.user
        return Response(self.assigned_courses_for(user))

    @property
    def throttle_scope(self):
        """
        В зависимости от типа запроса, динамически выставляем scope,
        для которого нужно применить throttling
        """
        if self.action == 'join':
            return 'course_join'
        return None

    def get_user_from_child(self):
        if 'child' in self.request.data:
            try:
                user = User.objects.get(
                    id=int(self.request.data['child']),
                    parents=self.request.user.id,
                )
            except (TypeError, ValueError):
                raise ValidationError('Wrong "child" parameter')
            except User.DoesNotExist:
                raise PermissionDenied
        else:
            user = self.request.user
        return user

    def get_codes(self):
        codes = [_f for _f in self.request.data.get('codes', []) if _f]
        if not codes:
            raise ValidationError('Codes not provided')
        return codes

    def get_courses_by_code(self, codes):
        try:
            return Course.objects.filter(code__in=codes)
        except (TypeError, ValueError):
            raise ValidationError('Wrong "codes" parameter type')

    @classmethod
    def check_courses_and_codes(cls, courses, codes):
        if len(courses) != len(codes):
            raise NotFound('Course with provided code(s) not found')

    @list_route(
        methods=['post'],
        permission_classes=[IsAuthenticated],
        throttle_classes=[ScopedRateThrottle],
    )
    def join(self, request):
        """
        По коду курса добавляем пользователя к курсу.

        По умолчанию включаем ограничение на 12 запросов в минуту
        или 1 в 5 секунд (на самом деле нет)
        """
        user = self.get_user_from_child()
        codes = self.get_codes()
        courses = self.get_courses_by_code(codes)
        self.check_courses_and_codes(courses, codes)

        with transaction.atomic():
            for course in courses:
                add_student(course, user)

        return Response(self.assigned_courses_for(user))

    @detail_route(permission_classes=[
        IsAuthenticated,
        Or(IsTeacher, IsParent, IsContentManager, ObjectForContentManager),
    ])
    def student_stats(self, request, pk):
        """
        Статистика ученика по курсу
        """
        # проверка параметра `student`
        try:
            student_id = int(self.request.query_params.get('student'))
        except (ValueError, TypeError):
            raise ValidationError('wrong `student` parameter')

        # проверка, что пользователь родитель или учитель ученика,
        # статистика которого запрашивается
        user_is_parent_qs = User.objects.filter(
            id=self.request.user.id,
            parent_profile__children__id=student_id,
        )
        user_is_teacher_qs = CourseStudent.objects.filter(
            student=student_id,
            course__owner=self.request.user,
        )
        if not (self.request.user.is_parent and user_is_parent_qs.exists() or
                self.request.user.is_teacher and user_is_teacher_qs.exists() or
                self.request.user.is_content_manager):
            raise Http404

        course = self.get_object()
        # формирование ответа
        journal = StudentJournal(student_id, course)
        return Response({
            'success_percent': journal.get_success_percent(),
            'journal': journal.data,
            'course_owner': UserCourseOwnerSerializer(instance=course.owner).data,
        })

    @detail_route(methods=['post'],
                  # TODO контент-менеджеру надо давать менять чужой курс?
                  permission_classes=[IsTeacher, ObjectForOwner])
    def add_clesson(self, request, pk):
        """
        Добавление занятия к курсу
        """
        if 'id' not in request.data:
            raise ValidationError('`id` field required')
        try:
            clesson = get_object_or_404(
                CourseLessonLink,
                id=request.data['id'],
            )
        except (ValueError, TypeError):
            raise ValidationError('wrong `id` value')

        if not clesson.accessible_to_teacher:
            raise PermissionDenied('lesson not accessible')

        course = self.get_object()
        course.add_clesson(clesson)

        return Response()

    @detail_route(permission_classes=[
        Or(
            And(ObjectForTeacher, ObjectForOwner),
            ObjectForContentManager,
        ),
    ])
    def stats(self, request, pk):
        """
        Возвращает статистику группы по всем занятиям курса
        """
        course = self.get_object()
        clessons = list(course.courselessonlink_set.all())
        clesson_stat_serializer = CourseLessonStatSerializer()
        student_stat_serializer = StudentCourseStatShortSerializer(
            context={
                'order': [clesson.id for clesson in clessons],
            }
        )
        students_count = CourseStudent.objects.filter(
            course__id=pk,
        ).count()

        student_stats = course.studentcoursestat_set.order_by('student_id')
        students_limit = self.request.query_params.get("students_limit", None)
        if students_limit:
            try:
                student_stats = student_stats[:int(students_limit)]
            except (TypeError, ValueError):
                pass

        return Response(
            {
                'lessons': {
                    clesson.id: clesson_stat_serializer.to_representation(
                        clesson.courselessonstat)
                    for clesson in clessons
                    if (
                            hasattr(clesson, 'courselessonstat') and (
                            clesson.mode != CourseLessonLink.CONTROL_WORK_MODE or clesson.can_view_results
                    )
                    )
                },
                'students': {
                    student_stat.student_id:
                        student_stat_serializer.to_representation(student_stat)
                    for student_stat in student_stats
                },
                'students_count': students_count,
            }
        )

    @detail_route(permission_classes=[
        Or(
            And(ObjectForTeacher, ObjectForOwner),
            ObjectForContentManager,
        ),
    ])
    def journal(self, request, pk):
        """
        Возвращает журнал группы по курсу
        """
        students = CourseStudent.objects.filter(
            course__id=pk,
        ).count()

        show_journal = students <= settings.MAX_COURSE_JOURNAL_STUDENTS
        show_csv = students >= settings.MIN_COURSE_CSV_STUDENTS
        csv_url = None
        data = None
        if show_journal:
            data = CourseGroupJournal(self.get_object()).data
        if show_csv:
            journal = Course.objects.get(pk=pk).journal
            csv_url = journal.url if journal else None

        return Response({"data": data, "csv_url": csv_url, 'students_count': students})

    @detail_route(permission_classes=[
        Or(IsTeacher, IsContentManager),
    ])
    def find_problems(self, request, pk):
        """
        Находит в каких занятиях встречаются задачи
        """
        try:
            problem_ids = set(map(
                int, self.request.query_params.get('problems', '').split(',')))
        except (ValueError, TypeError):
            return Response('wrong problem ids', status=status.HTTP_400_BAD_REQUEST)

        data = defaultdict(list)
        course = self.get_object()
        for clesson in course.courselessonlink_set.all():
            clesson_problems = set(
                link.problem_id for link in
                clesson.lesson.lessonproblemlink_set.all()
            )
            for problem_id in (clesson_problems & problem_ids):
                data[problem_id].append({
                    'id': clesson.id,
                    'name': clesson.lesson.name
                })

        return Response(data)


def as_xlsx(table):
    buffer = io.BytesIO()
    workbook = xlsxwriter.Workbook(buffer)
    worksheet = workbook.add_worksheet()
    worksheet.set_column(0, len(table[0]) if len(table) else 0, 40)
    for i, row in enumerate(table):
        worksheet.write_row(i, 0, [str(item) for item in row])
    workbook.close()
    buffer.seek(0)

    response = HttpResponse(
        buffer,
        content_type='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    )
    response['Content-Disposition'] = 'attachment; filename=report.xlsx'

    return response


def get_edge_response(edge):
    return {
        "id": edge.pk,
        "child_clesson_id": edge.child_clesson_id,
        "parrent_clesson_id": edge.parent_clesson_id,
        "criterion": {
            "id": edge.criterion.pk,
            "formula": edge.criterion.formula,
            "actions": edge.criterion.actions,
            "assignment_rule_id": edge.criterion.assignment_rule_id
        }
    }


class AssignmentRuleEdgesViewSet(APIView):
    def get(self, request, **kwargs):
        rule_id = self.request.parser_context['kwargs'].get('rule_id', None)
        edges = CourseLessonEdge.objects.filter(criterion__assignment_rule_id=rule_id)
        response = []
        for edge in edges:
            response.append(get_edge_response(edge))
        return Response({"data": response})

    @transaction.atomic
    def post(self, request, **kwargs):
        data = request.data.get('data', None)
        rule_id = self.request.parser_context['kwargs'].get('rule_id', None)

        edges = []
        for edge in data:
            criterion, _ = Criterion.objects.update_or_create(
                id=edge['criterion'].get('id', None),
                defaults={
                    "formula": edge['criterion']['formula'],
                    "clesson_id": None,
                    "formula": edge['criterion']['formula'],
                    "actions": edge['criterion']['actions'],
                    "name": '',
                    "priority": 0,
                    "assignment_rule_id": rule_id,
                }
            )
            edges.append((edge['parent_clesson_id'], edge['child_clesson_id'], criterion.pk))

        graph = CourseLessonGraph(assignment_rule_id=rule_id, edges=edges)

        for edge in data:
            graph.set_clesson_available(edge['child_clesson_id'], False)

        CourseLessonGraphFactory.serialize(graph)

        edges = CourseLessonEdge.objects.filter(criterion__assignment_rule_id=rule_id)
        response = []
        for edge in edges:
            response.append(get_edge_response(edge))

        return Response({"data": response})

    @transaction.atomic
    def delete(self, request, **kwargs):
        rule_id = self.request.parser_context['kwargs'].get('rule_id', None)
        edges = CourseLessonEdge.objects.filter(assignment_rule_id=rule_id)

        for edge in edges:
            Criterion.objects.get(id=edge.criterion.id).delete()
            edge.delete()

        CourseLessonNode.objects.filter(assignment_rule_id=rule_id).delete()

        return Response(data={"data": []}, status=status.HTTP_200_OK)


class AssignmentRuleEdgeViewSet(APIView):
    def get(self, request, **kwargs):
        edge_id = self.request.parser_context['kwargs'].get('edge_id', None)
        edge = CourseLessonEdge.objects.get(id=edge_id)

        return Response({"data": get_edge_response(edge)})

    @transaction.atomic
    def post(self, request, **kwargs):
        edge_id = self.request.parser_context['kwargs'].get('edge_id', None)
        edge = CourseLessonEdge.objects.get(id=edge_id)
        requestCriterion = request.data.get('criterion', None)
        Criterion.objects.filter(id=edge.criterion.id).update(formula=requestCriterion['formula'])

        edge = CourseLessonEdge.objects.get(id=edge_id)
        response = get_edge_response(edge)
        return Response({"data": response})


class AssignmentRuleTriggersViewSet(viewsets.ModelViewSet, GenericViewSet):
    serializer_class = AssignmentRuleTriggerSerializer
    permission_classes = [CanEditCourse]

    def get_queryset(self):
        try:
            rule_id = self.request.parser_context['kwargs']['rule_id']
        except Exception as e:
            raise ValidationError(
                "Cannot do a thing with AssignmentRulesTriggers without valid rule_id: {}".format(str(e))
            )

        return Criterion.objects.filter(assignment_rule_id=rule_id)

    def list(self, request, **kwargs):
        serializer = self.serializer_class(self.get_queryset(), many=True)
        return Response(serializer.data)

    def create(self, request, **kwargs):
        rule_id = self.request.parser_context['kwargs'].get('rule_id', None)

        if rule_id is None:
            raise ValueError("Cannot save trigger without AssignmentRule id")

        serializer = self.get_serializer_class()(data=self.request.data)
        serializer.is_valid(raise_exception=True)
        try:
            formula = serializer.validated_data['formula']
            validate_formula(formula)

            actions = serializer.validated_data['actions']
            validate_actions(actions)
        except DjangoValidationError as exc:
            raise ValidationError(str(exc))

        criterion = Criterion.objects.create(assignment_rule_id=rule_id, **serializer.validated_data)

        return Response(
            self.serializer_class(criterion).data,
            status=status.HTTP_201_CREATED
        )

    def update(self, request, **kwargs):
        rule_id = self.request.parser_context['kwargs']['rule_id']
        criterion_id = self.request.parser_context['kwargs']['pk']

        assignment_rule = get_object_or_404(AssignmentRule, pk=rule_id)

        serializer = self.get_serializer_class()(data=self.request.data)
        serializer.is_valid(raise_exception=True)

        try:
            formula = serializer.validated_data['formula']
            validate_formula(formula)

            actions = serializer.validated_data['actions']
            validate_actions(actions)
        except DjangoValidationError as exc:
            raise ValidationError(str(exc))

        criterion = Criterion.objects.get(pk=criterion_id)

        for k, v in serializer.validated_data.items():
            setattr(criterion, k, v)
        criterion.assignment_rule_id = rule_id

        # явно вызываем save на случай, если нам понадобится обработать сохранени в модели или сигналах
        criterion.save()

        return Response(
            self.serializer_class(criterion).data,
            status=status.HTTP_200_OK
        )


class AssignmentRuleViewSet(viewsets.ModelViewSet, GenericViewSet):
    serializer_class = AssignmentRuleSerializer
    permission_classes = [CanEditCourse]

    def validate_formula(self, value):
        """
        1. Формула не должна быть пустой.
        2. Формула должна быть массивом массивов.
        3. Во внутренних массивах должны быть теги одного типа.
        4. В каждом предикате должно выполняться условие is_valid каждого типа тегов.
        5. В каждом предикате должна использоваться только разрешенная в данном типе операция.
        """
        if not value:
            raise ValidationError("Formula should be non-empty")
        if not isinstance(value, list):
            raise ValidationError("Formula should be a list")
        for conjunction in value:
            if not conjunction:
                raise ValidationError("Formula should consist of non empty conjuncions")
            if not isinstance(conjunction, list):
                raise ValidationError("Formula should be a list of lists")
            types = set([])
            for predicate in conjunction:
                if not predicate:
                    raise ValidationError("Formula predicates should be non empty")
                if not isinstance(predicate, dict):
                    raise ValidationError("Formula predicate should be a dict")

                keys = list(predicate.keys())
                if not all(key in keys for key in ["semantic_type", "value", "operation"]):
                    raise ValidationError("Formula predicate should consist of mandatory keys: value, operation, "
                                          "semantic_type")
                semantic_type = TagTypeAdapter.get_tag_type_for_semantic_type(predicate["semantic_type"])
                if semantic_type is None:
                    raise ValidationError("Unknown semantic_type value: '{}'".format(predicate["semantic_type"]))
                if not types:
                    types.add(semantic_type)
                if semantic_type not in types:
                    raise ValidationError("Predicate group should consist of the same types")
                if not semantic_type.is_valid():
                    raise ValidationError("Predicate semantic type has invalid format")
        return value

    def validate_title(self, value):
        if not value:
            raise ValidationError("Title should be non empty")

    def get_queryset(self):
        try:
            course_id = self.request.parser_context['kwargs']['course_id']
        except Exception as e:
            raise ValidationError(
                "Cannot do a thing with AssignmentRules without valid course_id: {}".format(str(e))
            )

        # further db-request is needed to be able to return 404 for unknown course
        course = get_object_or_404(Course, pk=course_id)
        self.check_object_permissions(request=self.request, obj=course)
        return AssignmentRule.objects.filter(course=course)

    def retrieve(self, request, **kwargs):
        course_id = self.request.parser_context['kwargs']['course_id']
        rule_id = self.request.parser_context['kwargs']['pk']
        course = get_object_or_404(Course, pk=course_id)
        self.check_object_permissions(request=self.request, obj=course)
        rule = get_object_or_404(AssignmentRule, pk=rule_id)

        rule_dict = AssignmentRuleSerializer(rule).data
        # TODO: move further logic to serializer method readonly-field
        converted_rule_dict = {
            "id": rule_dict["id"],
            "title": rule_dict["title"],
            "mandatory": rule_dict["mandatory"],
            "values": [],
        }
        for disjunction_item in rule.formula:
            for conjunction_item in disjunction_item:
                tag_type_class = TagTypeAdapter.get_tag_type_for_semantic_type(conjunction_item["semantic_type"])
                conjunction_item["operations"] = tag_type_class.get_operations()
                conjunction_item["semantic_type_l10n"] = tag_type_class.get_semantic_type_name_l10n()
                converted_rule_dict["values"].append(conjunction_item)

        return Response(converted_rule_dict)

    def list(self, request, **kwargs):
        all_rules = self.get_queryset()

        data = []

        for rule in all_rules:
            # TODO: move futher logic to serializer method readonly-field
            rule_dict = AssignmentRuleSerializer(rule).data
            converted_rule_dict = {
                "id": rule_dict["id"],
                "title": rule_dict["title"],
                "mandatory": rule_dict["mandatory"],
                "values": [],
            }
            for disjunction_item in rule.formula:
                for conjunction_item in disjunction_item:
                    tag_type_class = TagTypeAdapter.get_tag_type_for_semantic_type(conjunction_item["semantic_type"])
                    conjunction_item["operations"] = tag_type_class.get_operations()
                    conjunction_item["semantic_type_l10n"] = tag_type_class.get_semantic_type_name_l10n()
                    converted_rule_dict["values"].append(conjunction_item)
            data.append(converted_rule_dict)

        return Response({"data": data})

    def create(self, request, **kwargs):
        course_id = self.request.parser_context['kwargs']['course_id']
        course = get_object_or_404(Course, pk=course_id)
        self.check_object_permissions(request=self.request, obj=course)

        raw_assignment_data = request.data
        response_content = []

        assignment_rule = AssignmentRule(
            title=raw_assignment_data.get("title", ""),
            formula=[],
            mandatory=raw_assignment_data.get("mandatory", False),
            course=course,
        )

        formula_dict = {}
        for predicate in raw_assignment_data["values"]:

            semantic_type = predicate["semantic_type"]
            if semantic_type in list(formula_dict.keys()):
                formula_dict[semantic_type].append(
                    predicate
                )
            else:
                formula_dict[semantic_type] = [
                    predicate
                ]

        assignment_rule.formula = list(formula_dict.values())

        try:
            assignment_rule.save()
        except IntegrityError as e:
            raise ValidationError(u"Cannot save rule with same 'formula' or 'title' in the same course")

        response_content.append(self.serializer_class(assignment_rule).data)

        return Response(response_content, status=status.HTTP_201_CREATED)

    def update(self, request, **kwargs):
        course_id = self.request.parser_context['kwargs']['course_id']
        rule_id = self.request.parser_context['kwargs']['pk']
        course = get_object_or_404(Course, pk=course_id)
        self.check_object_permissions(request=self.request, obj=course)
        rule = get_object_or_404(AssignmentRule, pk=rule_id)

        raw_assignment_data = request.data

        rule.title = raw_assignment_data["title"]
        rule.mandatory = raw_assignment_data.get("mandatory", False)
        rule.course = course

        formula_dict = {}
        for predicate in raw_assignment_data["values"]:
            semantic_type = predicate["semantic_type"]
            if semantic_type in list(formula_dict.keys()):
                formula_dict[semantic_type].append(
                    predicate
                )
            else:
                formula_dict[semantic_type] = [
                    predicate
                ]

        rule.formula = list(formula_dict.values())

        try:
            rule.save()
        except IntegrityError as e:
            raise ValidationError(u"Cannot save rule with same 'formula' or 'title' in the same course")

        return Response(
            self.serializer_class(rule).data,
            status=status.HTTP_200_OK
        )

    def destroy(self, request, **kwargs):
        rule = get_object_or_404(AssignmentRule, pk=kwargs["pk"])
        course = get_object_or_404(Course, pk=rule.course_id)
        self.check_object_permissions(request=self.request, obj=course)
        AssignmentRule.objects.filter(id=kwargs["pk"]).delete()
        return Response(status=status.HTTP_204_NO_CONTENT)


class CourseEditViewSet(viewsets.ModelViewSet):
    serializer_class = CourseEditSerializer
    permission_classes = [CanEditCourse]

    def get_serializer_context(self):
        context = super(CourseEditViewSet, self).get_serializer_context()
        context["request"] = self.request
        return context

    def get_queryset(self):
        queryset = Course.objects.all()
        if not self.request.user.is_superuser:
            queryset = queryset.filter(project__in=get_user_projects(self.request.user))
        return queryset


class ReportCourseViewSet(mixins.ListModelMixin, GenericViewSet):
    serializer_class = CourseEditSerializer
    permission_classes = [
        IsAuthenticated,
        Or(IsStaffChief, IsStaffHRBP, IsSuperuser),
    ]

    def get_queryset(self):
        queryset = Course.objects.all()
        if not self.request.user.is_superuser:
            queryset = queryset.filter(project__in=get_user_projects(self.request.user))
        return queryset

    @detail_route(permission_classes=[
        IsAuthenticated,
        Or(IsStaffHRBP, IsSuperuser),
    ])
    def hrreport(self, request, pk):
        limit = serializers.IntegerField().to_internal_value(data=request.GET.get('count', 0))
        offset = serializers.IntegerField().to_internal_value(data=request.GET.get('from', 0))

        completed = request.GET.get('completed')
        if completed is not None:
            completed = serializers.BooleanField().to_internal_value(data=completed)

        with_nested = request.GET.get('with_nested')
        if with_nested is not None:
            with_nested = serializers.BooleanField().to_internal_value(data=with_nested)

        min_delay = request.GET.get('min_delay')
        if min_delay is not None:
            min_delay = serializers.IntegerField().to_internal_value(data=min_delay)

        try:
            data = HRUsersStaffReport(
                courses=[self.get_object()],
                user=request.user,
                completed=completed,
                with_nested=with_nested,
                min_delay=min_delay,
                offset=offset,
                limit=limit,
            ).data
        except StaffReaderError:
            raise APIException('Can\'t get hierarchy from staff')

        return Response({'results': data, 'count': len(data)})

    @detail_route(permission_classes=[
        IsAuthenticated,
        Or(IsStaffHRBP, IsSuperuser),
    ])
    def hrreport_table(self, request, pk):
        try:
            table = HRUsersStaffReport(
                courses=[self.get_object()],
                user=request.user,
            ).table()
        except StaffReaderError:
            raise APIException('Can\'t get hierarchy from staff')

        return as_xlsx(table)

    @detail_route(permission_classes=[
        IsAuthenticated,
        Or(IsStaffChief, IsSuperuser),
    ])
    def chiefreport(self, request, pk):
        limit = serializers.IntegerField().to_internal_value(data=request.GET.get('count', 0))
        offset = serializers.IntegerField().to_internal_value(data=request.GET.get('from', 0))

        completed = request.GET.get('completed')
        if completed is not None:
            completed = serializers.BooleanField().to_internal_value(data=completed)

        with_nested = request.GET.get('with_nested')
        if with_nested is not None:
            with_nested = serializers.BooleanField().to_internal_value(data=with_nested)

        min_delay = request.GET.get('min_delay')
        if min_delay is not None:
            min_delay = serializers.IntegerField().to_internal_value(data=min_delay)

        try:
            data = ChiefUsersStaffReport(
                courses=[self.get_object()],
                user=request.user,
                completed=completed,
                with_nested=with_nested,
                min_delay=min_delay,
                offset=offset,
                limit=limit,
            ).data
        except StaffReaderError:
            raise APIException('Can\'t get hierarchy from staff')

        return Response({'results': data, 'count': len(data)})

    @detail_route(permission_classes=[
        IsAuthenticated,
        Or(IsStaffChief, IsSuperuser),
    ])
    def chiefreport_table(self, request, pk):
        try:
            table = ChiefUsersStaffReport(
                courses=[self.get_object()],
                user=request.user,
            ).table()
        except StaffReaderError:
            raise APIException('Can\'t get hierarchy from staff')

        return as_xlsx(table)

    @list_route(permission_classes=[
        IsAuthenticated,
        Or(CanViewCuratorStats, IsSuperuser),
    ])
    def curator_report(self, request):
        usernames = request.GET.get('usernames').split(',')
        course_ids = request.GET.get('course_ids').split(',')
        limit = int(request.GET.get('count', 0))
        offset = int(request.GET.get('from', 0))

        if not usernames:
            raise APIException('usernames is not provided')
        usernames = User.objects.filter(username__in=usernames).values_list('username', flat=True)
        courses = Course.objects.filter(id__in=course_ids)
        try:
            data = CuratorReport(
                courses=courses,
                usernames=usernames,
                offset=offset,
                limit=limit,
            ).data
        except StaffReaderError:
            raise APIException('Can\'t get hierarchy from staff')

        return Response({'results': data, 'count': len(data)})

    @list_route(permission_classes=[
        IsAuthenticated,
        Or(CanViewCuratorStats, IsSuperuser),
    ])
    def curator_report_table(self, request):
        usernames = request.GET.get('usernames').split(',')
        course_ids = request.GET.get('course_ids').split(',')
        if not usernames:
            raise APIException('usernames is not provided')
        usernames = User.objects.filter(username__in=usernames).values_list('username', flat=True)
        courses = Course.objects.filter(id__in=course_ids)
        try:
            table = CuratorReport(
                courses=courses,
                usernames=usernames,
            ).table()
        except StaffReaderError:
            raise APIException('Can\'t get hierarchy from staff')

        return as_xlsx(table)

    @list_route(permission_classes=[
        IsAuthenticated,
        Or(CanViewStudentStats, IsSuperuser),
    ])
    def student_report(self, request):
        course_ids = request.GET.get('course_ids').split(',')
        usernames = [request.user.username]
        courses = Course.objects.filter(id__in=course_ids)
        try:
            data = CuratorReport(
                courses=courses,
                usernames=usernames,
            ).data
        except StaffReaderError:
            raise APIException('Can\'t get hierarchy from staff')

        return Response({'results': data, 'count': len(data)})

    @list_route(permission_classes=[
        IsAuthenticated,
        Or(CanViewStudentStats, IsSuperuser),
    ])
    def student_report_table(self, request):
        course_ids = request.GET.get('course_ids').split(',')
        usernames = [request.user.username]
        courses = Course.objects.filter(id__in=course_ids)
        try:
            table = CuratorReport(
                courses=courses,
                usernames=usernames,
            ).table()
        except StaffReaderError:
            raise APIException('Can\'t get hierarchy from staff')

        return as_xlsx(table)

    @list_route(permission_classes=[
        IsAuthenticated,
        # TODO: добавить проверку, что пользователь куратор этого курса
    ])
    def curator(self, request):
        queryset = self.filter_queryset(self.get_queryset())
        queryset = queryset.filter(
            coursepermission__user=self.request.user,
            coursepermission__permission__gt=15  # TODO: заменить на & 16 (CURATOR)
        )

        page = self.paginate_queryset(queryset)
        if page is not None:
            serializer = self.get_serializer(page, many=True)
            return self.get_paginated_response(serializer.data)

        serializer = self.get_serializer(queryset, many=True)
        return Response(serializer.data)


class CourseViewSet(BaseCourseViewSet):
    """
    Методы для работы с курсами
    """
    permission_classes = [
        IsAuthenticated,
        Or(IsTeacher, IsContentManager, IsCurator, IsStaff, IsParent, IsDetailRequest),
        Or(ObjectForTeacher, ObjectForContentManager, ObjectForStaff,
           ObjectForOwner, ObjectCourseAssignedToStudent,
           ObjectCourseAssignedToChildren),
    ]
    lookup_value_regex = '\d+'

    @staticmethod
    def __validate_suggest_params(request):
        q = request.query_params.get('q', None)
        if not q:
            raise ValidationError("'q' parameter is mandatory")

        role = request.query_params.get('role', None)
        if role is not None:
            if not role.strip():
                raise ValidationError("'role' parameter is optional but should not be empty")
            if role not in list(UserCourseFilterFactory.AVAILABLE_ROLE_FILTERS.keys()):
                raise ValidationError("Unsupported role: {}".format(role))

    def get_base_queryset(self):
        """
        Переопределеяем метод базового класса, чтобы иметь возможность повлиять на все get_queryset-методы сразу,
        добавляя в них фильтрацию по проектам.
        В отличие от базового get_base_queryset наше переопределение не является статическим методом клаcса.
        Это нужно для того, чтобы иметь возможность получить user из запроса ( self.request.user )
        """
        queryset = BaseCourseViewSet.get_base_queryset()

        user = self.request.user

        if user.is_authenticated:
            if user.is_superuser:
                return queryset
            if user.is_support:
                return queryset.filter(available_for_support=True)

        queryset = queryset.filter(project__in=get_user_projects(user))
        if enabled_feature('E7N_ENABLE_NEW_COURSE_PERMISSIONS', False):
            queryset = queryset.filter(
                coursepermission__user=user,
                coursepermission__permission__gt=0
            )
        return queryset

    def get_student_stats_queryset(self):
        queryset = super(CourseViewSet, self).get_student_stats_queryset()
        return queryset.prefetch_related('courselessonlink_set__lesson__theme')

    def modify_queryset_expand_lessons(self, queryset):
        queryset = super(
            CourseViewSet, self,
        ).modify_queryset_expand_lessons(queryset)
        return queryset.prefetch_related('courselessonlink_set__lesson__theme')

    def get_list_queryset(self):
        """
        Переопределяем метод базового класса:
            - Суперпользьзователя не ограничиваем курсами
            - Для остальных сохраняем логику "смотреть на поле owner", если не
              включено использование нового механизма ролей
        """
        qs = self.get_base_queryset()

        if not self.request.user.is_superuser and not enabled_feature('E7N_ENABLE_NEW_COURSE_PERMISSIONS', False):
            qs = qs.filter(owner=self.request.user)

        return self.add_base_related_and_prefetches(qs)

    def get_courses_by_code(self, codes):
        try:
            courses = Course.objects.filter(
                code__in=codes,
            )
        except (TypeError, ValueError):
            raise ValidationError('Wrong "codes" parameter type')
        return courses

    @detail_route(permission_classes=[
        Or(ObjectCourseForAnonymous,
           And(ObjectForAuthenticated,
               Or(ObjectForTeacher, ObjectForContentManager, ObjectForStaff,
                   ObjectForOwner, ObjectCourseAssignedToStudent,
                   ObjectCourseAssignedToChildren),
               ),
           CanViewCourse,
           ),
    ])
    def web(self, request, *args, **kwargs):
        return super(CourseViewSet, self).web(request, *args, **kwargs)

    @detail_route(permission_classes=[
        IsAuthenticated,
        IsStaff,  # FIXME temporary permission
    ])
    def diagnostics(self, request, pk):
        """
        Результаты по диагностике
        """
        course = self.get_object()

        if not course.courselessonlink_set.filter(
                mode=CourseLessonLink.DIAGNOSTICS_MODE).count():
            raise ValidationError('no diagnostics')

        code = self.request.query_params.get('code')
        if code:
            # ищем статистику по коду при его наличии
            try:
                student_stat = (
                    StudentDiagnosticsStat.objects.select_related(
                        'course', 'course__subject', 'student'
                    ).get(uuid=code)
                )
            except (StudentDiagnosticsStat.DoesNotExist, ValueError):
                raise Http404
        elif self.request.user.is_authenticated():
            # если нет кода, то запрашиваем статистику по авторизованному
            # пользователю
            student_stat, created = (
                StudentDiagnosticsStat.objects
                .select_related('course', 'course__subject', 'student')
                .get_or_create(
                    student=self.request.user,
                    course=course,
                )
            )
        else:
            # другого доступа нет кроме кода и авторизованности
            raise PermissionDenied

        # проверяем надо ли посчитать статистику
        if not student_stat.calculated:
            # подсчитываем статистику ученика
            student_stat.calculate()

        response = StudentDiagnosticsStatSerializers().to_representation(
            student_stat)

        # добавляем средние результаты по занятиям
        averages = {
            stat.clesson_id: stat.average
            for stat in LessonDiagnosticStats.objects.filter(
                clesson__in=course.courselessonlink_set.filter(
                    mode=CourseLessonLink.DIAGNOSTICS_MODE)
            )
        }
        for clesson_result in response['result']:
            clesson_result['percent_average'] = averages.get(
                clesson_result['clesson_id'])

        return Response(response)

    @detail_route(
        methods=['get', 'post'],
        permission_classes=[
            CanManageCourseRoles,
        ],
    )
    def roles(self, request, pk):
        self.get_object()  # For checking object permissions
        if request.method == 'GET':
            permissions = CoursePermission.objects.filter(
                course_id=pk,
                permission__gt=0,
            )

            return Response({
                'data':
                    CoursePermissionSerializer(
                        many=True,
                        context={'request': request}
                    ).to_representation(permissions)
            })
        elif request.method == 'POST':
            user_username = request.data.get('username')
            role = request.data.get('role')

            user = User.objects.filter(username=user_username).first()
            if not user:
                raise NotFound('User not found')

            if role not in ('owner', 'content_manager', 'analyst', 'curator', ''):
                raise ValidationError('Invalid role')

            permission, _ = CoursePermission.objects.get_or_create(
                user=user,
                course_id=pk,
                defaults=dict(permission=0),
            )

            if role:
                if getattr(permission, 'is_' + role, False):
                    raise ValidationError('Role has already granted')
                setattr(permission, 'is_' + role, True)
            else:
                permission.permission = 0

            course_permissions = CoursePermission.objects.filter(course_id=pk).exclude(user=user)
            owners = 0
            for course_permission in course_permissions:
                owners += int((course_permission.permission & CoursePermission.OWNER) > 0)
            owners += int((permission.permission & CoursePermission.OWNER) > 0)
            if not owners:
                raise ValidationError('Course should have at least one owner')

            permission.save()

            return Response({'data': CoursePermissionSerializer(
                context={'request': request}
            ).to_representation(permission)})

    @detail_route(permission_classes=[
        Or(
            And(ObjectForTeacher, ObjectForOwner),
            ObjectForContentManager,
        ),
    ])
    @method_decorator(use_master)
    def journal(self, request, pk):
        """
        Возвращает журнал группы по курсу
        """

        students = CourseStudent.objects.filter(
            course__id=pk,
        ).count()

        show_journal = students <= settings.MAX_COURSE_JOURNAL_STUDENTS
        show_csv = students >= settings.MIN_COURSE_CSV_STUDENTS
        csv_url = None
        data = None
        if show_journal:
            data = CourseGroupJournal(self.get_object()).data_with_staff_groups()
        if show_csv:
            journal = Course.objects.get(pk=pk).journal_resource
            if enabled_feature('E7N_ENABLE_NEW_RESOURCE_URLS', False):
                csv_url = journal.shortened_file_url if journal else None
            else:
                csv_url = journal.file.url if journal else None

        return Response({"data": data, "csv_url": csv_url})

    def get_suggest_queryset(self, query_string):
        queryset = Course.objects.filter(
            project__in=get_user_projects(self.request.user)
        )
        return queryset.filter(
            Q(name__icontains=query_string) |
            Q(subject__name__icontains=query_string)
        ).order_by(Length('name'), Lower('name'))

    @list_route(methods=['get'])
    def suggest(self, request):
        CourseViewSet.__validate_suggest_params(self.request)
        query_string = request.query_params.get('q', None)
        role = request.query_params.get('role', None)
        action = request.query_params.get('action', None)
        objects = self.get_suggest_queryset(query_string)

        # Фронт сейчас присылает role=curator, а имеет ввиду action=stats.
        # То есть suggest должен работать по всем курсам, где пользователь видит отчет.
        # При этом пользователь может быть owner или content_manager.
        if role == 'curator':
            role = None
            action = 'stats'

        if role:
            objects = UserCourseFilterFactory.create_filter_instance_for_role_name(
                role, objects, request.user
            ).get_filtered_qs()

        if action:
            objects = UserCourseFilterFactory.create_filter_instance_for_action_name(
                action, objects, request.user
            ).get_filtered_qs()

        # TODO: выкинуть эту шляпу, сделать пагинацию на уровне всей вьюхи
        self.pagination_class = PaginationWithPageSize
        page = self.paginate_queryset(objects)
        self.pagination_class = None

        qs = page if page is not None else objects
        response_data = CourseSimpleSuggestSerializer(qs, many=True).data
        if page is not None:
            return self.get_paginated_response(response_data)
        else:
            return Response(response_data)

    @list_route(permission_classes=[
        IsAuthenticated,
    ])
    def info(self, request):
        course_ids = request.GET.get('course_ids')
        if not course_ids:
            raise ValidationError('course_ids is required')

        try:
            course_ids = list(map(int, course_ids.split(',')))
        except ValueError:
            raise ValidationError('course_ids must be integers')

        courses = Course.objects.filter(
            project__in=get_user_projects(request.user),
            id__in=course_ids,
        ).select_related('subject')
        return Response(
            CourseSimpleSuggestSerializer(
                many=True,
            ).to_representation(courses)
        )
