from builtins import object
from datetime import timedelta
from itertools import groupby

from django.conf import settings
from django.utils import timezone

from kelvin.courses.models import CourseLessonLink, CourseStudent
from kelvin.lesson_assignments.models import LessonAssignment
from kelvin.lessons.models import LessonProblemLink
from kelvin.results.models import CourseLessonResult


class BaseCourseWebView(object):
    def __init__(self, base_view):
        self.base_view = base_view
        self.user = self.base_view.request.user
        self.now = timezone.now()
        self.now_str = self.now.strftime(
            settings.REST_FRAMEWORK['DATETIME_FORMAT'],
        )

        self._course = None
        self._clessons = None
        self._all_assignments = None
        self._empty_clesson_ids = None
        self._all_results = None
        self._control_work_clessons = None
        self._start_dates = None
        self._available_lesson_problem_links = None
        self._lesson_problem_links = None
        self._clesson_problem_links = None

    @property
    def clesson_problem_links(self):
        if self._clesson_problem_links is None:
            self._clesson_problem_links = {
                clesson.id: (
                    # TODO Extract method
                    [
                        lesson_problem_link
                        for lesson_problem_link
                        in self.lesson_problem_links.get(clesson.lesson_id, [])
                        if lesson_problem_link.id in set(
                            self.available_lesson_problem_links[clesson.id]
                        )
                    ]
                    if clesson.id in self.available_lesson_problem_links else
                    self.lesson_problem_links.get(clesson.lesson_id, [])
                )
                for clesson in self.clessons
            }
        return self._clesson_problem_links

    @property
    def lesson_problem_links(self):
        if self._lesson_problem_links is None:
            problem_links = LessonProblemLink.objects.filter(
                lesson_id__in=[clesson.lesson_id for clesson in self.clessons],
                problem__isnull=False,
            ).order_by('lesson')
            self._lesson_problem_links = {
                lesson_id: list(links) for lesson_id, links in
                groupby(problem_links, lambda obj: obj.lesson_id)
            }
        return self._lesson_problem_links

    @property
    def available_lesson_problem_links(self):
        if self._available_lesson_problem_links is None:
            self._available_lesson_problem_links = {
                assignment.clesson_id: assignment.problems
                for assignment in self.all_assignments
            }
        return self._available_lesson_problem_links

    @property
    def start_dates(self):
        if self._start_dates is None:
            self._start_dates = {
                clesson.id: clesson.start_date for clesson in self.clessons
            }
        return self._start_dates

    @property
    def control_work_clessons(self):
        if self._control_work_clessons is None:
            self._control_work_clessons = {
                clesson.id: clesson for clesson in self.clessons
                if (
                    clesson.mode == CourseLessonLink.CONTROL_WORK_MODE and
                    clesson.id not in self.empty_clesson_ids
                )
            }
        return self._control_work_clessons

    @property
    def all_results(self):
        if self._all_results is None:
            if self.user.is_authenticated:
                self._all_results = {
                    result.summary.clesson_id: result
                    for result in CourseLessonResult.objects.filter(
                        summary__clesson__in=self.clessons,
                        summary__student=self.user,
                    ).order_by('summary', '-date_updated')
                    .distinct('summary')
                    .select_related('summary')
                }
            else:
                self._all_results = {}
        return self._all_results

    @property
    def all_assignments(self):
        if self._all_assignments is None:
            if self.user.is_authenticated:
                self._all_assignments = LessonAssignment.objects.filter(
                    student=self.user,
                    clesson__in=self.clessons,
                )
            else:
                self._all_assignments = []
        return self._all_assignments

    def is_student_in_course(self):
        return CourseStudent.objects.filter(
            course=self.course.id,
            student=self.base_view.request.user,
        ).exists()

    @property
    def empty_clesson_ids(self):
        if self._empty_clesson_ids is None:
            if (
                self.user.is_authenticated and
                self.is_student_in_course()
            ):
                self._empty_clesson_ids = set(
                    [
                        assignment.clesson_id
                        for assignment in self.all_assignments
                        if assignment.problems == []
                    ]
                )
            else:
                self._empty_clesson_ids = set()
        return self._empty_clesson_ids

    @property
    def course(self):
        if self._course is None:
            self._course = self.base_view.get_object()
        return self._course

    @property
    def clessons(self):
        if self._clessons is None:
            self._clessons = self.course.courselessonlink_set.all()
        return self._clessons

    def get_data(self):
        serializer = self.base_view.get_serializer(self.course)
        data = serializer.data

        data['lessons'] = [
            lesson for lesson in data['lessons']
            if lesson['id'] not in self.empty_clesson_ids
        ]

        return data

    def enrich_with_control_work(self, clesson):
        clesson.update(self.base_view.get_control_work_data(
            self.control_work_clessons.get(clesson['id']),
            self.all_results.get(clesson['id']),
            self.now,
        ))

    def set_date_assignment_passed(self, clesson):
        clesson['date_assignment_passed'] = (
            clesson['date_assignment'] < self.now_str
            if clesson['date_assignment'] is not None
            else False
        )

    def set_accessible_to_teacher(self, clesson):
        clesson['accessible_to_teacher'] = bool(
            clesson['accessible_to_teacher'] and
            clesson['accessible_to_teacher'] < self.now_str
        )

    def set_on_air(self, clesson):
        if clesson['start_date'] and clesson['duration']:
            start_date = self.start_dates[clesson['id']]
            clesson['on_air'] = (
                start_date < self.now <
                (start_date + timedelta(minutes=clesson['duration']))
            )
        else:
            clesson['on_air'] = None

    @classmethod
    def extract_results(cls, lesson_result, problem_links, count_problems):
        lesson_summary = lesson_result.get_summary(
            problem_links, lesson_result, with_max_points=False)
        count_right = lesson_result.get_correct_count(
            problem_links, summary=lesson_summary)
        count_wrong = lesson_result.get_incorrect_count(
            problem_links, summary=lesson_summary)
        return count_right, count_wrong, count_problems

    @classmethod
    def is_control_work(cls, clesson):
        return clesson['mode'] == CourseLessonLink.CONTROL_WORK_MODE

    def results_allowed(self, clesson):
        return clesson['evaluation_date'] <= self.now_str

    @classmethod
    def format_results(cls, right, wrong, all_count):
        return {
            "right": right,
            "wrong": wrong,
            "all": all_count,
        }

    def format_control_work_results(
            self, clesson, lesson_result, problem_links, count_problems
    ):
        if self.results_allowed(clesson):
            if not lesson_result:
                return self.format_results(0, count_problems, count_problems)
            else:
                return self.format_results(*self.extract_results(
                    lesson_result, problem_links, count_problems
                ))

    def set_results(self, clesson):
        lesson_result = self.all_results.get(clesson['id'])
        problem_links = self.clesson_problem_links.get(clesson['id'])
        count_problems = len(problem_links) if problem_links else 0
        if self.is_control_work(clesson):
            clesson['results'] = self.format_control_work_results(
                clesson, lesson_result, problem_links, count_problems
            )
        elif lesson_result:
            clesson['results'] = self.format_results(*self.extract_results(
                lesson_result, problem_links, count_problems
            ))
        else:
            clesson['results'] = None

    def enrich_lessons(self, data):
        for clesson in data['lessons']:
            self.enrich_with_control_work(clesson)
            self.set_date_assignment_passed(clesson)
            self.set_accessible_to_teacher(clesson)
            self.set_on_air(clesson)
            self.set_results(clesson)

    def build_data(self):
        data = self.get_data()
        self.enrich_lessons(data)
        return data


class CourseWebView(BaseCourseWebView):
    @property
    def all_results(self):
        if self._all_results is None:
            if self.user.is_authenticated:
                self._all_results = {
                    result.summary.clesson_id: result
                    for result in CourseLessonResult.objects.filter(
                        summary__clesson__in=self.clessons,
                        summary__student=self.user,
                        work_out=False,
                    ).order_by('summary', '-date_updated')
                    .distinct('summary')
                    .select_related('summary')
                }
            else:
                self._all_results = {}
        return self._all_results

    @classmethod
    def is_control_work(cls, clesson):
        return clesson['mode'] in [
            CourseLessonLink.CONTROL_WORK_MODE,
            CourseLessonLink.DIAGNOSTICS_MODE
        ]

    @property
    def control_work_clessons(self):
        if self._control_work_clessons is None:
            self._control_work_clessons = {
                clesson.id: clesson for clesson in self.clessons
                if (
                    clesson.mode in (
                        CourseLessonLink.CONTROL_WORK_MODE,
                        CourseLessonLink.DIAGNOSTICS_MODE,
                    ) and
                    clesson.id not in self.empty_clesson_ids
                )
            }
        return self._control_work_clessons

    def set_on_air(self, clesson):
        if clesson['start_date'] and clesson['duration']:
            start_date = self.start_dates[clesson['id']]
            clesson['on_air'] = (
                start_date < self.now <
                (start_date + timedelta(minutes=clesson['duration']))
            )
        else:
            clesson['on_air'] = None
