from django.db import transaction
from django.utils import timezone

from rest_framework.exceptions import ValidationError
from rest_framework.generics import get_object_or_404
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.status import HTTP_200_OK, HTTP_201_CREATED
from rest_framework.views import APIView

from kelvin.accounts.models import UserProject
from kelvin.courses.models import CourseInvite, UserCourseInviteActivation
from kelvin.courses.serializers.course_invite_activation_view_set import (
    CourseInviteViewSetInputSerializer, CourseInviteViewSetOutputSerializer,
)
from kelvin.courses.utils import add_courses_to_user


class CourseInviteViewSet(APIView):
    permission_classes = [IsAuthenticated,]

    input_serializer = CourseInviteViewSetInputSerializer
    output_serializer = CourseInviteViewSetOutputSerializer

    @transaction.atomic
    def post(self, request, **kwargs):
        serializer = self.input_serializer(data=self.request.data)
        serializer.is_valid(raise_exception=True)

        key = serializer.validated_data['key']
        course_invite = get_object_or_404(
            CourseInvite.objects.all().select_related('course', 'course__project'),
            key=key,
        )
        if not course_invite.is_active:
            raise ValidationError('Invite is not active')
        if course_invite.start_at and course_invite.start_at > timezone.now():
            raise ValidationError('Course invite not started')
        if course_invite.expire_at and course_invite.expire_at < timezone.now():
            raise ValidationError('Course invite expired')
        if (
            course_invite.max_attempts is not None and
            course_invite.max_attempts <= (
                UserCourseInviteActivation.objects.filter(
                    course_invite_id=course_invite.id,
                ).exclude(
                    user=self.request.user,
                ).count()
            )
        ):
            raise ValidationError('Activation max attempts exceeded')

        UserProject.objects.get_or_create(
            user=self.request.user,
            project=course_invite.course.project
        )

        add_courses_to_user(courses=[course_invite.course], user=self.request.user)

        activation, created = UserCourseInviteActivation.objects.get_or_create(
            course_invite=course_invite,
            user=self.request.user,
        )

        return Response(
            self.output_serializer({'course_id': course_invite.course_id}).data,
            status=HTTP_201_CREATED if created else HTTP_200_OK,
        )
