from rest_condition import And, Or

from django.conf import settings
from django.db import transaction
from django.db.models import Q
from django.http import Http404, QueryDict
from django.utils import timezone

from rest_framework import status, viewsets
from rest_framework.decorators import detail_route
from rest_framework.exceptions import ValidationError
from rest_framework.permissions import SAFE_METHODS, IsAuthenticated
from rest_framework.response import Response

from kelvin.accounts.permissions import (
    IsContentManager, IsStaff, IsTeacher, ObjectForAuthenticated, ObjectForContentManager, ObjectForTeacher,
    ObjectReadOnly,
)
from kelvin.accounts.utils import is_student
from kelvin.common.db_functions import safe_get_or_create
from kelvin.common.utils import positive_int_or_400
from kelvin.courses.journal import LessonAssignment, LessonJournal
from kelvin.courses.models import Course, CourseLessonLink, CourseStudent
from kelvin.courses.permissions import (
    CanEditClesson, CanViewClesson, ObjectAccessibleForContentManager, ObjectCLessonAssigned,
    ObjectCLessonAssignedToChildren, ObjectCLessonAssignedToStudent, ObjectCLessonAvailableToStudent,
    ObjectCLessonForAnonymous, ObjectCourseLessonOwner, ObjectCourseSourceAccessibleForTeacher,
    ObjectDenyEditForAssigned,
)
from kelvin.courses.serializers import CLessonSerializer, CourseLessonLinkSerializer
from kelvin.lessons.services import copy_lesson
from kelvin.results.models import CourseLessonResult, CourseLessonSummary
from kelvin.results.serializers import CourseLessonResultSerializer
from kelvin.switcher import get_feature_bool_option as enabled

from .web_view import CLessonWebView

# BaseCLessonViewSet = CLessonViewSetFactory.construct(
#     CLessonSerializer,
#     CourseLessonResultSerializer,
#     LessonJournal,
#     CLessonWebView,
#     CourseLessonLinkSerializer.control_work_data,
# )


class BaseCLessonViewSet(viewsets.ModelViewSet):
    """
        API связи курса и занятия
        """
    course_lesson_result_serializer = None
    journal = None
    WebView = None

    permission_classes = [
        IsAuthenticated,
        Or(ObjectAccessibleForContentManager,
           And(ObjectReadOnly, ObjectCLessonAssignedToStudent),
           ObjectCourseSourceAccessibleForTeacher,
           And(ObjectReadOnly, ObjectCLessonAssignedToChildren)),
        ObjectDenyEditForAssigned,
    ]

    def __init__(self, *args, **kwargs):
        super(BaseCLessonViewSet, self).__init__(*args, **kwargs)
        self._clesson = None

    @staticmethod
    def get_control_work_data(clesson, result, now):
        raise NotImplementedError

    @property
    def expand_problems(self):
        """
        Если указан определенный GET-параметр в запросе,
        то "расширяем" выдачу API.
        """
        return 'expand_problems' in self.request.query_params

    @property
    def hide_answers(self):
        """
        Если указан определенный GET-параметр в запросе,
        то исключаем из выдачи количество баллов за занятие.
        """
        return 'hide_answers' in self.request.query_params

    def perform_update(self, serializer):
        """
        При выдаче занятия затираем результаты учителя
        """
        super(BaseCLessonViewSet, self).perform_update(serializer)
        user = self.request.user
        clesson = self.get_object()
        owner = clesson.course.owner

        if (
                serializer.context.get('just_assigned', False) and
                owner.id == user.id
        ):
            CourseLessonSummary.objects.filter(
                clesson=clesson,
                student=user,
            ).delete()

    @classmethod
    def get_base_queryset(cls):
        return CourseLessonLink.objects.all()

    def get_teacher_queryset(self):
        teacher_source_courses = Course.objects.filter(
            course__owner=self.request.user
        )
        return self.get_base_queryset().filter(
            (
                Q(course__in=teacher_source_courses)
            ) | (
                Q(accessible_to_teacher__lt=timezone.now())
            )
        )

    def get_queryset_base_by_role(self):
        builders = {
            'teacher': self.get_teacher_queryset
        }
        return builders.get(self.get_role(), self.get_base_queryset)()

    def get_role(self):
        if not self.request.user.is_authenticated:
            return 'anonymous'
        if self.request.user.is_content_manager:
            return 'content manager'
        if self.request.user.is_teacher:
            return 'teacher'
        return 'other'

    @classmethod
    def add_related_to_queryset(cls, queryset):
        return queryset.select_related(
            'course',
            'lesson',
            'lesson__subject',
            'progress_indicator',
        )

    @classmethod
    def add_safe_methods_related_to_queryset(cls, queryset):
        return queryset.prefetch_related(
            'lesson__lessonproblemlink_set',
            'lesson__lessonproblemlink_set__problem',
            'lesson__lessonproblemlink_set__theory',
        )

    @property
    def is_student(self):
        return is_student(self.request.user)

    def add_expand_problems_queryset(self, queryset):
        queryset = queryset.prefetch_related(
            'lesson__lessonproblemlink_set__problem__resources',
            'lesson__lessonproblemlink_set__theory__resources',
            (
                'lesson__lessonproblemlink_set__theory'
                '__content_type_object'
            ),
            (
                'lesson__lessonproblemlink_set__theory'
                '__content_type_object__resource'
            ),
        )
        if not self.is_student:
            queryset = queryset.prefetch_related(
                'lesson__lessonproblemlink_set__problem__subject',
                'lesson__lessonproblemlink_set__problem__meta',
            )
        return queryset

    @classmethod
    def add_expand_meta_queryset(cls, queryset):
        queryset = queryset.prefetch_related(
            (
                'lesson__lessonproblemlink_set__problem__meta'
                '__main_theme'
            ),
            (
                'lesson__lessonproblemlink_set__problem__meta'
                '__additional_themes'
            ),
            'lesson__lessonproblemlink_set__problem__meta__group_levels',
        )
        return queryset

    def expand_queryset_for_safe_methods(self, queryset):
        queryset = self.add_safe_methods_related_to_queryset(queryset)
        if self.expand_problems:
            queryset = self.add_expand_problems_queryset(queryset)
        if (
                not self.is_student and
                self.request.query_params.get('expand_meta')
        ):
            queryset = self.add_expand_meta_queryset(queryset)
        return queryset

    def get_queryset(self):
        """
        Делаем меньше запросов
        """
        queryset = self.get_queryset_base_by_role()
        queryset = self.add_related_to_queryset(queryset)
        if self.request.method in SAFE_METHODS:
            queryset = self.expand_queryset_for_safe_methods(queryset)
        return queryset

    def get_serializer_context(self):
        """
        Добавляем параметры из запроса и дополнительные данные
        в контекст сериализатора
        """
        ctx = super(BaseCLessonViewSet, self).get_serializer_context()
        ctx['expand_problems'] = self.expand_problems
        ctx['hide_answers'] = self.hide_answers

        if self.request.user.is_authenticated and (
                self.action == 'retrieve' or self.action == 'web'):
            ctx['assignment'] = LessonAssignment.get_student_problems(
                self.request.user, self.get_object())
        ctx['for_student'] = is_student(self.request.user)
        return ctx

    def get_object(self):
        """
        Если требуется изменить занятие, которое нельзя редактировать, то
        копируем занятие
        """
        if self._clesson:
            return self._clesson
        clesson = super(BaseCLessonViewSet, self).get_object()
        # TODO думаю еще стоит отделить редактирование только связи
        # чтобы не делать всегда копии
        if (
                self.request.method not in SAFE_METHODS and
                clesson and not clesson.lesson_editable
        ):
            with transaction.atomic():
                old_to_new = copy_lesson(clesson.lesson, owner=self.request.user)
                self._change_data_ids(old_to_new)
                clesson.lesson_editable = True
                clesson.lesson_id = old_to_new['new_lesson_id']
                clesson.save()
        self._clesson = clesson
        return clesson

    def _change_data_ids(self, old_to_new):
        """
        Изменить данные запроса на новые, чтобы изменялись новые объекты
        """
        if isinstance(self.request.data, QueryDict):
            is_mutable = self.request.data._mutable
            self.request.data._mutable = True
        self.request.data['id'] = old_to_new['new_lesson_id']
        # меняем идентификаторы связей занятие-вопрос
        if (
                'problems' in self.request.data and
                isinstance(self.request.data['problems'], list)
        ):
            for problem in self.request.data['problems']:
                if isinstance(problem, dict) and 'id' in problem:
                    try:
                        old_problem_id = int(problem['id'])
                    except (ValueError, TypeError):
                        # не наше дело говорить, что данные неправильные
                        continue
                    problem['id'] = old_to_new['problems'].get(old_problem_id,
                                                               old_problem_id)

        if isinstance(self.request.data, QueryDict):
            self.request.data._mutable = is_mutable

    @detail_route(permission_classes=[
        Or(
            And(
                ObjectCLessonForAnonymous,
                ObjectCLessonAssigned,
            ),
            And(
                ObjectForAuthenticated,
                # At first look at permissions without additional queries,
                # than look at permissions for student, teacher, parent
                Or(
                    ObjectAccessibleForContentManager,
                    ObjectCLessonAssignedToStudent,
                    ObjectCourseSourceAccessibleForTeacher,
                    ObjectCLessonAssignedToChildren,
                ),
            ),
        ),
    ])
    def web(self, request, pk):
        """
        Получение курсозанятия вебом с дополнительной информацией по
        контрольной
        """
        web_view = self.WebView(self, pk)
        return Response(web_view.build_data())

    # TODO: fix ObjectCLessonAssignedToStudent permissions for School/Sirius
    @detail_route(permission_classes=[
        IsAuthenticated,
        Or(ObjectForTeacher,
           ObjectCLessonAssignedToChildren,
           ObjectCLessonAssignedToStudent,
           ObjectForContentManager),
    ])
    def results(self, request, pk):
        """
        Получить сводку результатов группы учеников
        """
        students = CourseStudent.objects.filter(
            course__courselessonlink__id=pk,
        ).count()

        show_journal = students <= settings.MAX_LESSON_JOURNAL_STUDENTS
        show_csv = students >= settings.MIN_LESSON_CSV_STUDENTS
        data = None
        csv_url = None

        if show_journal:
            data = {
                'students': self.journal(self.get_object()).data,
            }

        if show_csv:
            journal = CourseLessonLink.objects.get(pk=pk).journal
            csv_url = journal.url if journal else None

        return Response({"data": data, "csv_url": csv_url})

    @detail_route(permission_classes=[IsAuthenticated])
    def latest_result(self, request, pk):
        """
        Получить последний результат ученика по занятию
        """
        pk = positive_int_or_400(pk)

        try:
            latest_result = (
                CourseLessonResult.objects.filter(
                    summary__clesson=pk,
                    summary__student=request.user
                ).select_related('summary').latest('date_created')
            )
        except CourseLessonResult.DoesNotExist:
            raise Http404
        return Response(
            self.course_lesson_result_serializer(
                context=self.get_serializer_context(),
            ).to_representation(latest_result)
        )

    @detail_route(methods=['post'], permission_classes=[
        IsAuthenticated, ObjectCourseLessonOwner,
    ])
    def complete(self, request, pk):
        """
        Завершить занятие
        """
        clesson = self.get_object()
        if not clesson.date_assignment:
            raise ValidationError('CLesson must be assigned')
        if clesson.date_completed:
            raise ValidationError('CLesson is completed')
        clesson.date_completed = timezone.now()
        clesson.save()
        return Response(self.get_serializer().to_representation(clesson))

    @detail_route(methods=['post'], permission_classes=[
        IsAuthenticated, IsStaff,  # FIXME temporary permission IsStaff
    ])
    def finish(self, request, pk):
        """
        Делает отметку, что ученик завершил занятие
        """
        summary, created = safe_get_or_create(
            CourseLessonSummary,
            clesson=self.get_object(),
            student=self.request.user,
            defaults={'lesson_finished': True},
        )
        if not (created or summary.lesson_finished):
            summary.lesson_finished = True
            summary.save()
        return Response(status=(status.HTTP_201_CREATED if created else status.HTTP_200_OK))

    @detail_route(methods=['post'], permission_classes=[
        IsAuthenticated, Or(IsTeacher, IsContentManager),
    ])
    def make_homework(self, request, pk):
        """
        Создание домашнего задания из завершенного занятия
        """
        clesson = self.get_object()
        homework = self.journal(clesson).make_homework()
        if homework:
            return Response({'id': homework.id}, status=status.HTTP_201_CREATED)
        else:
            return Response({'id': None}, status=status.HTTP_200_OK)


class CorpBaseCLessonViewSet(BaseCLessonViewSet):
    """
    API связи курса и занятия
    """
    serializer_class = CLessonSerializer
    course_lesson_result_serializer = CourseLessonResultSerializer
    journal = LessonJournal
    WebView = CLessonWebView

    permission_classes = [
        IsAuthenticated,
        Or(ObjectAccessibleForContentManager,
           And(ObjectReadOnly, ObjectCLessonAssignedToStudent),
           ObjectCourseSourceAccessibleForTeacher,
           And(ObjectReadOnly, ObjectCLessonAssignedToChildren)),
        ObjectDenyEditForAssigned,
    ]

    def __init__(self, *args, **kwargs):
        super(CorpBaseCLessonViewSet, self).__init__(*args, **kwargs)
        self._clesson = None

    @staticmethod
    def get_control_work_data(clesson, result, now):
        return CourseLessonLinkSerializer.control_work_data(clesson, result, now)


class CLessonViewSet(CorpBaseCLessonViewSet):
    """
    API связи курса и занятия
    """
    permission_classes = [
        IsAuthenticated,
        CanEditClesson,
        Or(ObjectAccessibleForContentManager,
           And(ObjectReadOnly, ObjectCLessonAssignedToStudent),
           ObjectCourseSourceAccessibleForTeacher,
           And(ObjectReadOnly, ObjectCLessonAssignedToChildren)),
        ObjectDenyEditForAssigned,
    ]

    def expand_queryset_for_safe_methods(self, queryset):
        if not self.is_student:
            queryset = queryset.prefetch_related('lesson__methodology')
        return super(CLessonViewSet, self).expand_queryset_for_safe_methods(
            queryset
        )

    @classmethod
    def add_expand_meta_queryset(cls, queryset):
        return queryset.prefetch_related(
            'lesson__lessonproblemlink_set__problem__meta__'
            'main_theme',
            'lesson__lessonproblemlink_set__problem__meta__'
            'additional_themes',
            'lesson__lessonproblemlink_set__problem__meta__'
            'group_levels',
            'lesson__lessonproblemlink_set__problem__meta__skills',
            'lesson__lessonproblemlink_set__problem__meta__'
            'problemmetaexam_set',
            'lesson__lessonproblemlink_set__problem__meta__'
            'problemmetaexam_set__exam',
        )

    @detail_route(permission_classes=[
        Or(ObjectCLessonForAnonymous,
           And(ObjectForAuthenticated,
               ObjectCLessonAvailableToStudent,  # проверка на доступность в рамках траектории курсов
               # Первыми идут доступы, не требующие дополнительных запросов,
               # потом доступы ученика, учителя, родителя
               Or(
                  And(ObjectAccessibleForContentManager, CanViewClesson),
                  ObjectCLessonAssignedToStudent,
                  ObjectCourseSourceAccessibleForTeacher,
                  ObjectCLessonAssignedToChildren),
               )
           )
    ])
    def web(self, request, pk):
        res = super(CLessonViewSet, self).web(request, pk)

        return res

    @detail_route(permission_classes=[
        IsAuthenticated,
        Or(ObjectForTeacher,
           ObjectCLessonAssignedToChildren,
           ObjectCLessonAssignedToStudent,
           ObjectForContentManager),
    ])
    def results(self, request, pk):
        """
        Получить сводку результатов группы учеников
        """
        students = CourseStudent.objects.filter(
            course__courselessonlink__id=pk,
        ).count()

        show_journal = students <= settings.MAX_LESSON_JOURNAL_STUDENTS
        show_csv = students >= settings.MIN_LESSON_CSV_STUDENTS
        data = None
        csv_url = None
        if show_journal:
            data = {
                'students': self.journal(self.get_object()).data,
            }
        if show_csv:
            journal = CourseLessonLink.objects.get(pk=pk).journal_resource
            if enabled('E7N_ENABLE_NEW_RESOURCE_URLS', False):
                csv_url = journal.shortened_file_url if journal else None
            else:
                csv_url = journal.file.url if journal else None

        return Response({"data": data, "csv_url": csv_url})
