from builtins import object

from django.conf import settings
from django.core.cache import cache
from django.http.response import Http404
from django.utils import timezone

from kelvin.common.utils import dt_to_microseconds
from kelvin.courses.models import CourseLessonLink, UserCLessonState
from kelvin.group_levels.serializers import GroupLevelBasicSerializer
from kelvin.results.models import CourseLessonResult

DEFAULT_CLESSON_IS_AVAILABLE = settings.COURSES_DEFAULT_CLESSON_IS_AVAILABLE


class BaseCLessonWebView(object):
    ANONYMOUS_CLESSON_CACHE_KEY = (
        'clesson_{id}_{date_updated}_{expand_problems}{hide_answers}'
    )
    CLESSON_CACHE_TIMEOUT = 600

    group_level_basic_serializer = GroupLevelBasicSerializer

    def __init__(self, base_view, pk):
        self.base_view = base_view
        self.user = base_view.request.user
        self.now = timezone.now()
        self.pk = pk
        self._clesson = None
        self._serializer = None
        self._data = None

    @property
    def data(self):
        if self._data is None:
            self._data = self.serializer.data
            self.set_date_assignment_passed()
            self.set_accessible_to_teacher()
            self.set_group_levels()
        return self._data

    @property
    def serializer(self):
        if self._serializer is None:
            self._serializer = self.base_view.get_serializer(self.clesson)
            self.set_context()
        return self._serializer

    @property
    def clesson(self):
        if self._clesson is None:
            self._clesson = self.base_view.get_object()
        return self._clesson

    def set_context(self):
        if (
                self.base_view.request.query_params and
                self.base_view.request.query_params.get('expand_main_theme')
        ):
            self.serializer.context['expand_problems'] = True
            self.serializer.context['expand_main_theme'] = True

    def get_date_updated(self):
        return CourseLessonLink.objects.values_list(
            'date_updated',
            flat=True,
        ).get(id=self.pk)

    def check_cache(self):
        if not self.user.is_authenticated:
            try:
                date_updated = self.get_date_updated()
            except (
                    CourseLessonLink.DoesNotExist,
                    ValueError,
                    TypeError
            ):
                raise Http404

            cache_key = self.ANONYMOUS_CLESSON_CACHE_KEY.format(
                id=self.pk,
                date_updated=dt_to_microseconds(date_updated),
                expand_problems=int(self.base_view.expand_problems),
                hide_answers=int(self.base_view.hide_answers),
            )
            if cache_key in cache:
                return cache.get(cache_key)

    def set_date_assignment_passed(self):
        self.data['clesson']['date_assignment_passed'] = (
            self.clesson.date_assignment < self.now
            if self.clesson.date_assignment is not None
            else False
        )

    def set_accessible_to_teacher(self):
        self.data['clesson']['accessible_to_teacher'] = (
            self.clesson.accessible_to_teacher < self.now
            if self.clesson.accessible_to_teacher is not None
            else False
        )

    def set_group_levels(self):
        self.data['group_levels'] = (
            self.group_level_basic_serializer(many=True)
                .to_representation(self.clesson.course.group_levels)
        )

    def set_data_to_cache(self):
        cache_key = self.ANONYMOUS_CLESSON_CACHE_KEY.format(
            id=self.clesson.id,
            date_updated=dt_to_microseconds(self.clesson.date_updated),
            expand_problems=int(self.base_view.expand_problems),
            hide_answers=int(self.base_view.hide_answers),
        )
        cache.set(
            cache_key,
            self.data,
            timeout=self.CLESSON_CACHE_TIMEOUT,
        )

    def get_result(self):
        return CourseLessonResult.objects.filter(
            summary__clesson=self.clesson,
            summary__student=self.user,
        ).order_by('-date_updated').first()

    def enrich_control_work(self):
        self.data['clesson'].update(
            self.base_view.get_control_work_data(
                self.clesson,
                self.get_result(),
                self.now,
            ),
        )

    def build_data(self):
        cache_data = self.check_cache()
        if cache_data is not None:
            return cache_data

        if not self.user.is_authenticated:
            # сейчас не поддерживаем контрольные для анонимных
            # пользователей
            self.set_data_to_cache()
            return self.data

        if self.clesson.mode == CourseLessonLink.CONTROL_WORK_MODE:
            self.enrich_control_work()

        return self.data


class CLessonWebView(BaseCLessonWebView):
    @property
    def data(self):
        super(CLessonWebView, self).data
        self.set_available()
        return self._data

    def set_available(self):
        data = super(CLessonWebView, self).data

        if self.user.is_anonymous:
            data['clesson']['available'] = DEFAULT_CLESSON_IS_AVAILABLE
            return

        try:
            data['clesson']['available'] = UserCLessonState.objects.get(
                user=self.user,
                clesson=self.clesson
            ).available
        except UserCLessonState.DoesNotExist:
            data['clesson']['available'] = DEFAULT_CLESSON_IS_AVAILABLE

    def get_result(self):
        return CourseLessonResult.objects.filter(
            summary__clesson=self.clesson,
            summary__student=self.user,
            work_out=False,
        ).order_by('-date_updated').first()

    group_level_basic_serializer = GroupLevelBasicSerializer
