from collections import OrderedDict

from django.conf import settings

from rest_framework.mixins import ListModelMixin
from rest_framework.response import Response
from rest_framework.viewsets import GenericViewSet

from kelvin.accounts.models import User
from kelvin.courses.serializers.course_student_result_for_achievment import UserWithResultsSerializer
from kelvin.results.models import CourseLessonResult

from ..permissions import course_tvmservice_permission_factory

CourseTVMServicePermission = course_tvmservice_permission_factory()


class CourseStudentsResultsViewSet(ListModelMixin, GenericViewSet):

    authentication_classes = []
    queryset = User.objects.all()
    serializer_class = UserWithResultsSerializer

    permission_classes = [
        CourseTVMServicePermission
    ]

    def get_queryset(self, *args, **kwargs):
        users = super(GenericViewSet, self).get_queryset(*args, **kwargs)

        filters = {}
        logins = self.request.query_params.getlist('login')
        if logins:
            filters['username__in'] = logins

        return users.filter(**filters).only('id', 'username')

    def populate(self, queryset):
        course_id = self.kwargs['course_pk']
        students_map = OrderedDict()
        for user in queryset:
            user.clessons = []
            user.total_result = {'points': 0., "max_points": 0.}
            students_map[user.id] = user
        results = (
            CourseLessonResult.objects
            .filter(summary__student_id__in=list(students_map.keys()), summary__clesson__course_id=course_id)
            .select_related('summary', 'summary__clesson')
            .only('points', 'max_points', 'summary__student_id', 'summary__clesson_id')
        )

        for result in results:
            clesson_id = result.summary.clesson_id
            user = students_map[result.summary.student_id]
            points = result.points
            max_points = result.max_points
            percent = float(points) / max_points * 100 if max_points != 0 else 0.
            user.clessons.append(
                {
                    'clesson_id': clesson_id,
                    'result': {
                        'points': points,
                        'max_points': max_points,
                        'percent': percent,
                        'last_modified': result.date_updated,
                    }
                }
            )
            user.total_result['points'] += points
            user.total_result['max_points'] += max_points

        return list(students_map.values())

    def list(self, request, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())

        page = self.paginate_queryset(queryset)
        if page is not None:
            paginated_queryset = page
        else:
            paginated_queryset = queryset

        final_queryset = self.populate(paginated_queryset)

        serializer = self.get_serializer(final_queryset, many=True)

        if page is not None:
            return self.get_paginated_response(serializer.data)

        return Response(serializer.data)
