from builtins import str

from rest_framework import viewsets
from rest_framework.decorators import detail_route, list_route
from rest_framework.permissions import IsAdminUser, IsAuthenticated
from rest_framework.response import Response

from kelvin.common.utils import LoggableMixin, log_method
from kelvin.courses.models import CourseLessonLink
from kelvin.courses.serializers import CLessonSerializer, SiriusCourseLessonLinkSerializer
from kelvin.lessons.models import LessonProblemLink
from kelvin.results.models import CourseLessonResult
from kelvin.sapi.pagination import SiriusApiPaginator, SiriusExportPaginator
from kelvin.sapi.serializers import CLessonProblemSerializer, CourseLessonResultExportSerializer


class SiriusCLesson(LoggableMixin, viewsets.ReadOnlyModelViewSet):
    """
    API для работы с курсозанятиями

    URL: /api/v2/sirius-clesson/
    """

    serializer_class = CLessonSerializer
    permission_classes = [
        IsAuthenticated,
    ]
    pagination_class = SiriusApiPaginator

    def get_queryset(self, order_by=None):
        """
        Загружаем связанные поля моделей
        """
        if not order_by:
            order_by = []

        qs = CourseLessonLink.objects.all().prefetch_related(
            'courselessonsummary_set'
        ).select_related(
            'course', 'lesson'
        ).order_by(*order_by)

        return qs

    @log_method
    @list_route(methods=['get'])
    def controlwork(self, request):
        """
        Список назначенных пользователю контрольных работ

        `url: /api/v2/sirius-clesson/controlwork`
        """
        qs = self.get_queryset(order_by=['start_date', 'course_id']).filter(
            course__students=request.user,
            mode=CourseLessonLink.CONTROL_WORK_MODE,
        )

        serializer = SiriusCourseLessonLinkSerializer(qs, many=True)

        return Response(serializer.data)

    @log_method
    @detail_route(methods=['get'])
    def problems(self, request, pk=None):
        """
        Возвращает список заданий модуля с полями:
        - id
        - name
        - order
        - max_points
        - points
        - completed

        URL: /api/v2/sirius-clesson/<clesson_pk>/problems/
        """
        problem_links = LessonProblemLink.objects.filter(
            lesson__course__students=request.user,
            lesson__courselessonlink=pk,
        ).select_related(
            'problem',
            'theory'
        )

        try:
            clesson_result = CourseLessonResult.objects.filter(
                summary__clesson__pk=pk,
                summary__student_id=request.user.pk
            )[0]
            answers_data = clesson_result.answers

        except (CourseLessonResult.DoesNotExist, IndexError):
            answers_data = {}

        result_data = CLessonProblemSerializer(problem_links, many=True).data

        # forgive me Jesus
        for item in result_data:
            answer = answers_data.get(str(item['id']))

            if answer:
                item['points'] = answer[-1].get('points')
                item['completed'] = answer[-1].get('completed')
            else:
                item['points'] = 0
                item['completed'] = False

        return Response(result_data)

    @log_method
    @detail_route(methods=['get'], permission_classes=[
        IsAdminUser,
    ])
    def export_results(self, request, pk=None):
        """

        """
        qs = CourseLessonResult.objects.filter(
            summary__clesson=pk,
            work_out=False
        ).values(
            'summary__student__username',
            'points',
        )

        paginator = SiriusExportPaginator()
        page_objects = paginator.paginate_queryset(qs, request)
        serializer = CourseLessonResultExportSerializer(
            page_objects, many=True,
        )

        return paginator.get_paginated_response(serializer.data)
