import re
from builtins import str

from django_filters.rest_framework import DjangoFilterBackend
from django_replicated.decorators import use_state

from django.conf import settings
from django.contrib.auth import get_user_model
from django.db import transaction
from django.db.models import Prefetch, Q

from rest_framework import filters, status, viewsets
from rest_framework.decorators import detail_route, list_route
from rest_framework.exceptions import ValidationError
from rest_framework.permissions import IsAdminUser, IsAuthenticated
from rest_framework.response import Response

from kelvin.accounts.utils import (
    get_user_courses_qs, get_user_projects, invalidate_user_courses_cache, user_matched_to_course,
)
from kelvin.common.utils import LoggableMixin, log_method
from kelvin.courses.models import Course, CourseFeedback, CourseLessonLink, CourseStudent, UserCLessonState
from kelvin.courses.permissions import ObjectCourseAssignedToStudent
from kelvin.courses.serializers import (
    CourseDetailApiSerializer, CourseDetailSerializer, CourseFeedbackSerializer, CoursesAssignedSerializer,
    CoursesLibrarySerializer, SiriusCourseLessonWithResultSerializer,
)
# sapi serializators
from kelvin.lesson_assignments.models import LessonAssignment
from kelvin.lessons.models import LessonProblemLink
from kelvin.result_stats.models import StudentCourseStat
from kelvin.results.models import CourseLessonResult
from kelvin.sapi.pagination import SiriusApiPaginator
from kelvin.sapi.serializers import (
    CourseSubjectSerializer, GetClessonFeaturesApiSerializer, SiriusCourseLessonResultsGroupSerializer,
)
from kelvin.subjects.models import Subject

User = get_user_model()

use_master = use_state(forced_state='master')

CLESSON_LECTURE_SLUG = 'lecture'


def replace_url_dots(url):
    """
    allow process request in API with user logins which consist dots ('.')
    :param url: request url
    :return:
    """
    return re.sub(r'__dot__', '.', url)


class SiriusCourses(LoggableMixin, viewsets.ReadOnlyModelViewSet):
    """
    Хэндлер запросов к API курсов

    URL: /api/v2/sirius-courses/
    """
    serializer_class = CoursesLibrarySerializer
    permission_classes = [
        IsAuthenticated,
    ]
    pagination_class = SiriusApiPaginator
    lookup_value_regex = '\d+'
    filter_backends = [filters.SearchFilter, DjangoFilterBackend]
    search_fields = ['name', 'code']
    filter_fields = [
        'name', 'code', 'score_count', 'project', 'subject', 'mode',
        'owner', 'author', 'color', 'copy_of', 'free', 'date_closed',
    ]
    queryset = Course.objects.all()

    # # We have to use master here while we are using db cache
    # # TODO: as soon as we use redis we can remove this construction
    # @method_decorator(use_master)
    # def dispatch(self, request, *args, **kwargs):
    #     return super(SiriusCourses, self).dispatch(request, *args, **kwargs)

    def get_queryset(self, user=None, order_by=None):
        """
        Делаем префетч
        """
        conditions = []
        select_related = ['cover', 'subject']
        prefetch_related = [
            Prefetch(
                'feedback_set',
                queryset=CourseFeedback.objects.filter(user=self.request.user),
            ),
        ]
        if user is not None:
            conditions.append(
                Q(students=user, coursestudent__deleted=False),
            )

        if not order_by:
            order_by = []

        qs = (super().get_queryset().filter(
            *conditions
        ).select_related(
            *select_related
        ).prefetch_related(
            *prefetch_related
        ).order_by(*order_by))

        return qs

    @log_method
    @list_route(methods=['get'], permission_classes=[
        IsAuthenticated,
    ])
    def assigned(self, request):
        """
        Возвращает список назначенных для пользователя курсов

        URL: /api/v2/sirius-courses/assigned/
        """
        qs = self.get_queryset(user=request.user).filter(
            date_closed__isnull=True
        )

        serializer = CoursesAssignedSerializer(
            qs,
            many=True,
            context={
                'user_id': request.user.id,
                'request': request,
            }
        )

        return Response(serializer.data)

    @log_method
    @detail_route(methods=['get'], permission_classes=[
        IsAdminUser
    ])
    def assigned_to(self, request, pk=None):
        """
        Возвращает список назначенных пользователю курсов
        и входящие в них курсозанятия для API ЛК

        pk принимает username
        """
        if pk is None:
            return Response(status=status.HTTP_400_BAD_REQUEST)

        try:
            pk = replace_url_dots(pk)
            user = User.objects.get(pk=pk)
        except User.DoesNotExist:
            return Response(status=status.HTTP_404_NOT_FOUND)

        qs = self.get_queryset(user=user)

        courses_data = CourseDetailApiSerializer(qs, many=True).data
        courses_list = [x["id"] for x in courses_data]

        clessons_qs = CourseLessonLink.objects.filter(
            course__in=courses_list,
        ).select_related(
            'lesson',
            'course',
        ).prefetch_related(
            Prefetch(
                'userclessonstate_set',
                queryset=UserCLessonState.objects.filter(user=request.user),
            )
        )

        clesson_ids = clessons_qs.values_list('id', flat=True)
        course_lesson_results = SiriusCourseLessonResultsGroupSerializer(
            instance=CourseLessonResult.objects.filter(
                summary__clesson__course__in=courses_list,
                summary__student_id=user.id,
                summary__clesson_id__in=clesson_ids
            ).select_related(),
            many=True
        )

        lesson_problem_link = LessonProblemLink.objects.filter(
            lesson__course__id__in=courses_list,
        ).select_related(
            'problem',
            'lesson',
        ).order_by('id').distinct('id').values(
            'finish_date', 'group', 'id', 'lesson_id', 'options', 'order',
            'problem', 'problem_id', 'problem__max_points', 'start_date',
            'theory', 'theory_id', 'type'
        )

        assignments = LessonAssignment.objects.filter(
            student_id=request.user.id,
            clesson__in=clesson_ids
        ).values('clesson_id', 'problems')

        lesson_assignments_problems = [
            {a['clesson_id']: a['problems']} for a in assignments
        ]

        clessons_serializer = SiriusCourseLessonWithResultSerializer(
            clessons_qs, many=True, context={
                'course_lesson_results': course_lesson_results,
                'lesson_assignments_problems': lesson_assignments_problems,
                'lesson_problem_link_values': lesson_problem_link,
                'user_id': request.user.id,
            }
        )

        clessons = {}
        for clesson_data in clessons_serializer.data:
            if clesson_data['lesson_block_slug'] == CLESSON_LECTURE_SLUG \
                    or clesson_data['allow_time_limited_problems'] is True:
                # не выводим в апи для ЛК - лекционные модули и
                # модули непрерывной олимпиады
                continue

            course_id = clesson_data['course_id']
            clesson_data.pop('progress_indicator', None)
            clesson_data.pop('course_id', None)
            clesson_data.pop('lesson_block_slug', None)
            clesson_data.pop('allow_time_limited_problems', None)
            clesson_data.pop('completed', None)
            clesson_data.pop('problems_count', None)
            clesson_data.pop('problems_completed', None)
            if clessons.get(course_id) is None:
                clessons[course_id] = []
                clessons[course_id].append(clesson_data)
            else:
                clessons[course_id].append(clesson_data)

        for course in courses_data:
            course['clessons'] = clessons.get(course['id'], [])

        return Response(courses_data)

    @log_method
    @detail_route(methods=['get'], permission_classes=[
        IsAdminUser
    ])
    def assigned_to_init(self, request, pk=None):
        """
        Возвращает список назначенных пользователю курсов
        и входящие в них курсозанятия для API ЛК

        pk принимает username
        """
        if pk is None:
            return Response(status=status.HTTP_400_BAD_REQUEST)

        try:
            pk = replace_url_dots(pk)
            user = User.objects.get(username=pk)
        except User.DoesNotExist:
            return Response(status=status.HTTP_404_NOT_FOUND)

        qs = self.get_queryset(user=user)

        courses_data = CourseDetailApiSerializer(qs, many=True).data
        courses_list = [x["id"] for x in courses_data]

        mp = GetClessonFeaturesApiSerializer(
            courses_list=courses_list, user_id=user.id,
        )
        (max_points, started_at) = mp.get_clessons_features()

        courses_stat = list(StudentCourseStat.objects.filter(
            course__in=courses_list, student_id=user.id,
        ).values())

        results = {}
        for course_stat in courses_stat:
            data = course_stat['clesson_data']
            for result_item in data:
                results[result_item] = data[result_item]

        clessons_data = CourseLessonLink.objects.filter(
            course__in=courses_list,
        ).select_related(
            'lesson',
            'course',
        ).values(
            'id',
            'lesson_id',
            'lesson__name',
            'date_assignment',
            'date_completed',
            'finish_date',
            'evaluation_date',
            'start_date',
            'mode',
            'duration',
            'comment',
            'start_date',
            'course__id',
        )

        clessons = {}
        for clesson in clessons_data:
            course_id = clesson['course__id']
            clesson_result = results.get(str(clesson['id']), None)
            clesson['points'] = (
                clesson_result['points'] if clesson_result else 0
            )
            clesson['started_at'] = started_at.get(clesson['lesson_id'], None)
            clesson['max_points'] = max_points.get(clesson['lesson_id'], 0)
            clesson['name'] = clesson['lesson__name']
            clesson.pop('lesson__name', None)
            clesson.pop('course__id', None)
            if course_id in clessons:
                clessons[course_id].append(clesson)
            else:
                clessons[course_id] = []
                clessons[course_id].append(clesson)

        for course in courses_data:
            course['clessons'] = clessons.get(course['id'], [])

        return Response(courses_data)

    @log_method
    @list_route(methods=['get'], permission_classes=[
        IsAuthenticated
    ])
    def library(self, request):
        """
        Возвращает данные о библиотеке курсов

        URL: /api/v2/sirius-courses/library/
        """
        # Get courses that are not closed, free, and available to user
        # staff group
        use_matchman = request.user.experiment_flags.get('enable_matchman', False)
        qs = self.get_queryset().filter(
            id__in=get_user_courses_qs(
                user=request.user,
                use_matchman=use_matchman,
            )
        )

        qs = self.filter_queryset(qs)

        courses_serializer = CoursesLibrarySerializer(qs, many=True)

        # subjects section
        subjects_qs = Subject.objects.filter(public=True).order_by('name')
        subjects_serializer = CourseSubjectSerializer(subjects_qs, many=True)

        resp = {
            'courses': courses_serializer.data,
            'subjects': subjects_serializer.data,
        }

        return Response(resp)

    @log_method
    @list_route(methods=['post'], permission_classes=[
        IsAuthenticated,
    ])
    def join(self, request):
        """
        По коду курса добавляем пользователя к курсу

        URL: /api/v2/sirius-courses/join/
        """
        # list of unicode strings
        course_codes = [c.upper() for c in request.data.get('codes', []) if c]

        if len(course_codes) == 0:
            err_msg = u'Course codes list is empty'
            self.log.warning(err_msg)
            return Response(data=err_msg, status=status.HTTP_400_BAD_REQUEST)

        user = request.user

        # Check courses codes and is user available to use this courses
        courses = self.get_queryset().filter(
            code__in=course_codes
        )

        if not request.user.is_superuser:
            courses = courses.filter(project__in=get_user_projects(request.user))
            matched_courses_ids = []
            use_matchman = request.user.experiment_flags.get('enable_matchman', False)
            for course in courses:
                if user_matched_to_course(
                        user=user,
                        course=course,
                        use_matchman=use_matchman,
                ):
                    matched_courses_ids.append(course.id)

            courses = courses.filter(id__in=matched_courses_ids)

        if not courses.exists():
            err_msg = u'Course codes are invalid'
            self.log.warning(err_msg)
            return Response(data=err_msg, status=status.HTTP_400_BAD_REQUEST)

        if courses.count() < len(course_codes):
            existent_courses = courses.values_list('code', flat=True)
            non_existent_courses_ids = (
                set(course_codes) - set(existent_courses)
            )
            self.log.warning(u'Courses with ids %s does not exists',
                             non_existent_courses_ids)

        with transaction.atomic():
            for course in courses:
                CourseStudent.objects.update_or_create(
                    student=user,
                    course=course,
                    defaults={
                        'deleted': False,
                    }
                )
                LessonAssignment.ensure_student_assignments(course, user)

        qs = self.get_queryset(user=user)
        serializer = CoursesAssignedSerializer(qs, many=True)

        return Response(serializer.data)

    @log_method
    @detail_route(methods=['post'], permission_classes=[
        IsAuthenticated,
    ])
    def leave(self, request, pk=None):
        """
        Отвязывает пользователя от курса оставляя все результаты

        URL: /api/v2/sirius-courses/<pk>/leave/

        400: если связь уже удалена либо не существовала
        """
        try:
            # здесь мы не можем вызывать update_or_create как хотелось бы
            # потому что это приведёт к созданию записей, которые не должны были бы существовать
            # в случае, если кто-то напрямую дергает урл удаления из "моих"
            course_student = CourseStudent.objects.get(
                course=pk,
                student=request.user
            )

        except CourseStudent.DoesNotExist:
            return Response(status=status.HTTP_400_BAD_REQUEST)

        if course_student.assignment_rule:
            raise ValidationError("Cannot delete mandatory course")

        course_student.deleted = True
        course_student.save()

        invalidate_user_courses_cache(request.user, drop_reverse=True)

        return Response(status=status.HTTP_200_OK)

    def _retrieve(self, request, *args, **kwargs):
        user = kwargs.get('user')
        course_id = int(kwargs.get('pk'))

        try:
            course = self.get_queryset(
                user=user,
            ).get(pk=course_id)
            use_matchman = request.user.experiment_flags.get('enable_matchman', False)
            if not (
                request.user.is_superuser or
                (
                    request.user.is_support and
                    course.available_for_support
                ) or
                user_matched_to_course(request.user, course, use_matchman=use_matchman)
            ):
                return Response(status=status.HTTP_403_FORBIDDEN)

        except Course.DoesNotExist:
            return Response(status=status.HTTP_404_NOT_FOUND)

        course_serializer = CourseDetailSerializer(course, many=False, context={
            'user_id': request.user.id,
            'request': request,
        })

        clessons_qs = CourseLessonLink.objects.filter(
            course=course_id,
        ).select_related(
            'lesson',
            'course',
        ).prefetch_related(
            Prefetch(
                'userclessonstate_set',
                queryset=UserCLessonState.objects.filter(user=request.user),
            )
        )

        clesson_ids = clessons_qs.values_list('id', flat=True)
        course_lesson_results = SiriusCourseLessonResultsGroupSerializer(
            instance=CourseLessonResult.objects.filter(
                summary__clesson__course=course_id,
                summary__student_id=request.user.id,
                summary__clesson_id__in=clesson_ids
            ).select_related(),
            many=True
        )

        lesson_problem_link = LessonProblemLink.objects.filter(
            lesson__course__id=course_id,
        ).select_related(
            'problem',
            'lesson',
        ).order_by('id').distinct('id').values(
            'finish_date', 'group', 'id', 'lesson_id', 'options', 'order',
            'problem', 'problem_id', 'problem__max_points', 'start_date',
            'theory', 'theory_id', 'type'
        )

        assignments = LessonAssignment.objects.filter(
            student_id=request.user.id,
            clesson__in=clesson_ids
        ).values('clesson_id', 'problems')

        lesson_assignments_problems = {}
        for a in assignments:
            lesson_assignments_problems[a['clesson_id']] = a['problems']

        clessons_serializer = SiriusCourseLessonWithResultSerializer(
            clessons_qs, many=True, context={
                'course_lesson_results': course_lesson_results,
                'lesson_assignments_problems': lesson_assignments_problems,
                'lesson_problem_link_values': lesson_problem_link,
                'user_id': request.user.id,
            }
        )

        course_info = course_serializer.data
        course_info['points'] = {
            'max_points': 0, 'points': 0,
            'progress_max_points': 0, 'progress_points': 0,
        }
        for clesson_data in clessons_serializer.data:
            if (
                clesson_data['date_assignment'] is not None and
                clesson_data['allow_time_limited_problems'] is not True and
                clesson_data['lesson_block_slug'] != CLESSON_LECTURE_SLUG
            ):
                course_info['points']['max_points'] += clesson_data['max_points']
                course_info['points']['points'] += clesson_data['points']
                course_info['points']['progress_max_points'] += clesson_data['progress_max_points']
                course_info['points']['progress_points'] += clesson_data['progress_points']

        if course_info['points']['progress_max_points']:
            course_info['progress'] = int(
                course_info['points']['progress_points'] * 100 / course_info['points']['progress_max_points']
            )
        course_info['clessons'] = clessons_serializer.data

        return Response(course_info)

    @log_method
    @detail_route(methods=['get'], permission_classes=[
        IsAuthenticated
    ])
    def strict(self, request, *args, **kwargs):
        """
        Оверрайд метода получения данных единичного объекта (Курса)
        без проверки факта назначения курса текущему пользователю
        """
        if settings.COURSES_STRICT_MODE_ENABLED:
            kwargs['user'] = None
        else:
            kwargs['user'] = request.user

        return self._retrieve(request, *args, **kwargs)

    @log_method
    def retrieve(self, request, *args, **kwargs):
        """
        Оверрайд метода получения данных единичного объекта (Курса)
        с проверкой факта назначения курса текущему пользователю
        """
        kwargs['user'] = request.user
        return self._retrieve(request, *args, **kwargs)

    @detail_route(methods=['post'], permission_classes=[
        IsAuthenticated, ObjectCourseAssignedToStudent,
    ])
    def feedback(self, request, *args, **kwargs):
        """
        Метод, который принимает обратную связь по курсу с оценкой
        """
        current_feedback = CourseFeedback.objects.filter(
            user=request.user,
            course=self.get_object(),
        ).first()

        if current_feedback:
            feedback = CourseFeedbackSerializer(
                instance=current_feedback,
                data=request.data,
            )
        else:
            feedback = CourseFeedbackSerializer(
                data=request.data,
            )

        feedback.is_valid(raise_exception=True)
        feedback.save(user=request.user, course=self.get_object())

        return Response(feedback.validated_data)
