from django.db.models import Q, Subquery
from django.utils import timezone

from rest_framework import status
from rest_framework.decorators import detail_route, list_route
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response

from kelvin.accounts.utils import get_user_courses_qs, get_user_projects
from kelvin.courses.models import CourseStudent, Course
from kelvin.courses.serializers import CoursesLibrarySerializer, CourseIndexSerializer
from kelvin.courses.views.sirius_courses import SiriusCourses


class SiriusCourses3(SiriusCourses):
    """
    Хэндлер запросов к API курсов

    URL: /api/v3/sirius-courses/
    """
    permission_classes = []

    @list_route(methods=['get'], permission_classes=[
        IsAuthenticated
    ])
    def library(self, request):
        """
        Возвращает данные о библиотеке курсов

        URL: /api/v3/sirius-courses/library/
        """
        # Get courses that are not closed, free, and available to user
        # staff group
        use_matchman = request.user.experiment_flags.get('enable_matchman', False)
        qs = self.get_queryset().filter(
            id__in=get_user_courses_qs(
                user=request.user,
                use_matchman=use_matchman,
            )
        )
        qs = self.filter_queryset(qs)
        page = self.paginate_queryset(qs)
        if page is not None:
            courses_serializer = CoursesLibrarySerializer(page, many=True)
            return self.get_paginated_response(courses_serializer.data)

        courses_serializer = CoursesLibrarySerializer(qs, many=True)
        return Response(courses_serializer.data)

    def _get_index_queryset(self):
        return Course.objects.filter(
            Q(date_closed__isnull=True) | Q(date_closed__gt=timezone.now()),
            free=True,
            assignmentrule=None,
        ).select_related('subject')

    @list_route(methods=['get'])
    def index(self, request):
        """
        Метод, который возвращает курсы для индексации внутренним поиском
        @see https://wiki.yandex-team.ru/intranetpoisk/sourceapi/
        """
        qs = self._get_index_queryset()

        page = self.paginate_queryset(qs)
        if page is not None:
            courses_serializer = CourseIndexSerializer(page, many=True, context={'request': request})
            return self.get_paginated_response(courses_serializer.data)

        courses_serializer = CourseIndexSerializer(qs, many=True, context={'request': request})
        return Response(courses_serializer.data)

    @detail_route(methods=['get'])
    def index_detail(self, request, *args, **kwargs):
        """
        Метод, который возвращает информацию по курсу для индекса
        @see https://wiki.yandex-team.ru/intranetpoisk/sourceapi/
        """
        course_id = int(kwargs.get('pk'))

        qs = self._get_index_queryset()

        try:
            course = qs.get(pk=course_id)
        except Course.DoesNotExist:
            return Response(status=status.HTTP_404_NOT_FOUND)

        courses_serializer = CourseIndexSerializer(course, many=False, context={'request': request})

        return Response(courses_serializer.data)
