import logging

from django.db.models.signals import pre_save

from kelvin.formulas.models import ConvertedFormula
from kelvin.formulas.tasks import convert_problem, convert_text_resource
from kelvin.problems.models import Problem, TextResource

logger = logging.getLogger(__name__)


def check_formulas(formulas):
    """
    Проверяет требуется ли конвертация формул в словаре формул, заменяет
    данные в словаре на найденные формулы в базе

    @param formulas: словарь формул
    @return: требуется ли конвертация данного словаря
    """
    formula_codes = {
        formula_dict.get('code')
        for formula_dict in formulas.values()
        if 'code' in formula_dict
    }
    converted_formulas = {
        formula.code: formula
        for formula in ConvertedFormula.objects.filter(code__in=formula_codes)
    }
    need_conversion = False
    for formula_dict in formulas.values():
        if formula_dict['code'] in converted_formulas:
            # возможно уже есть сконвертированная формула
            if (formula_dict.get('url') ==
                    converted_formulas[formula_dict['code']].image.url and
                    formula_dict.get('style')):
                continue
            formula_dict['url'] = (
                converted_formulas[formula_dict['code']].image.url)
            formula_dict['style'] = (
                converted_formulas[formula_dict['code']].style)
        else:
            need_conversion = True
            formula_dict.pop('url', None)
            formula_dict.pop('style', None)

    return need_conversion


def convert_problem_formulas(sender, instance, **kwargs):
    """
    Создает задачу на конвертацию формул, если это необходимо, в задаче

    Если для формулы указана неправильное изображение, то удаляет адрес
    изображения и его стиль
    """
    if not (instance.markup and instance.markup.get('formulas')):
        return

    if check_formulas(instance.markup['formulas']):
        convert_problem.delay(instance.id)
        logger.info('convert formulas in %s', instance.id)


def convert_text_resource_formulas(sender, instance, **kwargs):
    """
    Создает задачу на конвертацию формул, если это необходимо, в текстовом
    ресурсе

    Если для формулы указана неправильное изображение, то удаляет адрес
    изображения и его стиль
    """
    if not instance.formulas:
        return

    if check_formulas(instance.formulas):
        convert_text_resource.delay(instance.id)
        logger.info('convert formulas in %s', instance.id)


pre_save.connect(convert_problem_formulas, sender=Problem)
pre_save.connect(convert_text_resource_formulas, sender=TextResource)
