import logging
from hashlib import md5

from donald.latex_to_png import latex_to_png

from django.core.files.base import ContentFile
from django.utils import timezone

from kelvin.celery import app
from kelvin.formulas.models import ConvertedFormula
from kelvin.problems.models import Problem, ProblemHistory, TextResource

FORMULA_EX_SIZE = '72'
MAX_UPDATE_RETRIES = 10

logger = logging.getLogger(__name__)


def update_formulas(formulas):
    """
    Уонвертирует формулы из словаря, преобразует сам словарь

    @param formulas: словарь формул
    @return: требуется ли конвертация данного словаря
    """
    formula_codes = {
        formula_dict.get('code')
        for formula_dict in formulas.values()
        if 'code' in formula_dict
    }
    if not formula_codes:
        return
    converted_formulas = {
        formula.code: formula
        for formula in ConvertedFormula.objects.filter(code__in=formula_codes)
    }
    for formula_dict in formulas.values():
        if formula_dict['code'] in converted_formulas:
            # возможно уже есть сконвертированная формула
            if (formula_dict.get('url') ==
                    converted_formulas[formula_dict['code']].image.url):
                continue
            formula = converted_formulas[formula_dict['code']]
        else:
            # надо генерировать картинку из формулы
            name = '{0}.png'.format(
                md5(formula_dict['code'].encode('utf-8')).hexdigest())
            image, style = latex_to_png(u'${0}$'.format(formula_dict['code']),
                                        ex_size=FORMULA_EX_SIZE)
            formula = ConvertedFormula.objects.create(
                code=formula_dict['code'],
                style=style,
            )
            formula.image.save(name, ContentFile(image))
            formula.save()
            converted_formulas[formula_dict['code']] = formula

        formula_dict['url'] = formula.image.url
        formula_dict['style'] = formula.style


def update_problem_formulas(problem):
    """
    Обновление формул в задаче
    """
    update_formulas(problem.markup.get('formulas', {}))

    # обновляем только при отсутствии изменений
    return Problem.objects.filter(
        id=problem.id,
        date_updated=problem.date_updated,
    ).update(markup=problem.markup, date_updated=timezone.now())


def update_text_resource_formulas(text_resource):
    """
    Обновление формул в текстовом ресурсе
    """
    update_formulas(text_resource.formulas or {})

    # обновляем только при отсутствии изменений
    return TextResource.objects.filter(
        id=text_resource.id,
        date_updated=text_resource.date_updated,
    ).update(formulas=text_resource.formulas, date_updated=timezone.now())


@app.task()
def convert_problem(problem_id, retries=0):
    """
    Генерирует картинки для формул в задаче
    """
    try:
        problem = Problem.objects.get(id=problem_id)
    except Problem.DoesNotExist:
        logger.error('Problem %s not found', problem_id)
        return
    updated = update_problem_formulas(problem)

    if updated:
        ProblemHistory.add_problem_version(
            problem, None, u'Конвертация формул')
    else:
        if retries < MAX_UPDATE_RETRIES:
            retries += 1
            convert_problem.delay(problem.id, retries=retries)
        else:
            logger.error('can not update formulas in Problem %s', problem.id)


@app.task()
def convert_text_resource(text_resource_id, retries=0):
    """
    Генерирует картинки для формул в текстовом ресурсе
    """
    try:
        text_resource = TextResource.objects.get(id=text_resource_id)
    except Problem.DoesNotExist:
        logger.error('TextResource %s not found', text_resource_id)
        return
    updated = update_text_resource_formulas(text_resource)

    if not updated:
        if retries < MAX_UPDATE_RETRIES:
            retries += 1
            convert_text_resource.delay(text_resource.id, retries=retries)
        else:
            logger.error('can not update formulas in TextResource %s',
                         text_resource.id)
