from builtins import map

from django.core.cache import cache

from rest_framework import viewsets
from rest_framework.response import Response

from kelvin.common.utils import group_level_sorted_key
from kelvin.group_levels.models import GroupLevel
from kelvin.group_levels.serializers import GroupLevelBasicSerializer


class GroupLevelView(viewsets.GenericViewSet):
    """
    Ручка для просмотра списка уровней группы. Сортирует сначала по базовому
    уровню, затем по имени уровня. Пагинация отсутствует
    """
    CACHE_TIME = 60 * 30
    LIST_CACHE_KEY = 'kelvin.group_levels.list.{0}'

    queryset = GroupLevel.objects.all()
    serializer_class = GroupLevelBasicSerializer
    filter_fields = ('baselevel',)

    def list(self, request, *args, **kwargs):
        """
        Кешированный список уровней групп
        """
        query_params = list(map(
            lambda param: '{0}={1}'.format(*param),
            iter(self.request.query_params.items()),
        ))
        cache_key = self.LIST_CACHE_KEY.format('&'.join(query_params))
        data = cache.get(cache_key)
        if data is None:
            data = self._get_data()
            cache.set(cache_key, data, self.CACHE_TIME)
        return Response(data)

    def _get_data(self):
        """
        Отсортированный список уровней групп без пагинации,
        с фильтрацией
        """
        group_levels = sorted(
            self.filter_queryset(self.get_queryset()),
            key=group_level_sorted_key
        )
        return self.get_serializer(group_levels, many=True).data
