from django_object_actions import DjangoObjectActions, takes_instance_or_queryset

from django.conf import settings
from django.contrib import admin, messages
from django.utils.html import format_html
from django.utils.translation import ugettext_lazy as _

from kelvin.common.admin import build_dropdown_filter_with_title

from .idm_client import IDMHTTPError
from .idm_client import client as idm_client
from .models import UserIDMRoleRequest
from .services.push_idm_request import push_idm_request


class UserIDMRoleRequestStatusFilter(admin.SimpleListFilter):
    title = _('Статус заявки')
    parameter_name = 'status'

    def lookups(self, request, model_admin):
        return UserIDMRoleRequest.STATUSES

    def queryset(self, request, queryset):
        if self.value() == UserIDMRoleRequest.STATUS_REQUESTED:
            return queryset.exclude(request_id=0).exclude(request_id__isnull=True)
        if self.value() == UserIDMRoleRequest.STATUS_REQUESTED_EARLIER:
            return queryset.filter(request_id=0)
        if self.value() == UserIDMRoleRequest.STATUS_NOT_REQUESTED:
            return queryset.filter(request_id__isnull=True)
        return queryset


@admin.register(UserIDMRoleRequest)
class UserIDMRoleRequestAdmin(DjangoObjectActions, admin.ModelAdmin):
    readonly_fields = (
        'created',
        'modified',
        'idm_url_link',
    )
    list_display = (
        'user',
        'idm_url_link',
        'system',
        'path',
        'criterion',
    )
    raw_id_fields = (
        'user',
        'criterion',
    )
    list_filter = (
        'system',
        'path',
        UserIDMRoleRequestStatusFilter,
        ('criterion__clesson__course', build_dropdown_filter_with_title('Курс')),
    )
    list_select_related = raw_id_fields
    search_fields = (
        'user__username',
    )

    change_actions = (
        'check_IDM_request',
        'request_role_force',
    )

    actions = (
        'request_role_force',
    )

    def idm_url_link(self, obj):
        if obj.request_id and obj.request_id != 0:
            url = '{IDM_WEB_URL}/system/{system}/roles#role={request_id},f-role-id={request_id}'.format(
                IDM_WEB_URL=settings.IDM_WEB_URL,
                system=obj.system,
                request_id=obj.request_id,
            )
            return format_html('<a href="{url}">{url}</a>', url=url)
        elif obj.request_id == 0:
            return _('Роль уже выдана ранее')

        return ''

    idm_url_link.short_description = 'Ссылка на запрос'
    idm_url_link.allow_tags = True

    def check_IDM_request(self, request, obj):
        message_level = messages.SUCCESS
        try:
            idm_client.request_role(
                system=obj.system,
                path=obj.path,
                fields_data=obj.fields_data,
                user=obj.user.username,
                comment=obj.comment,
                simulate=True,
            )
            message = _('IDM роль {}{} готова к использованию').format(obj.system, obj.path)

        except IDMHTTPError as ex:
            message = _(
                'Ошибка запроса IDM-роли: status code: {}, reason: {}, text: {}, x_system_request_id: {}'
            ).format(
                ex.response.status_code,
                ex.response.reason,
                ex.response.text,
                ex.x_system_request_id,
            )
            if ex.response.status_code != 409:
                message_level = messages.ERROR

        self.message_user(request=request, message=message, level=message_level)

    check_IDM_request.label = _("Тестовый запрос в IDM")
    check_IDM_request.short_description = _("Тестовый запрос в IDM")

    @takes_instance_or_queryset
    def request_role_force(self, request, queryset):
        for obj in queryset:
            message_level = messages.SUCCESS
            try:
                push_idm_request(user_idm_role_request=obj, force=True)
                message = _('IDM-роль запрошена')
            except IDMHTTPError as ex:
                message = _('Ошибка выдачи IDM-роли: status code: {}, reason: {}, text: {}, x_system_request_id: {}').format(
                    ex.response.status_code,
                    ex.response.reason,
                    ex.response.text,
                    ex.x_system_request_id,
                )
                if ex.response.status_code != 409:
                    message_level = messages.ERROR

            self.message_user(request=request, message=message, level=message_level)

    request_role_force.label = _("Запросить роль в IDM")
    request_role_force.short_description = _("Запросить роль в IDM")


class UserIDMRoleRequestInlineAdmin(admin.TabularInline):
    model = UserIDMRoleRequest

    readonly_fields = (
        'user',
        'system',
        'path',
        'request_id',
        'criterion',
        'created',
        'modified',
    )
    fields = readonly_fields
    extra = 0

    can_delete = False

    def has_add_permission(self, request):
        return False
