import logging
import uuid

from requests import Session
from requests.exceptions import HTTPError
from requests.status_codes import codes

from django.conf import settings

logger = logging.getLogger(__name__)


class IDMHTTPError(HTTPError):
    def __init__(self, x_system_request_id, **kwargs):
        self.x_system_request_id = x_system_request_id
        super().__init__(**kwargs)


class IDMClient(object):
    def __init__(self, idm_api_url, idm_token):
        self.IDM_API_URL = idm_api_url
        self.IDM_TOKEN = idm_token
        self.HEADERS = {
            'Authorization': 'OAuth {token}'.format(token=self.IDM_TOKEN),
            'Content-Type': 'application/json',
        }

    @staticmethod
    def get_session():
        return Session()

    def make_request(self, method, url, x_system_request_id=None, session=None, **kwargs):
        session = session or self.get_session()
        if x_system_request_id is None:
            x_system_request_id = str(uuid.uuid4())
        kwargs['headers'] = {**self.HEADERS, **{'X-System-Request-Id': x_system_request_id}}
        resp = session.request(method, url, allow_redirects=False, **kwargs)
        if resp.status_code < 300 or resp.status_code == 409:
            return resp.json()

        logger.exception(
            'IDM API error: x_system_request_id={}, status_code={}, reason={}, text={}'.format(
                x_system_request_id, resp.status_code, resp.reason, resp.text
            )
        )

        raise IDMHTTPError(x_system_request_id=x_system_request_id, response=resp)

    def request_role(
        self,
        system,
        path,
        user,
        comment,
        simulate=False,
        fields_data=None,
        deprive_at=None,
        review_at=None,
        deprive_after_days=None,
        silent=False,
        x_system_request_id=None,
        session=None,
    ):
        if x_system_request_id is None:
            x_system_request_id = str(uuid.uuid4())
        url = '{IDM_API_URL}/api/v1/rolerequests/'.format(IDM_API_URL=self.IDM_API_URL)
        data = {
            'simulate': simulate,
            'user': user,
            'system': system,
            'path': path,
            'fields_data': fields_data or {},
            'comment': comment,
            'silent': silent,
        }
        if deprive_at is not None:
            data['deprive_at'] = deprive_at
        if review_at is not None:
            data['review_at'] = review_at
        if deprive_after_days is not None:
            data['deprive_after_days'] = deprive_after_days

        return self.make_request(
            method='POST',
            url=url,
            x_system_request_id=x_system_request_id,
            session=session,
            json=data
        )


client = IDMClient(
    idm_api_url=settings.IDM_API_URL,
    idm_token=settings.IDM_TOKEN,
)
