import logging
import uuid

from rest_framework.response import Response
from rest_framework.views import APIView

from kelvin.common.exceptions import APIExceptionWithStatusCode

from .idm_client import IDMHTTPError
from .idm_client import client as idm_client
from .serializers import ValidateIDMRoleViewInputSerializer, ValidateIDMRoleViewOutputSerializer

logger = logging.getLogger(__name__)


class ValidateIDMRoleView(APIView):
    input_serializer = ValidateIDMRoleViewInputSerializer
    output_serializer = ValidateIDMRoleViewOutputSerializer

    def post(self, request):

        serializer = self.input_serializer(data=self.request.data)
        serializer.is_valid(raise_exception=True)

        system = serializer.validated_data['system']
        path = serializer.validated_data['path']
        fields_data = serializer.validated_data['fields_data']

        login = request.user.username
        x_system_request_id = str(uuid.uuid4())
        try:
            idm_client.request_role(
                system=system,
                path=path,
                user=login,
                fields_data=fields_data,
                comment='test request idm role',
                simulate=True,
                x_system_request_id=x_system_request_id,
                session=None,
            )

            return Response(self.output_serializer({'x_system_request_id': x_system_request_id}).data)

        except IDMHTTPError as ex:
            if ex.response.status_code == 409:
                return Response(self.output_serializer({'x_system_request_id': x_system_request_id}).data)
            else:
                raise APIExceptionWithStatusCode(
                    detail={'idm_api_response': ex.response.text, 'idm_api_x_system_request_id': ex.x_system_request_id},
                    code=ex.response.reason,
                    status_code=ex.response.status_code,
                )
