import random
from builtins import object
from collections import defaultdict

from django.conf import settings
from django.db import models, transaction
from django.utils.translation import ugettext_lazy as _

from kelvin.common.fields import JSONField
from kelvin.courses.models import CourseLessonLink
from kelvin.lessons.models import LessonProblemLink


class LessonAssignment(models.Model):
    """
    Назначение урока ученику

    Поле `problems` содержит список идентификаторов связей занятие-вопрос
    """
    clesson = models.ForeignKey(
        CourseLessonLink,
        verbose_name=_('Курсозанятие'),
    )
    student = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Ученик'),
    )
    problems = JSONField(
        verbose_name=_('Список назначенных задач'),
        blank=True,
        default=[],
    )

    @classmethod
    def get_student_problems(cls, student, clesson):
        """
        Возвращает список идентификаторов назначенных ученику задач или None,
        если назначенных нет
        """
        try:
            return cls.objects.get(clesson=clesson, student=student).problems
        except cls.DoesNotExist:
            pass

    def ensure_variations(self, problem_links):
        """
        Проверяет, что назначение соответствует группировке из :problem_links:,
        при неправильном назначении исправляет его.

        В каждом назначении есть ровно одна задача из каждой группы задач,
        а также все задачи без проставленной группы.
        Если есть несколько задач из одной группы, то оставляем 1.
        Если нет задач из группы, то добавляем 1.
        Если есть задача, которой нет в занятии, выкидываем её.

        :param problem_links: список связей задача-занятие; используется
                              вместо обращения к курсозанятию
        :return: правильность назначений
        :rtype: bool
        """
        # вспомогательные структуры
        groups = defaultdict(list)
        for link in problem_links:
            groups[link.group].append(link.id)

        group_by_link_id = {link.id: link.group for link in problem_links}

        assignment_groups = defaultdict(list)
        for link_id in self.problems:
            assignment_groups[group_by_link_id[link_id]].append(link_id)

        is_valid = True
        valid_assignment = groups.get(None, [])

        # проверка назначенности задач без группы
        if set(valid_assignment) != set(assignment_groups.get(None, [])):
            is_valid = False

        # проверка каждой группы
        for group, links_in_group in groups.items():
            if group is None:
                continue
            assigned_links_len = len(assignment_groups.get(group, []))
            if assigned_links_len == 0:
                is_valid = False
                valid_assignment.append(random.choice(links_in_group))
            elif assigned_links_len == 1:
                valid_assignment.append(assignment_groups[group][0])
            else:
                is_valid = False
                valid_assignment.append(
                    random.choice(assignment_groups[group]))

        if not is_valid:
            self.problems = valid_assignment

        return is_valid

    class Meta(object):
        verbose_name = _('Назначение вопросов занятия ученикам')
        verbose_name_plural = _('Назначения вопросов занятия ученикам')
        unique_together = ('clesson', 'student')

    @classmethod
    def ensure_student_assignments(cls, course, student):
        """
        Проверяет и при необходимости изменяет назначения у ученика в курсе
        """
        # из курса берем все идентификаторы занятий с вариациями
        lesson_ids = (
            LessonProblemLink.objects.filter(
                lesson__courselessonlink__course=course,
                group__isnull=False,
            ).order_by(
                'lesson',
            ).values_list(
                'lesson_id',
                flat=True,
            ).distinct('lesson')
        )

        if not lesson_ids:
            # в курсе нет занятий с вариациями
            return

        clessons = (
            course.courselessonlink_set.filter(
                lesson__in=lesson_ids,
            ).select_related(
                'lesson',
            ).prefetch_related(
                'lesson__lessonproblemlink_set',
            )
        )
        assignments_by_clesson = {
            assignment.clesson_id: assignment
            for assignment in cls.objects.filter(
                student=student,
                clesson__in=clessons,
            )
        }

        assignments_to_update = []
        assignments_to_update_old_ids = []
        for clesson in clessons:
            assignment = assignments_by_clesson.get(clesson.id)
            if not assignment:
                assignment = cls(
                    clesson=clesson,
                    student=student,
                    problems=[],
                )
            if not assignment.ensure_variations(
                clesson.lesson.lessonproblemlink_set.all(),
            ):
                if assignment.id:
                    assignments_to_update_old_ids.append(assignment.id)
                assignment.pk = None
                assignments_to_update.append(assignment)

        if assignments_to_update:
            with transaction.atomic():
                cls.objects.filter(
                    id__in=assignments_to_update_old_ids,
                ).delete()
                cls.objects.bulk_create(assignments_to_update)

        return len(assignments_to_update)
