from builtins import map

from rest_condition.permissions import Or

from django.db import transaction
from django.utils import timezone

from rest_framework import viewsets
from rest_framework.decorators import list_route
from rest_framework.exceptions import ValidationError
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response

from kelvin.accounts.permissions import IsContentManager, IsStaff, IsTeacher
from kelvin.courses.models import Course, CourseLessonLink
from kelvin.lesson_assignments.models import LessonAssignment
from kelvin.lessons.services import copy_lesson
from kelvin.result_stats.tasks import recalculate_courselessonstat, recalculate_studentcoursestat
from kelvin.results.models import CourseLessonResult


class LessonAssignmentViewSet(viewsets.GenericViewSet):
    """
    Методы работы с назначениями занятия ученикам
    """
    queryset = LessonAssignment.objects.all()
    filter_fields = (
        'clesson',
    )

    @list_route(permission_classes=(IsAuthenticated,
                                    Or(IsTeacher, IsContentManager, IsStaff)))
    def get(self, request):
        """
        Получить матрицы назначенных вопросов
        """
        problem_link_ids, student_ids = self.get_problems_and_students()
        assignments = {
            assignment.student_id: assignment.problems
            for assignment in self.filter_queryset(self.get_queryset())
        }
        return Response({
            student_id: (assignments[student_id] if student_id in assignments
                         else problem_link_ids)
            for student_id in student_ids
        })

    @list_route(methods=['post'],
                permission_classes=(IsAuthenticated,
                                    Or(IsTeacher, IsContentManager, IsStaff)))
    def save(self, request):
        """
        Сохранить матрицу назначения
        """
        # TODO права
        problem_link_ids, student_ids = self.get_problems_and_students()
        problem_link_ids = set(problem_link_ids)
        student_ids = set(map(str, student_ids))
        clesson_id = int(self.request.query_params.get('clesson'))
        assignment_data = self.request.data

        if not isinstance(assignment_data, dict):
            raise ValidationError('Assignment data should be a dictionary')

        # Приводим списки id ссылок к числам. Если содержатся не списки
        # или привести не получится - поднимем ошибку
        try:
            for student_id, assignments in assignment_data.items():
                assignment_data[student_id] = list(map(int, assignments))
        except (TypeError, ValueError):
            raise ValidationError('Values for problems should be integers')

        # параметры для обновления статистик
        stats_recalculate = set()

        with transaction.atomic():
            try:
                clesson = (
                    CourseLessonLink.objects.select_related('lesson')
                    .get(id=clesson_id)
                )
            except (CourseLessonLink.DoesNotExist, TypeError, ValueError):
                raise ValidationError('Invalid clesson')

            # Если нельзя редактировать урок, то делаем его копию
            # и актуализируем ссылки на задачи, а также назначения
            if not clesson.lesson_editable:
                old_to_new = copy_lesson(clesson.lesson, owner=self.request.user)
                clesson.lesson_editable = True
                clesson.lesson_id = old_to_new['new_lesson_id']
                clesson.save()
                problem_link_ids = set(old_to_new['problems'].values())

                # Также откидываем те id назначений, которых не было среди
                # старых, так как на них нельзя полагаться
                for student_id, assignments in assignment_data.items():
                    assignment_data[student_id] = (
                        old_to_new['problems'].get(old_id, old_id)
                        for old_id in assignments
                    )
                lesson_copied = True
            else:
                # Если можно редактировать, то нужно обновить версию позже
                lesson_copied = False

            assignments_to_create = []
            for student_id, problem_links in assignment_data.items():
                if student_id not in student_ids:
                    continue
                problem_links = set(problem_links) & problem_link_ids
                if len(problem_links) == len(problem_link_ids):
                    continue
                assignment = LessonAssignment(problems=sorted(problem_links))
                assignment.clesson_id = clesson_id
                assignment.student_id = student_id
                assignments_to_create.append(assignment)

            # удаляем старые назначения и создаем новые
            self.filter_queryset(self.get_queryset()).delete()
            LessonAssignment.objects.bulk_create(assignments_to_create)

            # обновляем максимум баллов в результатах при наличии результатов
            # обновляем результаты по-одному, потому что считаем, что
            # изменения назначения при наличии результатов происходят
            # индивидуально (добавление дополнительных задач ученику)
            assignments_by_student = {
                int(assignment.student_id): assignment
                for assignment in assignments_to_create
            }
            for result in CourseLessonResult.objects.filter(
                    summary__student_id__in=student_ids,
                    summary__clesson=clesson).select_related('summary'):
                assigned = (
                    assignments_by_student[result.summary.student_id].problems
                    if result.summary.student_id in assignments_by_student
                    else None
                )

                max_points = clesson.lesson.get_max_points(assigned=assigned)
                if result.max_points != max_points:
                    CourseLessonResult.objects.filter(id=result.id).update(
                        max_points=max_points)

                    # надо будет пересчитать статистику ученика
                    stats_recalculate.add(
                        (result.summary.student_id, clesson.course_id))

                    # надо будет пересчитать статистику по курсозанятию
                    stats_recalculate.add(clesson.id)

            # Если нужно, обновляем версии курса и курсозанятия
            if not lesson_copied:
                now = timezone.now()
                Course.objects.filter(id=clesson.course_id).update(
                    date_updated=now)
                CourseLessonLink.objects.filter(id=clesson_id).update(
                    date_updated=now)

        # добавляем задания для пересчета статистик
        for params in stats_recalculate:
            if isinstance(params, tuple):
                recalculate_studentcoursestat.delay(*params)
            else:
                recalculate_courselessonstat.delay(params)

        # Если была создана копия, то нужно уведомить клиент
        return Response({'reload': lesson_copied})

    def get_problems_and_students(self):
        """
        Вернуть списки идентификаторов вопросов в занятии и учеников в группе
        """
        try:
            clesson = int(self.request.query_params.get('clesson'))
        except (ValueError, TypeError):
            raise ValidationError(
                'use with integer argument `clesson`')

        # TODO подумать про 1 запрос
        problem_link_ids = (CourseLessonLink.objects.filter(id=clesson)
                            .values_list('lesson__lessonproblemlink__id',
                                         flat=True)
                            .exclude(lesson__lessonproblemlink=None)
                            .order_by('lesson__lessonproblemlink__id'))
        student_ids = (CourseLessonLink.objects.filter(id=clesson)
                       .exclude(course__students=None)
                       .values_list('course__students__id', flat=True))
        return problem_link_ids, student_ids
