import re
from builtins import object, str

from django.conf import settings
from django.conf.urls import url
from django.contrib import admin, messages
from django.contrib.auth import get_user_model
from django.core.exceptions import PermissionDenied, ValidationError
from django.db.models import Q
from django.http import Http404
from django.shortcuts import redirect
from django.urls import reverse
from django.utils.translation import ugettext_lazy as _

from kelvin.common.admin import (
    AvailableForSupportAdminMixin, AvailableForSupportInlineAdminMixin, LabellessRawIdFieldsMixin,
    UserBlameModelAdminMixin,
)
from kelvin.problems.admin.utils import get_big_resource_preview, get_short_text

from ..courses.models import CourseLessonLink
from .filters import LessonProblemLinkGroupFilter, LessonTypeFilter
from .forms import CorpLessonAdminForm, LessonAdminForm, ProblemLinkForm
from .models import Lesson, LessonProblemLink, LessonScenario, TextTemplate

User = get_user_model()


class CourseLessonLinkInline(admin.TabularInline):
    model = CourseLessonLink
    fields = ('course_link',)
    readonly_fields = fields
    extra = 0
    can_delete = False
    max_num = 0
    verbose_name_plural = _("Используется в курсах")

    def course_link(self, obj):
        return (
            u'<a href="{}" class="changelink" target="_blank">{}</a>'.format(
                reverse('admin:courses_course_change', args=(obj.course.id,)),
                obj.course,
            )
            if obj.id else u''
        )
    course_link.short_description = _("Курс")
    course_link.allow_tags = True

    def has_add_permission(self, request):
        return False


class LessonProblemLinkInline(
    LabellessRawIdFieldsMixin,
    AvailableForSupportInlineAdminMixin,
    admin.TabularInline,
):
    """
    Инлайн для связей задач с занятиями на странице занятия
    """
    model = LessonProblemLink
    verbose_name = _('Вопрос занятия')
    verbose_name_plural = _('Вопросы занятия')
    fields = (
        'id',
        'order',
        'cm_order',
        'problem',
        'get_problem_link',
        'theory',
        'get_theory_link',
        'type',
        'options',
        'group',
        'block_id',
        'start_date',
        'finish_date',
        'get_tips_links',
        'available_for_support',
    )
    readonly_fields = (
        'id',
        'get_problem_link',
        'get_theory_link',
        'get_tips_links',
    )
    labelless_raw_id_fields = ('problem', 'theory',)
    form = ProblemLinkForm

    def get_problem_link(self, obj):
        """Ссылка на задачу"""
        if obj.problem_id:
            return (
                '<a href="{}" class="changelink" target="_blank">'
                '</a>'.format(
                    reverse('admin:problems_problem_change', args=(obj.problem_id,)),
                )
                if obj.id else ''
            )
        return ''

    get_problem_link.allow_tags = True
    get_problem_link.short_description = _('Ссылка')

    def get_theory_link(self, obj):
        """Ссылка на теорию"""
        if obj.theory_id:
            return (
                '<a href="{}" class="changelink" target="_blank">'
                '</a>'.format(
                    reverse('admin:problems_textresource_change', args=(obj.theory_id,)),
                )
                if obj.id else ''
            )
        return ''

    get_theory_link.allow_tags = True
    get_theory_link.short_description = _('Ссылка на теорию')

    def get_tips_links(self, obj):
        """Ссылки на шпаргалки задачи"""
        return ''.join([
            '<p>'
            '<a href="{0}" class="changelink" target="_blank">{1}</a>'
            '</p>'.format(
                reverse('admin:problems_textresource_change', args=(tip.id,)),
                str(tip)
            )
            for tip in obj.problem.tips.all()
        ])

    get_tips_links.allow_tags = True
    get_tips_links.short_description = _('Ссылки на шпаргалки')

    def get_queryset(self, request):
        """Чтобы было меньше запросов к шпаргалкам"""
        queryset = super(LessonProblemLinkInline, self).get_queryset(request)
        return queryset.prefetch_related('problem__tips')

    class Media(object):
        css = {
            'all': (
                'lessons/css/admin_hide_original.css',
                'lessons/css/admin_problem_order_width.css',
            ),
        }


class LessonScenarioInline(admin.TabularInline):
    """Инлайн для сценариев занятия на странице занятия"""
    model = LessonScenario
    extra = 0


class BaseLessonAdmin(UserBlameModelAdminMixin, admin.ModelAdmin):
    """
    Админка модели учебного занятия
    """

    # change_form_template = 'lessons/admin/change_form.html'

    # TODO меньше запросов к БД
    list_display = (
        'id',
        'name',
        'owner',
        'get_edit_url',
        'date_updated',
        'date_created',
        'theme',
    )
    list_select_related = ('owner', 'theme')
    raw_id_fields = ('owner',)

    inlines = (
        CourseLessonLinkInline,
        LessonProblemLinkInline,
        LessonScenarioInline,
    )
    form = LessonAdminForm

    search_fields = ('id', 'name')
    list_filter = (
        LessonTypeFilter,
        ('owner', admin.RelatedOnlyFieldListFilter),
    )
    readonly_fields = (
        'get_front_link',
        'get_problem_links_link',
        'get_reorder_link',
        'created_by',
        'modified_by',
    )

    actions = [
        'copy_action',
    ]

    @staticmethod
    def _fill_order(data):
        """
        Копирует данные и изменяет их, добавляя порядок задачи,
        если указана задача,
        но не указан порядок. После чего возвращает копию.

        @param data: данные формы создания/изменения курса
        """
        order_key = 'lessonproblemlink_set-{0}-order'
        problem_key = 'lessonproblemlink_set-{0}-problem'
        i = 0
        orders_to_set = []
        max_order = 0
        while True:
            if problem_key.format(i) not in data:
                break
            if not data.get(problem_key.format(i)):
                i += 1
                continue
            order = data.get(order_key.format(i))
            if order:
                max_order = max(max_order, int(order))
            else:
                orders_to_set.append(order_key.format(i))
            i += 1
        resulting_data = data.copy()
        for key in orders_to_set:
            max_order += 1
            resulting_data[key] = max_order
        return resulting_data

    def add_view(self, request, form_url='', extra_context=None):
        """
        Заполняет поле порядка для задачи
        """
        if request.method == 'POST':
            request.POST = self._fill_order(request.POST)

        return super(BaseLessonAdmin, self).add_view(
            request, form_url, extra_context)

    def change_view(self, request, object_id, form_url='', extra_context=None):
        """
        Заполняет поле порядка для задачи
        """
        if request.method == 'POST':
            request.POST = self._fill_order(request.POST)

        return super(BaseLessonAdmin, self).change_view(
            request, object_id, form_url, extra_context)

    def get_urls(self):
        """
        Add url for:
        - copy lesson
        - reorder problems
        """
        info = self.model._meta.app_label, self.model._meta.model_name
        copy_url = url(
            regex=r'^(.+)/copy/$',
            view=self.admin_site.admin_view(self.copy_view),
            name='{0}_{1}_copy'.format(*info)
        )
        reorder_url = url(
            regex=r'^(.+)/reorder/$',
            view=self.admin_site.admin_view(self.reorder_view),
            name='{0}_{1}_reorder'.format(*info)
        )
        return [copy_url, reorder_url] + super(BaseLessonAdmin, self).get_urls()

    def copy_action(self, request, queryset):
        """ Копирует урок в админке """
        for lesson in queryset.all():
            lesson.copy(name='{0} (копия)'.format(lesson.name))

    copy_action.short_description = 'Скопировать урок'

    def copy_view(self, request, pk):
        """
        Копирует занятие с изменением названия и автора
        """
        if request.method != 'POST':
            raise Http404
        try:
            lesson = self.model.objects.get(pk=pk)
        except (ValueError, TypeError, self.model.DoesNotExist):
            raise Http404

        if not self.has_change_permission(request, lesson):
            raise PermissionDenied

        lesson.copy(owner=request.user,
                    name='{0} (копия)'.format(lesson.name))
        messages.add_message(request, messages.INFO,
                             'Занятие успешно создано')

        return redirect('admin:lessons_lesson_change', lesson.id)

    def reorder_view(self, request, pk):
        """Переупорядочить задачи согласно порядку контент-менеджера"""
        try:
            lesson = Lesson.objects.prefetch_related('lessonproblemlink_set').get(pk=pk)
        except (ValueError, TypeError, self.model.DoesNotExist):
            raise Http404

        if not self.has_change_permission(request, lesson):
            raise PermissionDenied

        expr = re.compile(r'(\d*)\.?(\d*)')

        def key_func(link):
            """
            Функция сортировки порядка КМа,
            первыми идут связи, которые имеют цифровые значения, потом
            строчные значения, потом пустые значения
            """
            groups = expr.match(link.cm_order).groups()
            key = []
            has_digits = False
            for group in groups:
                if group.isdigit():
                    has_digits = True
                    key.append(int(group))
                elif group:
                    key.append(group)
                else:
                    key.append(None)

            if link.cm_order:
                if has_digits:
                    return 0, key
                else:
                    return 1, link.cm_order
            return 2, key

        for i, link in enumerate(sorted(lesson.lessonproblemlink_set.all(), key=key_func), start=1):
            link.order = i
            link.save()

        return redirect('admin:lessons_lesson_change', lesson.id)

    def get_edit_url(self, obj):
        """
        Возвращает html ссылку на редактирование задания на фронте
        """
        return (
            '<a href="{0}lab/lessons/{1}/" class="changelink"></a>'
            .format(settings.FRONTEND_HOST, obj.id)
        )

    get_edit_url.allow_tags = True
    get_edit_url.short_description = 'Редактировать на фронте'

    def get_field_queryset(self, db, db_field, request):
        """
        Для поле `owner` возвращает контент-менеджеров и учителей
        """
        if db_field.name == 'owner':
            return User.objects.filter(
                Q(is_content_manager=True) | Q(is_teacher=True))

        return super(BaseLessonAdmin, self).get_field_queryset(
            db, db_field, request)

    def save_related(self, request, form, formsets, change):
        """
        Добавляем дефолтный сценарий при создании, если не был добавлен
        """
        super(BaseLessonAdmin, self).save_related(request, form, formsets, change)
        if change:
            return
        instance = form.instance
        if instance.lessonscenario_set.count() == 0:
            LessonScenario.objects.create(
                lesson=instance, primary=True,
                mode=LessonScenario.TRAINING_MODE,
                duration=45,
            )

    def get_object(self, request, object_id, from_field=None):
        """
        Переопределен, чтобы указать prefetch_related только для формы
        редактирования элемента
        """
        qs = self.get_queryset(request)
        qs = qs.prefetch_related('courselessonlink_set', 'courselessonlink_set__course')

        field = qs.model._meta.pk if from_field is None else (
            qs.model._meta.get_field(from_field))
        try:
            object_id = field.to_python(object_id)
            return qs.get(**{field.name: object_id})
        except (qs.model.DoesNotExist, ValidationError, ValueError):
            return None

    def formfield_for_foreignkey(self, db_field, request=None, **kwargs):
        """Проставляет дефолтное значение для владельца"""
        if (db_field.name == 'owner' and request and
                request.user.is_teacher or request.user.is_content_manager):
            kwargs['initial'] = request.user.id

        return super(BaseLessonAdmin, self).formfield_for_foreignkey(db_field, request, **kwargs)

    def get_front_link(self, obj):
        """
        Выводит ссылку на фронт
        """
        return ('<a href="{0}lab/lessons/{1}/">Посмотреть</a>'.format(
            settings.FRONTEND_HOST, obj.id, ))

    get_front_link.short_description = _('Ссылка на фронт')
    get_front_link.allow_tags = True

    def get_problem_links_link(self, obj):
        """
        Выводит ссылку на страницу управления группами
        """
        raw_url = reverse('admin:lessons_lessonproblemlink_changelist')
        url = '{url}?lesson={lesson_id}&{group_param}={all_value}'.format(
            url=raw_url,
            lesson_id=obj.id,
            group_param=LessonProblemLinkGroupFilter.parameter_name,
            all_value=LessonProblemLinkGroupFilter.all_value
        )

        return '<a href="{url}">Перейти</a>'.format(url=url)

    get_problem_links_link.short_description = _('Управление группами задач')
    get_problem_links_link.allow_tags = True

    def get_reorder_link(self, obj):
        """
        Кнопка для переупорядочивания задач в занятии по порядку
        контент-менеджера
        """
        return (
            '<b>Сохраните изменения</b> перед упорядочиванием! '
            '<a href="{}">Упорядочить задачи</a>'.format(
                reverse('admin:lessons_lesson_reorder', args=(obj.id,))
            )
        )

    get_reorder_link.short_description = _('Упорядочить задачи (КМ)')
    get_reorder_link.allow_tags = True


class LessonAvailableForSupportAdminMixin(AvailableForSupportAdminMixin):
    def get_readonly_fields(self, request, obj=None):
        ro_fields = super().get_readonly_fields(request, obj)
        if obj is not None:
            user = request.user
            if user and not user.is_anonymous and user.is_authenticated and user.is_support:
                ro_fields = list(
                    set(ro_fields) |
                    {
                        'name',
                        'subject',
                        'theme',
                        'hooks',
                        'owner',
                    }
                )

        return ro_fields


class LessonAdmin(BaseLessonAdmin, LessonAvailableForSupportAdminMixin):
    form = CorpLessonAdminForm


class LessonScenarioAdmin(admin.ModelAdmin):
    """
    Админка сценариев занятия
    """
    list_display = (
        'lesson',
        'mode',
        'duration',
        'show_answers_in_last_attempt',
        'visual_mode',
        'max_attempts_in_group',
        'show_all_problems',
    )


class LessonProblemLinkAdmin(UserBlameModelAdminMixin, AvailableForSupportAdminMixin, admin.ModelAdmin):
    """
    Админка модели занятие-вопрос
    """
    list_display = (
        'id',
        'order',
        'cm_order',
        'problem',
        'get_big_resource_preview',
        'get_problem_text',
        'type',
        'group',
        'start_date',
        'finish_date',
    )

    list_editable = (
        'cm_order',
        'type',
        'group',
    )

    list_filter = (
        LessonProblemLinkGroupFilter,
    )

    list_per_page = 20

    raw_id_fields = (
        'lesson',
        'problem',
        'theory',
    )

    readonly_fields = (
        'created_by',
        'modified_by',
    )

    def get_big_resource_preview(self, obj):
        """
        Столбец ресурса задачи
        """
        return get_big_resource_preview(obj.problem) if obj.problem else ''

    get_big_resource_preview.short_description = 'Ресурс из задачи'
    get_big_resource_preview.allow_tags = True

    def get_problem_text(self, obj):
        """
        Столбец текста задачи
        """
        return get_short_text(obj.problem) if obj.problem else ''

    get_problem_text.short_description = _('Текст')
    get_problem_text.allow_tags = True

    def get_queryset(self, request):
        """
        Делаем prefetch для связанных моделей Problem и Resource
        """
        qs = super(LessonProblemLinkAdmin, self).get_queryset(request)
        return qs.prefetch_related(
            'problem__resources',
        ).select_related(
            'problem',
        )

    def save_model(self, request, obj, form, change):
        """
        Проставляем порядковый номер вопроса согласно его группе
        при сохранении
        """
        if form.cleaned_data.get('group'):
            obj.order = form.cleaned_data.get('group')

        super(LessonProblemLinkAdmin, self).save_model(request, obj, form, change)


admin.site.register(Lesson, LessonAdmin)
admin.site.register(LessonScenario, LessonScenarioAdmin)
admin.site.register(TextTemplate)
admin.site.register(LessonProblemLink, LessonProblemLinkAdmin)
